/* $Date: 92/02/16 08:43:31 $    $Revision: 1.2 $  by $Author: joe $  */
/* 
  Copyright (C) 1991 by the Massachusetts Institute of Technology

   Export of this software from the United States of America is assumed
   to require a specific license from the United States Government.
   It is the responsibility of any person or organization contemplating
   export to obtain such a license before exporting.

WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
distribute this software and its documentation for any purpose and
without fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright notice and
this permission notice appear in supporting documentation, and that
the name of M.I.T. not be used in advertising or publicity pertaining
to distribution of the software without specific, written prior
permission.  M.I.T. makes no representations about the suitability of
this software for any purpose.  It is provided "as is" without express
or implied warranty.

  */

/****************************************************************************
  CmFigureEditor.cc -  Figure Handler/Editor Code
  Contact: sniyogi, maschaub
 ***************************************************************************/

#include <CmFigureEditor.h>
#include <CmFigureClass.h>
#include <CmFigureHandler.h>
#include <CmFigureRegistrar.h>
#include <CmFigureParser.h>
#include <CmDialog.h>
#include <CmMenu.h>
#include <CmModel.h>
#include <CmCDS.h>
#include <CmFigure.h>
#include <CmPage.h>
#include <CmNavigator.h>
#include <CmStringMotif.h>

#include <Xm/Form.h>
#include <Xm/LabelG.h>

static const CmRegex WhiteSpace("[ \n\t\r\v\f]*", 1);

enum CmPageTextType {
  CMPAGE_CONSTRUCTOR,
  CMPAGE_ENTRY,
  CMPAGE_EXIT,
};

struct CmEditorDialogWad {
  CmFigureEditor* Editor;
  CmDialog* Dialog;
  CmPageTextType PageTextType;
  CmEditorDialogWad(CmFigureEditor *e, CmDialog *d, CmPageTextType ptt) {
    Editor = e; Dialog = d; PageTextType = ptt;
  };
};

// Figure Editing Callbacks (called underneath "Edit" in menu)
void CmCutFigure(Widget, CmFigureEditor* editor, XtPointer) { 
  editor->CutFigure();   
}

void CmCopyFigure(Widget, CmFigureEditor* editor, XtPointer)  
{ editor->CopyFigure();  }
void CmPasteFigure(Widget, CmFigureEditor* editor, XtPointer) 
{ editor->PasteFigure(); }
void CmCreateFigure(Widget, CmEditorCmFigureClassWad* wad, 
		    XtPointer) 
{  wad->Editor->CreateFigure(wad->CmClass); }
void CmToggleEdit(Widget, CmFigureEditor* editor, XtPointer)
{  editor->ToggleEditing(); }
void CmShowAll(Widget, CmFigureEditor* editor, XtPointer) {
  editor->ShowAll();
}
void CmHide(Widget, CmFigureEditor* editor, XtPointer) {
  editor->HideCurrentFigure(); 
}

/************ model text for each page *******************/

void CmPopDownPageText(Widget, CmDialog* dialog, XtPointer) { 
  dialog->PopdownDialog(); 
}

void CmExecutePageText(Widget, CmEditorDialogWad* ewad, 
		      XtPointer) { 
  switch (ewad->PageTextType) {
  case CMPAGE_CONSTRUCTOR:
    ewad->Editor->CDSPtr()->Model()->Evaluate(    
      ewad->Dialog->GetString("page_text", "Page \n Constructor \n Text"));
    break;
  case CMPAGE_ENTRY:
    ewad->Editor->CDSPtr()->Model()->Evaluate(    
      ewad->Dialog->GetString("page_text", "Page \n Entry \n Text"));
    break;
  case CMPAGE_EXIT:
    ewad->Editor->CDSPtr()->Model()->Evaluate(    
      ewad->Dialog->GetString("page_text", "Page \n Exit \n Text"));
    break;
  }
}

/* constructor text */
void CmUpdatePageText(Widget, CmEditorDialogWad* ewad, 
		      XtPointer) { 
  switch (ewad->PageTextType) {
  case CMPAGE_CONSTRUCTOR:
    ewad->Editor->Page->PageConstructorText = 
      ewad->Dialog->GetString("page_text", "Page \n Constructor \n Text");
    break;
  case CMPAGE_ENTRY:
    ewad->Editor->Page->PageEntryText = 
      ewad->Dialog->GetString("page_text", "Page \n Entry \n Text");
    break;
  case CMPAGE_EXIT:
    ewad->Editor->Page->PageExitText = 
      ewad->Dialog->GetString("page_text", "Page \n Exit \n Text");
    break;
  }
  ewad->Dialog->PopdownDialog();
}

void CmPageText(Widget, CmEditorDialogWad *ewad, XtPointer) {
  CmString pane = "page_text", textbox, page_dialog_name, page_text;
  CmString help_text;

  switch (ewad->PageTextType) {
  case CMPAGE_CONSTRUCTOR:
    textbox = "Page \n Constructor \n Text";
    page_dialog_name = "Page Constructor Text Window";
    help_text = "Enter the scheme expressions to be run everytime the page is displayed\n\n";
    page_text = ewad->Editor->Page->PageConstructorText;
    break;
  case CMPAGE_ENTRY:
    textbox = "Page \n Entry \n Text";
    page_dialog_name = "Page Entry Text Window";
    help_text = "Enter the scheme expressions to be run everytime the page is displayed\n\n";
    page_text = ewad->Editor->Page->PageEntryText;
    break;
  case CMPAGE_EXIT:
    textbox = "Page \n Exit \n Text";
    page_dialog_name = "Page Exit Text Window";
    help_text = "Enter the scheme expressions to be run everytime the page is displayed\n\n";
    page_text = ewad->Editor->Page->PageExitText;
    break;
  }

  CmDialog* page_dialog = new CmDialog(ewad->Editor->CDSPtr()->TopLevel(), 
				       page_dialog_name, CMDIALOG_MODELESS);
  page_dialog->SetHelp(help_text);
  page_dialog->DisplayPane(False);
  page_dialog->AddPane(pane,"", CMDIALOG_ATTACH_NONE);
  page_dialog->AddItem(CMDIALOG_PUSHBUTTON,pane,"Execute");
  page_dialog->AddItem(CMDIALOG_TEXTSTRING,pane,textbox);
  page_dialog->AddOption(pane,textbox,
			 CMDIALOG_EDITMODE, (XtArgVal)( CMDIALOG_MULTI_LINE ));
  page_dialog->AddOption(pane,textbox,
			 CMDIALOG_ROWS, (XtArgVal)24);
  page_dialog->AddOption(pane,textbox,
			 CMDIALOG_COLUMNS, (XtArgVal)80);
  page_dialog->AddOption(pane,textbox,
			 CMDIALOG_SCROLLED_TEXT, (XtArgVal)0);

  page_dialog->AddOption(pane,textbox,
		   CMDIALOG_VALUE, (XtArgVal)(page_text.Chars()));

  page_dialog->ManageItems();
  page_dialog->PopupDialog();

  // addcallbacks
  CmEditorDialogWad* newewad = 
    new CmEditorDialogWad(ewad->Editor, page_dialog, ewad->PageTextType);

  XtAddCallback(page_dialog->GetWidget(pane,"Execute"), 
		XmNactivateCallback,(XtCallbackProc) CmExecutePageText, 
		(XtPointer) newewad);
  page_dialog->SetCallbacks((XtCallbackProc) CmUpdatePageText, 
			    (XtPointer) newewad, 
			    (XtCallbackProc) CmPopDownPageText,
			    (XtPointer)page_dialog);
}


/************************
 CreateFigure
  Effects:
   Brand new Widget is about to be made
   When "Edit"-"Create..." is selected, this function is called.
   The figure is created, and edited immediately by the FigureHandler.
   An event handler is added to the new figures, so they may be edited
   through resizing, dragging, and double-clicking.
   It is then added to the editor through AddFigure.
 *************************/

void CmFigureEditor::CreateFigure(CmFigureClass * const cwc) {  
  CreateFigure(cwc, "untitled");
}


void CmFigureEditor::CreateFigure(CmFigureClass * const cwc,
				  const CmString &name) {
  CmFigure  * const new_cmfigure = 
    new CmFigure(Page->PageCmFigure, name, cwc, cds);

  cmdebug << "Created new CmFigure called " << name << "\n";
  new_cmfigure->Resize(0, 0, 200, 100);
  new_cmfigure->Widgetize();
}


/*************************
 EditFigure
  Called by the figure activated routine of the figure handler
  Calls the EditFigure method of the figure_registrar
*************************/
void CmFigureEditor::EditFigure(Widget w) {   
  figure_registrar->EditFigure(w); 
}

/*************************
 ToggleEdit
*************************/
void CmFigureEditor::ToggleEditing() {   
  Editing = cds->Menu()->GetToggleState("Edit", "toggle_edit");
  if (Editing) {
    cds->Menu()->SetSensitive("Edit", "figure_edit", True);
    cds->Menu()->SetSensitive("Edit", "create", True);
    cds->Menu()->SetSensitive("Edit", "cut", True);
    cds->Menu()->SetSensitive("Edit", "copy", True);
    cds->Menu()->SetSensitive("Edit", "paste", True);
    cds->Menu()->SetSensitive("Page", "edit_page_constructor_text", True);
    cds->Menu()->SetSensitive("Page", "edit_page_entry_text", True);
    cds->Menu()->SetSensitive("Page", "edit_page_exit_text", True);
  } else {
    cds->Menu()->SetSensitive("Edit", "figure_edit", False);
    cds->Menu()->SetSensitive("Edit", "create", False);
    cds->Menu()->SetSensitive("Edit", "cut", False);
    cds->Menu()->SetSensitive("Edit", "copy", False);
    cds->Menu()->SetSensitive("Edit", "paste", False);
    cds->Menu()->SetSensitive("Page", "edit_page_constructor_text", False);
    cds->Menu()->SetSensitive("Page", "edit_page_entry_text", False);
    cds->Menu()->SetSensitive("Page", "edit_page_exit_text", False);
    figure_registrar->HidePalette();
  }
}

/*************************
 CutFigure
  Called when the cut item in the menu is activated 
  Callback to this method is added to the menu in CmFigureEditor
  Removes the CmFigure from the page and adds it to the clipboard collection
*************************/
void CmFigureEditor::CutFigure() { 
  CmFigure* const hcw = figure_handler->GetHighlightedCmFigure();
  if ( hcw ) {
    CopyFigure();
    figure_handler->SetHighlightedCmFigure(NULL);
    Page->PageCmFigure->RemoveCmFigure(hcw);
  }
}

/*************************
 CmFigureEditor CopyFigure
  Called when the copy item in the menu is activated 
  Callback to this method is added to the menu in CmFigureEditor
  Creates a new figure just like the highlighted one.
*************************/
void CmFigureEditor::CopyFigure() {
  CmFigure* const hcw = figure_handler->GetHighlightedCmFigure();
  if ( hcw ) {
    if (!clipboarded_cmfigure)
      delete clipboarded_cmfigure;
    clipboarded_cmfigure = hcw->Copy("CDS Clipboard");
  }
}

/*************************
 CmFigureEditor PasteFigure
  Called when the paste item in the menu is activated 
  Callback to this method is added to the menu in CmFigureEditor
  Copies the CmFigure from the clipboard to the current page
*************************/
void CmFigureEditor::PasteFigure() { 
  if (clipboarded_cmfigure)
    clipboarded_cmfigure->Copy(Page->Name, Page->PageCmFigure)->Widgetize();
}

/************************************************************
  CmFigureEditor ChangePage
  Changes page, usually sparked by navigator.
  Changes a page and Widgetizes it (creates the figures)
  A Next/Prev/Outliner selection shall result in this
************************************************************/
void CmFigureEditor::ChangePage(CmPage* n) { 
  // exit a page

  cds->ShowWaitCursor();

  if (Page) {
    Page->PageCmFigure->Unmap();  // erase old screen
    if (!Page->PageExitText.Matches(WhiteSpace))
      cds->Model()->Evaluate(Page->PageExitText);
  }

  Page = n;
  if ( !Page->PageCmFigure ) {
    CmFigure * const cw = new CmFigure(workarea, "testscreen", 
		      figure_registrar->
		      GetCmFigureClass(xmFormWidgetClass),
		      cds);
    Page->PageCmFigure = cw;
    
    if ( Page->ScreenDefinitionString.Length() > 1 ) 
      figure_parser->Parse(cw, Page->ScreenDefinitionString);
    cw->Resize(0, 0, 1000, 1000);
    cw->ChangeResource(XmNfractionBase, "1000");
    cw->Widgetize();

    // construct page
    if (!Page->PageConstructorText.Matches(WhiteSpace))
      cds->Model()->Evaluate(Page->PageConstructorText);
  }
  // show page
  Page->PageCmFigure->Map();
  if (!Page->PageEntryText.Matches(WhiteSpace))
    cds->Model()->Evaluate(Page->PageEntryText);
  cds->ShowNormCursor();
}


/************************************************************
 CmFigureEditor Constructor
   Adds items to the menu pane Edit along with their callbacks
   Creates figure_parser, figure_registrar, figure_handler
   Creates trashcan and clipboard icons and figure collections 
    which are displayed in figure collections (arg) in lower right
 ************************************************************/
CmFigureEditor::CmFigureEditor(CmCDS *c, const Widget warea)  {  
   cds = c; Widget collections = c->Collections(); workarea = warea;
   Page = NULL; clipboarded_cmfigure = NULL;

  figure_registrar = new CmFigureRegistrar(this);
  figure_handler = new CmFigureHandler(this);
  figure_parser = new CmFigureParser(this);
  cmdebug << "   Figure Parser created.\n";
  
  statusbox =  XmCreateLabelGadget(collections, "statusbox", NULL, 0);
  XtManageChild(statusbox);
  
  register_figure_classes();
  cmdebug << " registered figure classes\n";

  cds->Menu()->AddItem("Edit", "create", 
		       (XtCallbackProc) CmPopupPalette, 
		       (XtPointer) figure_registrar);
  cds->Menu()->AddItem("Edit", "cut", 
		       (XtCallbackProc) CmCutFigure, (XtPointer)this);     
  cds->Menu()->AddItem("Edit", "copy", 
		       (XtCallbackProc) CmCopyFigure, (XtPointer)this);
  cds->Menu()->AddItem("Edit", "paste", 
		       (XtCallbackProc) CmPasteFigure, (XtPointer)this);

  cds->Menu()->AddToggle("Edit", "toggle_edit",
		       (XtCallbackProc) CmToggleEdit, 
		       (XtPointer) this);
  ToggleEditing();
  cds->Menu()->AddItem("Page", "edit_page_constructor_text",
		       (XtCallbackProc) CmPageText, 
		       (XtPointer) new CmEditorDialogWad(this, 0, 
							 CMPAGE_CONSTRUCTOR));
  cds->Menu()->AddItem("Page", "edit_page_entry_text",
		       (XtCallbackProc) CmPageText,
		       (XtPointer) new CmEditorDialogWad(this, 0,
							 CMPAGE_ENTRY));
  cds->Menu()->AddItem("Page", "edit_page_exit_text",
		       (XtCallbackProc) CmPageText,
		       (XtPointer) new CmEditorDialogWad(this, 0,
							 CMPAGE_EXIT));

  cds->Menu()->AddItem("Options", "show_all",
		       (XtCallbackProc)CmShowAll, (XtPointer) this);
  cds->Menu()->AddItem("Options", "hide",
		       (XtCallbackProc)CmHide, (XtPointer) this);

}

/**************************************************************
 CmFigureEditor Destructor
  Destroys all CmFigureEditor objects creates in constructor.
 **************************************************************/
CmFigureEditor::~CmFigureEditor() {
   delete figure_parser;
   delete figure_registrar;
   delete figure_handler;
}

/************************************************************
CmFigureEditor SetStatusBox
changes the status box
************************************************************/
void CmFigureEditor::SetStatusBox( CmString status) {
  status += "\n";
  status += cds->Navigator()->CurrentFilename;
  XmString xm_string = CmToXmString(status);
  XtVaSetValues(statusbox, 
		XmNlabelString, (XtArgVal) xm_string, NULL);
  XmStringFree(xm_string);
}

void CmFigureActivated(const Widget w, CmFigureEditor * const figure_editor,
                       XEvent * const event) {
  figure_editor->figure_handler->FigureActivated(w, event);
}

/**************************************************************
 CmFigureRegistrar ShowAll
 called when the user selects Show All
 displays all the figures on the page
 **************************************************************/
void CmFigureEditor::ShowAll() {  
  Page->PageCmFigure->Show();
}

/**************************************************************
 CmFigureRegistrar HideCurrentFigure
 called when the user selects Hide
 Hides the current figure
 **************************************************************/
void CmFigureEditor::HideCurrentFigure() {  
  CmFigure* const cw = GetHighlightedCmFigure();

  if (cw) {
    cw->SetIsMapped(False);
    cw->Unmap();
  }
}

CmFigure *CmFigureEditor::GetHighlightedCmFigure() {
  return figure_handler->GetHighlightedCmFigure();
}

void CmFigureEditor::SetHighlightedCmFigure(CmFigure *const cmw) {
  figure_handler->SetHighlightedCmFigure(cmw);
}

CmFigureClass *CmFigureEditor::GetCmFigureClass(const CmString &class_name) {
  return figure_registrar->GetCmFigureClass(class_name);
}

CmFigureClass *CmFigureEditor::GetCmFigureClass(const WidgetClass wc) {
  return figure_registrar->GetCmFigureClass(wc);
}



