#include <stdio.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <CDS.h>
#include <CmCDS.h>
#include <CmModel.h>
#include <CmFigure.h>
#include <CmFigureEditor.h>
#include <CmNavigator.h>
#include <CmPage.h>
#include <Xm/Xm.h>
#include <X11/StringDefs.h>
#include <CmMenu.h>

extern "C" {
#include <At/Plotter.h>
#include <At/Axis.h>
#include <At/XYPlot.h>
#include <At/Barchart.h>
#include <At/TextPlot.h>
}

void handle_hide_figure(const char * arg, CmModel* s) {
  CmFigure* cmw = s->GetCmFigure(arg);
  if ( cmw ) 
    cmw->Unmap();
}

void handle_show_figure(const char * arg, CmModel* s) {
  CmFigure* cmw = s->GetCmFigure(arg);
  if ( cmw )
    cmw->Show();
}

void handle_set_resource(const char * arg, CmModel* s) {
  CmFigure *cmw = s->GetCmFigure(arg);
  CmString resource_name = s->CDSPtr()->NextToken(arg);
  CmString resource_value = s->CDSPtr()->NextToken(arg);

  if ( cmw ) {
    cmw->ChangeResource(resource_name, resource_value);

    XtVaSetValues(cmw->GetWidget(),
		  XtVaTypedArg, resource_name.Chars(), XtRString,
		  resource_value.Chars(), resource_value.Length() + 1, 
		  NULL);
  }
}

void handle_create_figure(const char * arg, CmModel* s) {
  CmString figure_name = s->CDSPtr()->NextToken(arg);
  CmString figure_class = s->CDSPtr()->NextToken(arg);

  CmFigureClass* cwc = 
    s->CDSPtr()->Editor()->GetCmFigureClass(figure_class);
  if ( cwc ) {
    s->CDSPtr()->Editor()->CreateFigure(cwc);
    CmFigure* ncw = s->CDSPtr()->Editor()->GetHighlightedCmFigure();
    if ( ncw ) 
      ncw->Name = figure_name;
  }
}

void handle_system(const char *arg, CmModel *) {
  system(arg);
}

struct CmWad {
  CmString function_name;
  CmModel* s;
};

void menu_callback_handler(Widget, CmWad* wad, XtPointer) {
  wad->s->Evaluate(wad->function_name);
}

void handle_add_menu_pane(const char *arg, CmModel* s) {
  s->CDSPtr()->Menu()->AddPane(s->CDSPtr()->NextToken(arg));
}

void handle_add_menu_item(const char *arg, CmModel* s) {
  CmString pane_name = s->CDSPtr()->NextToken(arg);
  CmString item_name = s->CDSPtr()->NextToken(arg);
  s->CDSPtr()->Menu()->AddItem(pane_name, item_name);
}

void handle_add_menu_function(const char *arg, CmModel* s) {
  CmString pane_name = s->CDSPtr()->NextToken(arg);
  CmString item_name = s->CDSPtr()->NextToken(arg);
  CmWad* wad = new CmWad; //temporary data structure to pass to callback
  CmMenu *menu = s->CDSPtr()->Menu();
  wad->function_name = arg; // send name of function to eval
  wad->s = s; // send model in which to evaluate it
  if (menu->GetItem(pane_name, item_name))
    XtAddCallback(menu->GetItem(pane_name, item_name), XmNactivateCallback,
		  (XtCallbackProc) menu_callback_handler, (XtPointer) wad);
}

void handle_remove_menu_function(const char *arg, CmModel* s) {
  CmString pane_name = s->CDSPtr()->NextToken(arg);
  CmString item_name = s->CDSPtr()->NextToken(arg);
  CmMenu *menu = s->CDSPtr()->Menu();
  XtRemoveAllCallbacks(menu->GetItem(pane_name, item_name), XmNactivateCallback);
}

void handle_exit_cds(const char *, CmModel* s) {
  s->CDSPtr()->Navigator()->ExitCDS();
}

void handle_show_wait_cursor(const char *, CmModel* s) {
  cmdebug << "handling wait\n";
  s->CDSPtr()->ShowWaitCursor();
}

void handle_show_norm_cursor(const char *, CmModel* s) {
  s->CDSPtr()->ShowNormCursor();
}

void handle_goto_page(const char *arg, CmModel* s) {
  s->CDSPtr()->Navigator()->Navigate("goto", s->CDSPtr()->NextToken(arg));
}

void handle_open_file(const char *arg, CmModel* s) {
  s->CDSPtr()->Navigator()->OpenFile(s->CDSPtr()->NextToken(arg));
}

void handle_change_title(const char *arg, CmModel* s) {
  s->CDSPtr()->Navigator()->ChangeTitle(s->CDSPtr()->NextToken(arg));
}

void handle_place_figure(const char *arg, CmModel* s) {
  CmFigure *cmw = s->GetCmFigure(arg);
  if (cmw) {
    int x = atoi(s->CDSPtr()->NextToken(arg));
    int y = atoi(s->CDSPtr()->NextToken(arg));
    int width = atoi(s->CDSPtr()->NextToken(arg));
    int height = atoi(s->CDSPtr()->NextToken(arg));

    cmw->Resize(x, y, width, height);
  }
}

void handle_toggle_editing(const char *, CmModel* s) {
  s->CDSPtr()->Editor()->ToggleEditing();
}

void handle_cut_figure(const char *arg, CmModel* s) {
  CmFigure *cmw = s->GetCmFigure(arg);
  if ( cmw ) {
    s->CDSPtr()->Editor()->SetHighlightedCmFigure(cmw);
    s->CDSPtr()->Editor()->CutFigure();
  }
}

void handle_copy_figure(const char *arg, CmModel* s) {
  CmFigure *cmw = s->GetCmFigure(arg);
  if ( cmw ) {
    s->CDSPtr()->Editor()->SetHighlightedCmFigure(cmw);
    s->CDSPtr()->Editor()->CopyFigure();
  }
}

void handle_paste_figure(const char *, CmModel* s) {
  s->CDSPtr()->Editor()->PasteFigure();
}

void handle_trace_level(const char *arg, CmModel* s) {
  CmString level_string  = s->CDSPtr()->NextToken(arg);
  if (!level_string.Empty())
    s->SetTraceLevel(atoi(level_string));
}

void handle_error(const char * arg, CmModel *) {
  cerr << "Error: " << arg;
}

void handle_call_callback(const char *arg, CmModel *s) {
  CmFigure *cmw = s->GetCmFigure(arg);
  CmString callback = s->CDSPtr()->NextToken(arg);

  s->Evaluate(cmw->GetCallbackString(callback));
}

void handle_call_action_proc(const char *arg, CmModel *s) {
  CmFigure *cmw = s->GetCmFigure(arg);
  CmString action = s->CDSPtr()->NextToken(arg);

  if (cmw && cmw->GetWidget())
    XtCallActionProc(cmw->GetWidget(), action.Chars(), NULL, NULL, 0);
}


void handle_plot(const char *arg, CmModel* s) {
  CmFigure *figure = s->GetCmFigure(arg);

  if (!figure || !figure->GetWidget())
    return;

  CmString command;
  while(1) {
    command = s->CDSPtr()->NextToken(arg);
    if (command.Empty())
      break; 
    else
    if (command == "barchart") {
      CmString barchart_name = s->CDSPtr()->NextToken(arg);
      
      int npoints = atoi(s->CDSPtr()->NextToken(arg));
      double *data = new double[npoints];
      
      int i =0;
      while (i < npoints)
	data[i++] = atof(s->CDSPtr()->NextToken(arg));
      
      // Notify Xt of new data.
      Widget barchart = 
	XtNameToWidget(figure->GetWidget(), barchart_name.Chars());
      
      if (!barchart) {
	AtAxisObject xaxis = 
	  AtPlotterGetXAxis((AtPlotterWidget) figure->GetWidget());
	double xmin, xmax; // retrieve x axis min and max from plotter

	XtVaGetValues((Widget) xaxis, 
		      XtNmin, &xmin,
		      XtNmax, &xmax, NULL);

	
	XtVaCreateManagedWidget(barchart_name.Chars(), atBarchartWidgetClass,
				figure->GetWidget(),
				XtNnumPoints, npoints,
				XtNyPoints, data, 
				XtNxMin, &xmin,
				XtNxMax, &xmax, NULL);
      }
      else 
	XtVaSetValues(barchart, 
		      XtNnumPoints, npoints,
		      XtNyPoints, data, NULL);
      delete data;
    } 
    else if (command == "xyplot") {
      CmString xyplot_name = s->CDSPtr()->NextToken(arg);
      int npoints = atoi(s->CDSPtr()->NextToken(arg));
      
      double *xdata = new double[npoints];
      double *ydata = new double[npoints];
      
      int resolution = 0;
      do {
	xdata[resolution] = atof(s->CDSPtr()->NextToken(arg));
	ydata[resolution] = atof(s->CDSPtr()->NextToken(arg));
	resolution++;
      } while (resolution < npoints);

      Widget xy_plot = 
	XtNameToWidget(figure->GetWidget(), xyplot_name.Chars());
      
      if ( ! xy_plot )
	XtVaCreateManagedWidget(xyplot_name, atXYPlotWidgetClass,
				figure->GetWidget(),
				XtNnumPoints, resolution,
				XtNxPoints, xdata,
				XtNyPoints, ydata,
				XtNmarkPoints, False, NULL);
      else 
	XtVaSetValues(xy_plot,
		      XtNnumPoints, resolution,
		      XtNxPoints, xdata,
		      XtNyPoints, ydata, NULL);
      delete xdata;
      delete ydata;
    }
    else if (command == "text") {
      CmString text_name = s->CDSPtr()->NextToken(arg);
      double pos_x = atof(s->CDSPtr()->NextToken(arg));
      double pos_y = atof(s->CDSPtr()->NextToken(arg));
      CmString text_value = s->CDSPtr()->NextToken(arg);
      
      Widget text_plot =
	XtNameToWidget(figure->GetWidget(), text_name.Chars());
      if ( !text_plot) 
	XtVaCreateManagedWidget(text_name, atTextPlotWidgetClass,
				figure->GetWidget(),
				XtNfloatingPosition, True,
				XtNfloatingX, &pos_x,
				XtNfloatingY, &pos_y,
				XtNlabel, text_value.Chars(), NULL);
      else
	XtVaSetValues(text_plot,
		      XtNfloatingPosition, True,
		      XtNfloatingX, &pos_x,
		      XtNfloatingY, &pos_y,
		      XtNlabel, text_value.Chars(), NULL);
    }	       
    else if (command == "set-plot-resource") {
      CmString plot_name = s->CDSPtr()->NextToken(arg);
      CmString resource = s->CDSPtr()->NextToken(arg);
      CmString value =  s->CDSPtr()->NextToken(arg);

      // Obtain the barchart figure itself
      Widget plot =
	XtNameToWidget(figure->GetWidget(), plot_name.Chars());
      // Due to a bug in the Xt Intrinsic VarArgs.c routines we cannot as
      // of R4 pass a double using this TypedArg.  Thus if the 
      // resource is of type double, we do the resource conversion
      // ourselves
      if (plot)  {
	if (resource == "density" || resource == "xMax" || resource == "xMin") {
	  double foo = atof(value.Chars());
	  XtVaSetValues(plot,
			resource.Chars(), &foo, NULL);
	} else
	  XtVaSetValues(plot,
			XtVaTypedArg, resource.Chars(), XtRString,
			value.Chars(), value.Length() + 1, NULL);
      }
    } else if (command == "delete-plot") {
      Widget plot;
      CmString plot_name = s->CDSPtr()->NextToken(arg);
      while (plot = XtNameToWidget(figure->GetWidget(), plot_name.Chars()))
	XtDestroyWidget(plot);
    } else if (command == "axis") {
      CmString xmin_string  = s->CDSPtr()->NextToken(arg); 
      CmString xmax_string  = s->CDSPtr()->NextToken(arg); 
      CmString ymin_string  = s->CDSPtr()->NextToken(arg); 
      CmString ymax_string  = s->CDSPtr()->NextToken(arg); 
      figure->ChangeResource("xaxis.min", xmin_string);
      figure->ChangeResource("xaxis.max", xmax_string);
      figure->ChangeResource("yaxis.min", ymin_string);
      figure->ChangeResource("yaxis.max", ymax_string);

      AtPlotterSetAxisBounds((AtPlotterWidget) figure->GetWidget(),
			     atof(xmin_string.Chars()),
			     atof(xmax_string.Chars()),
			     atof(ymin_string.Chars()),
			     atof(ymax_string.Chars()));
    } 
  }
}

void CmModel::FindMethod(const char *command) {
  CmModelMethodProc cmm;

  switch(command[0]) {
  case 'a': /* cds */
    switch(command[1]) {
    case 'a':
      cmm = handle_exit_cds; break;
    case 'b':
      cmm = handle_show_norm_cursor; break;
    case 'c':
      cmm = handle_show_wait_cursor; break;
    case 'd':
      cmm = handle_system; break;
    default:
      return;
    } break;
  case 'b': /* editor */
    switch(command[1]) {
    case 'a':
      cmm = handle_cut_figure; break;
    case 'b':
      cmm = handle_copy_figure; break;
    case 'c':
      cmm = handle_paste_figure; break;
    case 'd':
      cmm = handle_toggle_editing; break;
    default:
      cmm = NULL; break;
    } break;
  case 'c': /* figure */
    switch(command[1]) {
    case 'a':
      cmm = handle_call_action_proc; break;
    case 'b':
      cmm = handle_call_callback; break;
    case 'c':
      cmm = handle_create_figure; break;
    case 'd':
      cmm = handle_hide_figure; break;
    case 'e':
      cmm = handle_place_figure; break;
    case 'f':
      cmm = handle_set_resource; break;
    case 'g':
      cmm = handle_show_figure; break;
    default:
      return;
    } break;
  case 'd': /* menu */
    switch (command[1]) {
    case 'a':
      cmm = handle_add_menu_function; break;
    case 'b':
      cmm = handle_add_menu_item; break;
    case 'c':
      cmm = handle_add_menu_pane; break;
    case 'd':
      cmm = handle_remove_menu_function; break;
    default:
      return;
    } break;
  case 'e': /* model */
    switch (command[1]) {
    case 'a':
      cmm = handle_error; break;
    case 'b':
      cmm = handle_trace_level; break;
    default:
      return;
    } break;
  case 'f': /* navigator */
    switch (command[1]) {
    case 'a':
      cmm = handle_goto_page; break;
    case 'b':
      cmm = handle_open_file; break;
    case 'c':
      cmm = handle_change_title; break;
    default:
      return;
    } break;
  case 'g': /* plotter */
    switch (command[1]) {
    case 'a':
      cmm = handle_plot; break;
    default:
      return;
    } break;
  default:
    return;
  }

 // skip past the opcode
  while (*command && !isspace(*command))
    command++;

 // skip past the spaces, command should now be pointing
 // at the beginning of the arguments

  while (*command && isspace(*command)) 
    command++;                    

  cmm(command, this);
}
