#include <CmModelConnection.h>
#ifdef __GNUG__
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/time.h>
#else
#ifdef SYSV
#include <unistd.h>
#else 
#include <sys/wait.h>
#include <sys/fcntl.h>
#endif
#include <sysent.h>  //Files needed for AT&T C++
#endif

#include <setjmp.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sgtty.h>
#include <ctype.h>
#include <expect.h>
#include <signal.h>

// Somehow the procedures below are not it any header file.
extern "C" {
  int setenv(char *, char *, int);
  void unsetenv(char *);
}

// This callback tries to send part of the connection buffer to the
// backend procedure.  It is called each time the Xt loop goes
// We set the write buffer to no delay and count how many characters have
// been sent.

int CmModelConnectionWorkProc(CmModelConnection* connection) {
  if (connection->is_blocked || connection->connect_buffer.Empty())
    return False;

  int num_sent =
    write(connection->master, connection->connect_buffer.Chars(),
	  connection->connect_buffer.Length());

  if (num_sent != -1)
    connection->connect_buffer.Del(0, num_sent);
  return (False);
}

// Block till you get input.  Otherwise, there is a chance that you'd be 
// sending input to a process that hasn't started up yet.  This causes sh
// to freeze until you hit evaluator_interrupt.

void CmModelConnectionCallback(CmModelConnection *connection,
			      int *source, XtInputId *input) {
  if (connection->is_blocked) {
    connection->is_blocked = False;
    connection->SetEcho(connection->has_echo);  
                        // This is to insure that the Model Connection echo is
                        // set correctly
  }
 
  if (connection->connection_callback) 
    (connection->connection_callback)(connection->connection_client_data, 
				      source, input);
}

CmModelConnection::CmModelConnection(const CmString &command,
				     XtAppContext ac) {
  struct sgttyb s;
  input_id = 0;
  work_id = 0;
  child_pid = 0;
  is_blocked = True;
  connection_callback = NULL;

  app_context = ac;
  io_stream = exp_popen((char *) command.Chars());
  if (io_stream) {
    master = fileno(io_stream);
    input_id = 
      XtAppAddInput(app_context, master, 
		    (XtPointer) XtInputReadMask, 
		    (XtInputCallbackProc) CmModelConnectionCallback, 
		    (XtPointer) this);
    work_id = 
      XtAppAddWorkProc(app_context, 
		       (XtWorkProc) CmModelConnectionWorkProc,
		       (XtPointer) this);
    child_pid = exp_pid;
    setpgrp(child_pid, child_pid);
      fcntl(master, F_SETFL, FNDELAY);
    ioctl(master, TIOCGETP, &s);
    s.sg_flags &=
      ~( CRMOD | ANYP | ALLDELAY | RAW | LCASE | CBREAK | TANDEM);
    s.sg_flags |= ECHO;
    ioctl(master, TIOCSETP, &s);
    has_echo = True;
  }
}

void CmModelConnection::AddInput(XtInputCallbackProc callback, 
				 XtPointer client_data) {
  connection_callback = callback;
  connection_client_data = client_data;
}

CmModelConnection::~CmModelConnection() {
    // If there is a child process
  if (child_pid) {
    // remove the input
    XtRemoveInput(input_id);
    XtRemoveWorkProc(work_id);


    // close the connection to the child
    close(master);

    // Kill all the child processes.  I don't know if this should be
    // a SIGTERM or SIGKILL
    if (!SendSignal(SIGKILL)) {
      int wait_pid = 0;
      while(wait_pid != (child_pid) && wait_pid != -1)
	wait_pid = wait(0);               // Wait till the child dies
    }
  }
}

int CmModelConnection::Read(char *input, int size) {
  return read(master, input, size);
}

// Because the UNIX tty buffer is only 255 characters long 
// We have to keep our own buffer so that all the commands to
// the interpreter backend are sent.  We add to this buffer here,
// and rely on CmModelConnectionWorkProc to actually send the buffer

void CmModelConnection::Write(CmString message) {
  connect_buffer += message;
}

/*********
CmModel::SetEcho
requires: TRUE to set, FALSE to unset
effects: turns on and off echoing to the evaluator
returns: nothing
**********/

void CmModelConnection::SetEcho(Boolean echo_set) {
  struct sgttyb b;
  has_echo = echo_set;
  ioctl(master, TIOCGETP, &b);
  
  if (echo_set)
    b.sg_flags |= ECHO;
  else
    b.sg_flags &= ~ECHO;
  
  ioctl(master, TIOCSETP, &b);
}

int CmModelConnection::SendSignal(int sig) {
  return killpg(child_pid, sig);
}

void CmModelConnection::Interrupt() {
  SendSignal(SIGINT);
}
