/*********************************************************************
 * File: ltxo.c 
 * This file contains public routines for LTxo.  
 * It contains routines written by John Barba '88 and Linh Giang '89.  
 *********************************************************************/

#include <stdio.h>		/* C include files */
#include <strings.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <ltxo.h>		/* Public LTxo include file */
#include <ltxo_pri.h>		/* Private LTxo include file */

/* extern variables to be used in ltxo_pri.c & ltxo_fill.c */
Display *LTx_dpy;
char *calloc();	
unsigned long Blackcolor, Whitecolor; 

unsigned long Backgd, Foregd; /* user's preferred colors */

/* Bitmap for stipple field in GC */
static char stipple_bits[] = {
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
  0xaa, 0xaa, 0xaa, 0xaa, 0x55, 0x55, 0x55, 0x55,
};


/* tables defined in ltxo_tbl.c */
extern font_map font_tbl[][MAXSIZE];  
extern family_map family_tbl[];       

/* default GCs created in LTxLoadFont  */
GC LTxText_gc;  

/* GC used to set background */

GC LTxBackgd_gc;

/* external private routines defined in ltxo_pri.c & ltxo_fill.c */
extern char *text_width(), *text_height();
extern char *text_out(), *fill_text_out();
extern char *fill_text_width(), *fill_text_height();  
extern void ltxErrAlert();
extern int ltxErrFlag;		/* error flag to back out */

/* The following global variables are used by almost every routine in 
   the file.  The reason behind having these global variables is that 
   most of the width and height finding routines are recursive and
   therefore require global variables to hold running totals.  As a
   convention in this file, the global variables have a capitalized
   first letter. */

Window Win;			/* window to draw text on */
XRectangle Box;			/* LTxo bounding box */

unsigned Boxwidth, Boxheight;   /* width & height of 
				   bounding box in pixels */
int X, Y;              /* x,y of bounding box in pixels - use 
			  for determining the x,y of baseline */

int Xbase, Ybase;          /* x,y of baseline to draw text */

unsigned Current;	   /* current line index */
unsigned MaxCurrent;	/* the actual max lines of text output */

int *Linewidth, *Lineheight;   /* arrays holding each text 
				  line's width and height */

int *Lineascent;		/* array holding each text line's 
				   ascent - use for determining
				   the baseline. */

int *Linenumchar;	/* array holding num of chars per line */

int Currentlength;	/* length of current line -- use for 
			   fill_text_height & fill_text_out */

/* The following holds information about symbol & math formatting */
unsigned MaxCount, Count;
int *Topwidth, *Botwidth;
int *Topdescent, *Botascent;
int Overflag;



/**********************************************************************
 * Public Routine: LTxLoadFont(win)
 * Requires: 
 *  dpy = display.
 * Description: 
 *  This routine loads in the set of popular fonts and creates
 *  several GCs. Whenever a font is needed and the available GCs are
 *  not of this font, one of these GCs will be set to the new font.
 *  Each GC will be checked to determine if it can be used, and only
 *  if none can, should one of the GCs be changed. This implementation
 *  design is a compromise between speed efficiency and memory usage
 *  but is needed in order to have many fonts available for users.
 *  Each application should not use too many fonts because it is
 *  aesthetically distracting but fonts of different sizes are needed
 *  to be available for resizing.    
 **********************************************************************/

void LTxLoadFont(dpy)
     Display *dpy;
{
  XGCValues vals;
  unsigned long valuemask;
  Window win;

#ifdef LTXO_DEBUG 
  printf("LTxLoadFont(display)\n");
#endif 
  
  /* Before doing anything, check if any of the GCs have been created 
     already because more than one routine called LTxLoadFont(). If
     they have been created, just return. */

  if (LTxText_gc) {
    return;
  }

  LTx_dpy = dpy;

  win = RootWindow(LTx_dpy, DefaultScreen(LTx_dpy));
  valuemask = GCForeground | GCBackground |
    GCCapStyle | GCJoinStyle | GCFillStyle | GCGraphicsExposures |
      GCFunction | GCStipple; 

  vals.function = GXcopy;  
  vals.line_style = LineSolid;
  vals.cap_style = CapButt;
  vals.join_style = JoinMiter;
  vals.stipple = XCreateBitmapFromData(LTx_dpy, win, stipple_bits, 32, 32);
  vals.fill_style = FillSolid;
  vals.graphics_exposures = False;
  vals.background = Whitecolor = WhitePixel(LTx_dpy, DefaultScreen(LTx_dpy));
  vals.foreground = Blackcolor = BlackPixel(LTx_dpy, DefaultScreen(LTx_dpy));
 
  /* Create two different GCs. */ 
  LTxText_gc = XCreateGC(LTx_dpy, win, valuemask, &vals);
  LTxBackgd_gc = XCreateGC(LTx_dpy, win, valuemask, &vals);
} 

/**********************************************************************
 * Public Routine: LTxCreateInfo()
 * Returns:
 *  Pointer to the LTxInfo structure that holds information for the
 *  text output. 
 * Description:
 *  Returns a pointer to the  LTxoInfo structure with the default
 *  values. Any changes should be set with the convenience procedures. 
 *  Note: The default values have nothing to do with each individual
 *        pidgets using LTxo.  The user has to set the necessary 
 *        information after creating the structure.
 **********************************************************************/

LTxInfo *LTxCreateInfo()
{
  LTxInfo *info;

  if((info = (LTxInfo *)calloc((unsigned)1 , sizeof(LTxInfo))) ==
     (LTxInfo *)0) 
    printf("Out of Memory! (LTxCreateInfo)");

  LTxSetSize(info, MEDIUM);
  LTxSetXflag(info, H_LEFT);
  LTxSetYflag(info, V_CENTER);
  LTxSetMode(info, VERBATIM);
  LTxSetResize(info, ON);
  LTxSetSmall(info, CLIP_RIGHT_BOT);
  LTxSetForegd(info, Blackcolor);
  LTxSetBackgd(info, Whitecolor);
  LTxSetXoffset(info, 0);
  LTxSetYoffset(info, 0);

  /* Other defaults: dimensions of bounding box are all zero.
                     border is zero.
		     text is null.
		     font is HELVET.
		     short text is null. */

  return(info);
}



/**********************************************************************
 * Public Routine: LTxDrawText(win, win_width, win_height, info)
 * Requires: 
 *  win  = window for text output
 *  win_height = height of window
 *  win_width = width of window
 *  info = a pointer LTxInfo structure that holds information for text
 * Description:
 *  Draws the text given in the LTxInfo structure. If resize is
 *  set to ON, LTxDrawText will try to find smaller size fonts to
 *  fit in  the bounding box given.  If resize is set to OFF, then
 *  it checks for the too_small flag to decide what to do with text.
 *  The default is to clip arbitrarily from the right and bottom.
 **********************************************************************/

void LTxDrawText(win, win_width, win_height, info)
     Window win;
     unsigned win_height, win_width;
     LTxInfo *info;
{
  unsigned font, size, border;	/* variables to hold fields of info */
  int resize_flag, mode, xflag, yflag;
  char *text;
  float per_width, per_height; /* fraction width & height */
  int tmp;		
  int maxwidth;          /* max of all widths */
  int totheight;         /* total height of text */ 
  int numchar = 0;     /* num of char per word - for fill & justify env */
  int wtotal, dtotal, atotal;  /* vars to send to find width & height */ 

#ifdef LTXO_DEBUG
  printf("Passed to LTxDrawText() ... \n");
  printf(" x: %f\n y: %f\n height: %f\n width: %f\n border: %d\n",
	 LTxGetX(info), LTxGetY(info), LTxGetHeight(info),
	 LTxGetWidth(info), LTxGetBorder(info));
  printf(" font: %d\n size: %d\n xflag: %d\n yflag: %d\n",
	 LTxGetFont(info), LTxGetSize(info), LTxGetXflag(info),
	 LTxGetYflag(info), LTxGetX(info));
  printf(" mode: %d\n resize: %d\n too_small: %d\n str: %s\n short:%s\n", 
	 LTxGetMode(info), LTxGetResize(info), LTxGetSmall(info),
	 LTxGetText(info), LTxGetShort(info)); 
  printf(" win_height: %u\n win_width: %u\n", win_height, win_width);
#endif

  /* Set automatic variables for easy reference. */
  font = LTxGetFont(info);
  size = LTxGetSize(info);
  text = LTxGetText(info);
  border = LTxGetBorder(info);
  resize_flag = LTxGetResize(info);
  mode = LTxGetMode(info);
  xflag = LTxGetXflag(info);
  yflag = LTxGetYflag(info);
  per_width = LTxGetWidth(info);
  per_height = LTxGetHeight(info);

  /* set current window to draw text on */
  Win = win;		 

  /* Set global variable for GC changes */
  Foregd = LTxGetForegd(info);
  Backgd = LTxGetBackgd(info);

  /* Check if the font and size ids are within range & that the 
     the bounding box's dimensions' are non-zero. */
  if(font >= MAXFONT) {
    ltxErrAlert("Invalid Font Id!", " ");
    ltxErrFlag = 0;
    return;
  }
  if(size >= MAXSIZE) {
    ltxErrAlert("Invalid Font Size!", " ");
    ltxErrFlag = 0;
    return;
  }
  if( per_width == 0.0 || per_height == 0.0) {
    ltxErrAlert("Bounding Box has either zero width and/or height!"," ");
    ltxErrFlag = 0;
    return;
  }

  /* This is a waste bec. only GCs that are actually being use should */
  /* be changed. Should put in text_draw routine in ltxo_pri.c */

  /* Set GCs to new preferred foreground & background if any */

  XSetForeground(LTx_dpy, LTxText_gc, Foregd);
  XSetBackground(LTx_dpy, LTxText_gc, Backgd);
  XSetForeground(LTx_dpy, LTxBackgd_gc, Backgd);

  /* Set stipple on if user wants it. */
  if ( LTxGetWantStipple(info) )
    XSetFillStyle(LTx_dpy, LTxText_gc, FillOpaqueStippled);
  else
    XSetFillStyle(LTx_dpy, LTxText_gc, FillSolid);
   
  /* Set static variables for all other routines in file to use. 
     And set up x,y and bounding box width and height in actual pixels
     from the normalized coordinates given. */
  X = Y = 0;		   /* assume flush left and top oriented */

  /* Set XRectangle x,y for clip_mask for GC */
  Box.x = (int)((float)LTxGetX(info) * (float)win_width) 
    + LTxGetXOffset(info);
  Box.y = (int)((float)LTxGetY(info) * (float)win_height)
    + LTxGetYOffset(info);

  /* Set XRectangle width and height */
  Boxwidth = (unsigned) (per_width * (float)win_width); 
  Boxheight = (unsigned) (per_height * (float)win_height); 
  Box.width = Boxwidth;	      /* redundant but too late to make change */
  Box.height = Boxheight;

  /* Clear area if text is different */
  if (LTxGetChangeTextFlag(info)) {
    LTxSetChangeTextFlag(info, OFF);
  }

  /* Set the bounding box rectangle now - could also be done in */
  /* text_draw in ltxo_pri.c */
  XSetClipRectangles(LTx_dpy, LTxText_gc, 0,0, &Box, 1, Unsorted);

#ifdef LTXO_DEBUG 
  printf("Actual Box.x: %d  Box.y:%d\n", Box.x, Box.y);
  printf("Actual Boxwidth: %u\n", Box.width);
  printf("Actual Boxheight: %u\n", Box.height);
#endif 

  /* Now, everything is set: allocate space for temporary variables */
  /* holding line by line info */ 
  if((Linewidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Linewidth)");
  if((Lineheight = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Lineheight)");
  if((Lineascent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Lineascent)");
  if((Topwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Topwidth)");
  if((Botwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Botwidth)");
  if((Topdescent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Topdescent)");
  if((Botascent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Botascent)");

  if ( mode & (FILL | JUSTIFY)) {
    if((Linenumchar = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
       == (int *)0)
      XtError("Out of Memory! (LTxo*Linenumchar)");
  }

  /* Call width and height computing routines to determine the width 
     and height of the complex text. 
     The width of each line is held in the 'Linewidth' array. 
     The height of each line is held in the 'Lineheight' array.  
     The ascent of each line is held in the 'Lineascent' array which 
     is needed to determine the baseline of the text to be drawn.
     Fill env: 'Linenumchar' holds the number of chars per line.
     'Current' keeps track of which line we are on . */

  Current = Count = wtotal = 0;

  atotal = get_font_struct(font, size)->ascent;
  dtotal = get_font_struct(font, size)->descent;
  switch ( mode ) {
    case VERBATIM:
      (void) text_width( font, size, &wtotal, text, (char)0 );
      if (ltxErrFlag) {
	ltxErrFlag = 0;
	 
	return;
      }
      Current = Count = 0;
      (void) text_height( font, size, &atotal, &dtotal, text, (char)0 );
      break;
    case FILL:
    case JUSTIFY:
      (void) fill_text_width( font, size, &wtotal, &numchar, text,
			     (char)0 ); 
      if (ltxErrFlag) {
	ltxErrFlag = 0;
	 
	return;
      }
      Current = Count = 0;
      Currentlength = Linenumchar[Current];
      (void) fill_text_height( font, size, &atotal, &dtotal, text, (char)0 );  
      break;
    default:
      ltxErrAlert("Invalid Mode For Text!", " ");
      ltxErrFlag = 0;
       
      return;
    }

  maxwidth = totheight = 0; 
  for (tmp = 0; tmp <= MaxCurrent; tmp++) {
    maxwidth = max(maxwidth, Linewidth[tmp]);
    totheight += Lineheight[tmp];
  }

#ifdef LTXO_DEBUG
  printf("Got back from width routine ... maxwidth: %d\n", maxwidth);
  printf("Got back from height routine ... totheight: %d\n", totheight);
  for (tmp = 0; tmp <= MaxCurrent; tmp++) {
    printf("Current: %d\n", tmp);
    printf(" Linewidth: %d\n Lineheight: %d\n Lineascent: %d\n",
	   Linewidth[tmp], Lineheight[tmp], Lineascent[tmp]);
    if (mode & (FILL | JUSTIFY))
      printf(" Linenumchar: %d\n", Linenumchar[tmp]);
  }
#endif

  /* This is the loop which tries to find the best size font to fit into */
  /* bounding box.  Once LTxLoadFont is deleted, this loop needs to check */
  /* the XFontStruct for the given font is loaded and if not call */
  /* load_font(font_id, size_id).  You should also check the following */
  /* places for related problems: */
  /*    ltxo_pri.c:  text_width(), text_height() */
  /*    ltxo_fill.c:  fill_text_width(), fill_text_height() */

  /* lkgiang 6/3/89 */

  while ((maxwidth > Boxwidth || totheight > Boxheight) && resize_flag
	 && (size != SMALLEST)) { 
    /* clear variables from earlier computations */
    for (tmp = 0; tmp <= MaxCurrent; tmp++)	
      Linewidth[tmp] = Lineheight[tmp] = Lineascent[tmp] = 0;
    for (tmp = 0; tmp <= MaxCount; tmp++) {
      Topwidth[tmp] = Botwidth[tmp] = Topdescent[tmp] =
	Botascent[tmp] = 0;
    }
    size--;
    Current = Count =  wtotal = 0;
    
    atotal = get_font_struct(font, size)->ascent;
    dtotal = get_font_struct(font, size)->descent;
    switch ( mode ) {
      case VERBATIM:
	(void) text_width( font, size, &wtotal, text, (char)0 );  
	Current = Count = 0;
	(void) text_height( font, size, &atotal, &dtotal, text, (char)0 );  
	break;
      case FILL:
      case JUSTIFY:
	/* first clear variables from earlier computations */
	for (tmp = 0; tmp <= MaxCurrent; tmp++)
	  Linenumchar[tmp] = 0;
	numchar = 0;
        (void) fill_text_width( font, size, &wtotal, &numchar, text,
			       (char)0 ); 
	Current = Count = 0;
        Currentlength = Linenumchar[Current];
	(void) fill_text_height( font, size, &atotal, &dtotal, text,
				(char)0 );
	break;
      default:
	ltxErrAlert("Invalid Mode For Text!", " ");
	ltxErrFlag = 0;
	 
	return;
      }

    maxwidth = totheight = 0;
    for (tmp = 0; tmp <= MaxCurrent; tmp++)  {
      maxwidth = max(maxwidth, Linewidth[tmp]);
      totheight += Lineheight[tmp];
    }

#ifdef LTXO_DEBUG
    printf("Got back from width routine ... maxwidth: %d\n", maxwidth);
    printf("Got back from height routine ... totheight: %d\n", totheight);
    for (tmp = 0; tmp <= MaxCurrent; tmp++) {
      printf("Current: %d\n", tmp);
      printf(" Linewidth: %d\n Lineheight: %d\n Lineascent: %d\n",
	     Linewidth[tmp], Lineheight[tmp], Lineascent[tmp]);
      if (mode & (FILL | JUSTIFY))
	printf(" Linenumchar: %d\n", Linenumchar[tmp]);
    }
#endif
  }
				
  /*  Box still too small to hold text */
  if (maxwidth > Boxwidth || totheight > Boxheight) {
    switch(LTxGetSmall(info)) {
    case SHOW_SHORT:
      text = LTxGetShort(info);
      /* clear variables from earlier computations */
      for (tmp = 0; tmp <= MaxCurrent; tmp++)	
	Linewidth[tmp] = Lineheight[tmp] = Lineascent[tmp] = 0;
      for (tmp = 0; tmp <= MaxCount; tmp++) {
	Topwidth[tmp] = Botwidth[tmp] = Topdescent[tmp] =
	  Botascent[tmp] = 0;
      }
      Current = Count = wtotal = 0;
      atotal = get_font_struct(font, size)->ascent;
      dtotal = get_font_struct(font, size)->descent;
      switch ( mode ) {
      case VERBATIM:
	(void) text_width( font, size, &wtotal, text, (char)0 );  
	Current = Count = 0;
	(void) text_height( font, size, &atotal, &dtotal, text, (char)0 );  
	break;
      case FILL:
      case JUSTIFY:
	/* first clear variables from earlier computations */
	for (tmp = 0; tmp <= MaxCurrent; tmp++)
	  Linenumchar[tmp] = 0;
	numchar = 0;
	(void) fill_text_width( font, size, &wtotal, &numchar, text,
			       (char)0 ); 
	Current = Count = 0;
	Currentlength = Linenumchar[Current];
	(void) fill_text_height( font, size, &atotal, &dtotal, text,
				(char)0 );
	break;
      default:
	ltxErrAlert("Invalid Mode For Text!", " ");
	ltxErrFlag = 0;
	 
	return;
      }
      
      maxwidth = totheight = 0;
      for (tmp = 0; tmp <= MaxCurrent; tmp++)  {
	maxwidth = max(maxwidth, Linewidth[tmp]);
	totheight += Lineheight[tmp];
      }
      
      break;
    case CLIP_RIGHT_BOT:
      if (maxwidth > Boxwidth) {
	switch( xflag ) {
	case H_CENTER:
	  X = (maxwidth - Boxwidth)/2;
	  break;
	case H_RIGHT:
	  X = (maxwidth - Boxwidth);
	  break;
	}}
      if (totheight > Boxheight) {
	switch( yflag ) {
	case V_CENTER:
	  Y = (totheight - Boxheight)/2;
	  break;
	case V_BOTTOM:
	  Y = (totheight - Boxheight);
	  break;
	}}
      break;
    case CLIP_LEFT_BOT:
      if (maxwidth > Boxwidth) {
	switch( xflag ) {
	case H_CENTER:
	  X = (Boxwidth - maxwidth)/2;
	  break;
	case H_RIGHT:
	  X = (Boxwidth - maxwidth);
	  break;
	}}
      if (totheight > Boxheight) {
	switch( yflag ) {
	case V_CENTER:
	  Y = (totheight - Boxheight)/2;
	  break;
	case V_BOTTOM:
	  Y = (totheight - Boxheight);
	  break;
	}}
      break;
    case CLIP_RIGHT_TOP:
      if (maxwidth > Boxwidth) {
	switch( xflag ) {
	case H_CENTER:
	  X = (maxwidth - Boxwidth)/2;
	  break;
	case H_RIGHT:
	  X = (maxwidth - Boxwidth);
	  break;
	}}
      if (totheight > Boxheight) {
	switch( yflag ) {
	case V_CENTER:
	  Y = (Boxheight - totheight)/2;
	  break;
	case V_BOTTOM:
	  Y = (Boxheight - totheight);
	  break;
	}}
      break;
    case CLIP_LEFT_TOP:
      if (maxwidth > Boxwidth) {
	switch( xflag ) {
	case H_CENTER:
	  X = (Boxwidth - maxwidth)/2;
	  break;
	case H_RIGHT:
	  X = (Boxwidth - maxwidth);
	  break;
	}}
      if (totheight > Boxheight) {
	switch( yflag ) {
	case V_CENTER:
	  Y = (Boxheight - totheight)/2;
	  break;
	case V_BOTTOM:
	  Y = (Boxheight - totheight);
	  break;
	}}
      break;
    case NO_SHOW:
      return;
    default:
      ltxErrAlert("Invalid Too_Small Flag for Text!", " ");
      ltxErrFlag = 0;
       
      return;
    }
  }

  /* Now, depending on given x,y flags, set x,y of first baseline to 
     draw text. */

  switch( xflag ) {
  case H_LEFT:
    Xbase = Box.x + X;
    break;
  case H_CENTER:
    Xbase = Box.x + X + (Boxwidth - Linewidth[0])/2;
    break;
  case H_RIGHT:
    Xbase = Box.x + X + (Boxwidth - Linewidth[0]);
    break;
  default:
    ltxErrAlert("Invalid Horizontal Flag For Text!", " ");
    ltxErrFlag = 0;
     
    return;
  }

  switch( yflag ) {
  case V_CENTER:
    Ybase = Box.y + Y + (Boxheight - totheight)/2 + Lineascent[0];
    break;
  case V_TOP:
    Ybase = Box.y + Y + Lineascent[0];
    break;
  case V_BOTTOM:
    Ybase = Box.y + (Y + Boxheight) - (totheight - Lineascent[0]);
    break;
  default:
    ltxErrAlert("Invalid Vertical Flag For Text!", " ");
    ltxErrFlag = 0;
     
    return;
  }

  /* User wants a border */

  if (border) {
  if ( LTxText_gc->values.line_width != border )
    XSetLineAttributes(LTx_dpy, LTxText_gc, border, LineSolid,
		       CapButt, JoinMiter);
  XDrawRectangle(LTx_dpy, Win, LTxText_gc, Box.x, Box.y, Box.width,
		 Box.height);}

  /* Reset Current to zero to start drawing text.
     Depending on which mode, draw text accordingly. */

  Current = Count = 0;
  switch ( mode ) {
    case VERBATIM:
      (void) text_out( font, size, text, (long) 0, (char) 0);
      break;
    case FILL:
    case JUSTIFY:
      Currentlength = Linenumchar[Current];
      (void) fill_text_out( font, size, text, (long) 0, (char) 0);
      break;
    default:
      ltxErrAlert("Invalid Mode For Text!", " ");
      ltxErrFlag = 0;
       
      return;
    }

  /* Once done, free temporary storage for line arrays */
  free((char *)Linewidth);
  free((char *)Lineheight);
  free((char *)Lineascent);
  free((char *)Topwidth);
  free((char *)Botwidth);
  free((char *)Topdescent);
  free((char *)Botascent);
  if (mode & (FILL | JUSTIFY)) {
    free((char *)Linenumchar);
  }

   
}



/**********************************************************************
 * Public Routine: LTxSelectSize(win, win_width, win_height, info)
 * Requires: 
 *  win  = window for text output
 *  win_height = height of window
 *  win_width = width of window
 *  info = a pointer LTxInfo structure that holds information for text
 * Modifies:
 *  The size field in the LTxInfo structure. Procedure ignores what 
 *  the orginal size field is. 
 * Description: 
 *  This routine will find the best size that will fit the text into
 *  the given bounding box using the given font.  
 **********************************************************************/

void LTxSelectSize(win, win_width, win_height, info)
     Window win;
     unsigned win_height, win_width;
     LTxInfo *info;
{
  unsigned font, size;	/* variables to hold fields of info */
  int mode;
  char *text;
  float per_width, per_height;
  int maxwidth;          /* max of all widths */
  int totheight;         /* total height of text */ 
  int numchar = 0;     /* num of char per word - for fill & justify env */
  int wtotal, dtotal, atotal;  /* vars to send to find width & height */ 
  int tmp;

  /* LErr error trapping */

#ifdef LTXO_DEBUG
  printf("Passed to LTxSelectSize() ... \n");
  printf(" x: %d\n y: %d\n height: %d\n width: %d\n border: %d\n",
	 LTxGetX(info), LTxGetY(info), LTxGetHeight(info),
	 LTxGetWidth(info), LTxGetBorder(info));
  printf(" font: %d\n size: %d\n xflag: %d\n yflag: %d\n",
	 LTxGetFont(info), LTxGetSize(info), LTxGetXflag(info),
	 LTxGetYflag(info), LTxGetX(info));
  printf(" mode: %d\n resize: %d\n too_small: %d\n str: %s\n short:%s\n", 
	 LTxGetMode(info), LTxGetResize(info), LTxGetSmall(info),
	 LTxGetText(info), LTxGetShort(info)); 
  printf(" win_height: %u\n win_width: %u\n", win_height, win_width);
#endif

  /* Set automatic variables for easy reference. */
  font = LTxGetFont(info);
  text = LTxGetText(info);
  mode = LTxGetMode(info);
  size = BIGGEST + 1;	     /* to start off loop to find best size */
  per_width = LTxGetWidth(info);
  per_height = LTxGetHeight(info);

  if( per_width == 0.0 || per_height == 0.0) {
    ltxErrAlert("Bounding Box has either zero width and/or height!"," ");
    ltxErrFlag = 0;
     
    return;
  }

  /* Check if the font and size ids are within range */
  if(font >= MAXFONT) {
    ltxErrAlert("Invalid Font Id!", " ");
    ltxErrFlag = 0;
     
    return;
  }

  /* Allocate space for temporary variables holding line by line info */
  if((Linewidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Linewidth)");
  if((Lineheight = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Lineheight)");
  if((Lineascent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Lineascent)");
  if((Topwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Topwidth)");
  if((Botwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Botwidth)");
  if((Topdescent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Topdescent)");
  if((Botascent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Botascent)");

  if ( mode & (FILL | JUSTIFY)) {
    if((Linenumchar = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
       == (int *)0)
      XtError("Out of Memory! (LTxo*Linenumchar)");
  }

  Boxwidth = (unsigned) (per_width * (float)win_width); 
  Boxheight = (unsigned) (per_height * (float)win_height); 

  maxwidth = Boxwidth + 1;    /* to start off the following loop */
  totheight = Boxheight + 1;

  /* Call width and height computing routines to determine the width 
     and height of the complex text. 
     The width of each line is held in the 'Linewidth' array. 
     The height of each line is held in the 'Lineheight' array.  
     The ascent of each line is held in the 'Lineascent' array which 
     is needed to determine the baseline of the text to be drawn.
     Fill env: 'Linenumchar' holds the number of chars per line.
     'Current' keeps track of which line we are on . */

  while ((maxwidth > Boxwidth || totheight > Boxheight) 
	 && (size != SMALLEST)) { 
    /* clear variables from earlier computations */
    for (tmp = 0; tmp <= MaxCurrent; tmp++)	
      Linewidth[tmp] = Lineheight[tmp] = Lineascent[tmp] = 0;
    for (tmp = 0; tmp <= MaxCount; tmp++) {
      Topwidth[tmp] = Botwidth[tmp] = Topdescent[tmp] =
	Botascent[tmp] = 0;
    }
    size--;
    Current = Count = wtotal = 0;

    atotal = get_font_struct(font, size)->ascent;
    dtotal = get_font_struct(font, size)->descent;
    switch ( mode ) {
      case VERBATIM:
	(void) text_width( font, size, &wtotal, text, (char)0 );  
	if (ltxErrFlag) {
	  ltxErrFlag = 0;
	   
	  return;
	}
	Current = Count = 0;
	(void) text_height( font, size, &atotal, &dtotal, text, (char)0 );  
	break;
      case FILL:
      case JUSTIFY:
	/* first clear variables from earlier computations */
	for (tmp = 0; tmp <= MaxCurrent; tmp++)
	  Linenumchar[tmp] = 0;
	numchar = 0;
        (void) fill_text_width( font, size, &wtotal, &numchar, text,
			       (char)0 ); 
	if (ltxErrFlag) {
	  ltxErrFlag = 0;
	   
	  return;
	}
	Current = Count = 0;
        Currentlength = Linenumchar[Current];
	(void) fill_text_height( font, size, &atotal, &dtotal, text,
				(char)0 );
	break;
      default:
	ltxErrAlert("Invalid Mode For Text!", " ");
	ltxErrFlag = 0;
	 
	return;
      }

    maxwidth = totheight = 0;
    for (tmp = 0; tmp <= MaxCurrent; tmp++)  {
      maxwidth = max(maxwidth, Linewidth[tmp]);
      totheight += Lineheight[tmp];
    }

#ifdef LTXO_DEBUG
    printf("Got back from width routine ... maxwidth: %d\n", maxwidth);
    printf("Got back from height routine ... totheight: %d\n", totheight);
    for (tmp = 0; tmp <= MaxCurrent; tmp++) {
      printf("Current: %d\n", tmp);
      printf(" Linewidth: %d\n Lineheight: %d\n Lineascent: %d\n",
	     Linewidth[tmp], Lineheight[tmp], Lineascent[tmp]);
      if (mode & (FILL | JUSTIFY))
	printf(" Linenumchar: %d\n", Linenumchar[tmp]);
    }
#endif
  }

  /* Once done, free temporary storage for line arrays */
  free((char *)Linewidth);
  free((char *)Lineheight);
  free((char *)Lineascent);
  free((char *)Topwidth);
  free((char *)Botwidth);
  free((char *)Topdescent);
  free((char *)Botascent);
  if (mode & (FILL | JUSTIFY)) {
    free((char *)Linenumchar);
  }

  LTxSetSize(info, size);	/* set the best fitting size found */
   
}



/**********************************************************************
 * Public Routine: LTxSetBoxWidth(win, win_width, win_height, info)
 * Requires: 
 *  win  = window for text output
 *  win_height = height of window
 *  win_width = width of window
 *  info = a pointer LTxInfo structure that holds information for text
 * Modifies:
 *  The width field in the LTxInfo structure.
 * Description:
 *  This routine returns to the user the width required to hold text
 *  in LTxInfo structure.  This will only make sense with either
 *  labels or with verbatim text.  The width set will be in normalized
 *  coordinates .
 **********************************************************************/

void LTxSetBoxWidth(win, win_width, win_height, info)
     Window win;
     unsigned win_height, win_width;
     LTxInfo *info;
{
  unsigned font, size;     /* variables to hold fields of info */
  int mode;
  char *text;
  int tmp, maxwidth;    /* maxwidth holds the max of all widths */
  int wtotal;		    /* send to text_width routine */

  /* LErr error trapping */

#ifdef LTXO_DEBUG
  printf("Passed to LTxSetBoxWidth() ... \n");
  printf(" x: %d\n y: %d\n height: %d\n width: %d\n border: %d\n",
	 LTxGetX(info), LTxGetY(info), LTxGetHeight(info),
	 LTxGetWidth(info), LTxGetBorder(info));
  printf(" font: %d\n size: %d\n xflag: %d\n yflag: %d\n",
	 LTxGetFont(info), LTxGetSize(info), LTxGetXflag(info),
	 LTxGetYflag(info), LTxGetX(info));
  printf(" mode: %d\n resize: %d\n too_small: %d\n str: %s\n short: %s\n",
	 LTxGetMode(info), LTxGetResize(info), LTxGetSmall(info),
	 LTxGetText(info), LTxGetShort(info)); 
  printf(" win_height: %u\n win_width: %u\n", win_height, win_width);
#endif

  /* Set automatic variables for easy reference. */
  font = LTxGetFont(info);
  size = LTxGetSize(info);
  text = LTxGetText(info);
  mode = LTxGetMode(info);

  /* Check if the font and size ids are within range */
  if (mode != VERBATIM) {
    ltxErrAlert("Trying to find width when not in VERBATIM!"," ");
    ltxErrFlag = 0;
     
    return;
  }
  if(font >= MAXFONT) {
    ltxErrAlert("Invalid Font Id!", " ");
    ltxErrFlag = 0;
     
    return;
  }
  if(size >= MAXSIZE) {
    ltxErrAlert("Invalid Font Size!", " ");
    ltxErrFlag = 0;
     
    return;
  }

  /* Allocate space for temporary variables holding line by line info */
  if((Linewidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Linewidth)");
  if((Topwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Topwidth)");
  if((Botwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
      == (int *)0)
     XtError("Out of Memory! (LTxo*Botwidth)");


  /* The width of each line is held in the 'Linewidth' array. 
     'Current' keeps track of which line we are on. */

  Current = Count = wtotal = 0;
  (void) text_width( font, size, &wtotal, text, (char)0 );  
  if (ltxErrFlag) {
    ltxErrFlag = 0;
     
    return;
  }

  maxwidth = 0;
  for (tmp = 0; tmp <= MaxCurrent; tmp++) 
    maxwidth = max(maxwidth, Linewidth[tmp]);

  /* Once done, free temporary storage for line arrays */
  free((char *)Linewidth);
  free((char *)Topwidth);
  free((char *)Botwidth);

  LTxSetWidth(info, (float)maxwidth/(float)win_width); 
   
}



/**********************************************************************
 * Public Routine: LTxSetBoxHeight(win, win_width, win_height, info)
 * Requires: 
 *  win  = window for text output
 *  win_height = height of window
 *  win_width = width of window
 *  info = a pointer LTxInfo structure that holds information for text
 * Modifies:
 *  The height field in the LTxInfo structure.
 * Description:
 *  This routine returns to the user the height required to hold text
 *  in LTxInfo structure.  This will only make sense with either
 *  labels or with verbatim text. If the environment is filled, then 
 *  the width field in the LTxInfo structure must be set in order for
 *  the routine to find the best fitting height for the text.  The
 *  height set will be in normalized coordinates .
 **********************************************************************/

void LTxSetBoxHeight(win, win_width, win_height, info)
     Window win;
     unsigned win_height, win_width;
     LTxInfo *info;
{
  unsigned font, size;	     /* variables to hold fields of info */
  int mode;
  char *text;
  int tmp;
  int numchar = 0;		/* for fill_text_width */
  int totheight;		/* total height of text */
  int wtotal, atotal, dtotal;	/* for width & height finding routines*/


#ifdef LTXO_DEBUG
  printf("Passed to LTxSetBoxHeight() ... \n");
  printf(" x: %d\n y: %d\n height: %d\n width: %d\n border: %d\n",
	 LTxGetX(info), LTxGetY(info), LTxGetHeight(info),
	 LTxGetWidth(info), LTxGetBorder(info));
  printf(" font: %d\n size: %d\n xflag: %d\n yflag: %d\n",
	 LTxGetFont(info), LTxGetSize(info), LTxGetXflag(info),
	 LTxGetYflag(info), LTxGetX(info));
  printf(" mode: %d\n resize: %d\n too_small: %d\n str: %s\n short: %s\n",
	 LTxGetMode(info), LTxGetResize(info), LTxGetSmall(info),
	 LTxGetText(info), LTxGetShort(info)); 
  printf(" win_height: %u\n win_width: %u\n", win_height, win_width);
#endif

  /* Set automatic variables for easy reference. */
  font = LTxGetFont(info);
  size = LTxGetSize(info);
  text = LTxGetText(info);
  mode = LTxGetMode(info);

  /* Check if the font and size ids are within range */
  if(font >= MAXFONT) {
    ltxErrAlert("Invalid Font Id!", " ");
    ltxErrFlag = 0;
     
    return;
  }
  if(size >= MAXSIZE) {
    ltxErrAlert("Invalid Font Size!", " ");
    ltxErrFlag = 0;
     
    return;
  }
  
  /* Set static variables for all other routines in file to use. */
  Boxwidth = (unsigned)(LTxGetWidth(info) * (float)win_width); 
  if (Boxwidth == 0 && mode != VERBATIM) {
    ltxErrAlert("No width to find height when not in VERBATIM!", " ");
    ltxErrFlag = 0;
     
    return;
  }

  /* Allocate space for temporary variables holding line by line info */
  if((Lineheight = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
     == (int *)0)
    XtError("Out of Memory! (LTxo*Lineheight)");
  if((Lineascent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
     == (int *)0)
    XtError("Out of Memory! (LTxo*Lineascent)");
  if((Topdescent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
     == (int *)0)
    XtError("Out of Memory! (LTxo*Topdescent)");
  if((Botascent = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
     == (int *)0)
    XtError("Out of Memory! (LTxo*Botascent)");
  if ( mode & (FILL | JUSTIFY)) {
    if((Linewidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
       == (int *)0)
      XtError("Out of Memory! (LTxo*Linewidth)");
    if((Topwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
       == (int *)0)
      XtError("Out of Memory! (LTxo*Topwidth)");
    if((Botwidth = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
       == (int *)0)
      XtError("Out of Memory! (LTxo*Botwidth)");
    if((Linenumchar = (int *)calloc((unsigned)MAX_LINES, sizeof(int)))
       == (int *)0)
      XtError("Out of Memory! (LTxo*Linenumchar)");
  }

  /* The height of each line is held in the 'Lineheight' array. 
     The ascent of each line is held in the 'Lineascent' array. 
     'Current' keeps track of which line we are on ... set it to zero.
       Note: The maximum of Current is MAX_LINES. */

  Current = wtotal = 0;

  atotal = get_font_struct(font, size)->ascent;
  dtotal = get_font_struct(font, size)->descent;
  switch ( mode ) {
    case VERBATIM:
      (void) text_height( font, size, &atotal, &dtotal, text, (char)0 );  
      if (ltxErrFlag) {
	ltxErrFlag = 0;
	 
	return;
      }
      break;
    case FILL:
    case JUSTIFY:
      (void) fill_text_width( font, size, &wtotal, &numchar, text,
			     (char)0 ); 
      if (ltxErrFlag) {
	ltxErrFlag = 0;
	 
	return;
      }
      Current = 0;
      Currentlength = Linenumchar[Current];
      (void) fill_text_height( font, size, &atotal, &dtotal, text,
			      (char)0 );
      break;
    default:
      ltxErrAlert("Invalid Mode For Text!", " ");
      ltxErrFlag = 0;
       
      return;
    }

  totheight = 0;
  for (tmp = 0; tmp <= MaxCurrent; tmp++)
    totheight += Lineheight[tmp];

  LTxSetHeight(info, (float) totheight/ (float)win_height); 

  /* Once done, free temporary storage for line arrays */
  free((char *)Lineheight);
  free((char *)Lineascent);
  free((char *)Topdescent);
  free((char *)Botascent);
  if (mode & (FILL | JUSTIFY)) {
    free((char *)Linewidth);
    free((char *)Topwidth);
    free((char *)Botwidth);
    free((char *)Linenumchar);
  }

  
}



/**********************************************************************
 * Public Routine: LTxFreeFont()
 * Description: 
 *  This procedure frees the text fonts and GCs which were loaded and
 *  created in LTxLoadFont(). Should be call only at the conclusion
 *  of the process.
 **********************************************************************/

void LTxFreeFont()
{
  unsigned fontid, sizeid;

  for (fontid = 0; fontid < MAXFONT; fontid++) {
    for (sizeid = 0; sizeid < MAXSIZE; sizeid++) {
      if (font_tbl[fontid][sizeid].fstruct != NULL ) {
	XFreeFont(LTx_dpy, font_tbl[fontid][sizeid].fstruct);
      }
    }
  }

  XFreeGC(LTx_dpy, LTxText_gc);

   
}


/**********************************************************************
 * The next set of procedures sets information in the LTxInfo 
 * structure. ** Should change to macros.
 **********************************************************************/

void LTxSetX(info, value)
     LTxInfo *info;
     float value;
{ 
  info->box.x = value;
}

void LTxSetY(info, value)
     LTxInfo *info;
     float value;
{ 
  info->box.y = value;
}

void LTxSetHeight(info, value)
     LTxInfo *info;
     float value;
{ 
  info->box.height = value;
}

void LTxSetWidth(info, value)
     LTxInfo *info;
     float value;
{ 
  info->box.width = value;
}

void LTxSetBorder(info, value)
     LTxInfo *info;
     unsigned value;
{ 
  info->box.border = value;
}

void LTxSetText(info, value)
     LTxInfo *info;
     char *value;
{ 
  if ((info->str = (char *)calloc(strlen(value)+1, sizeof(char))) ==
      (char *)0) 
    printf("Out of Memory! (LTxSetText)");

  info->str = value;
}

void LTxSetFont(info, value)
     LTxInfo *info;
     unsigned value;
{ 
  info->font = value;
}

void LTxSetSize(info, value)
     LTxInfo *info;
     unsigned value;
{ 
  info->size = value;
}

void LTxSetXflag(info, value)
     LTxInfo *info;
     int value;
{ 
  info->xflag = value;
}

void LTxSetYflag(info, value)
     LTxInfo *info;
     int value;
{ 
  info->yflag = value;
}

void LTxSetMode(info, value)
     LTxInfo *info;
     int value;
{ 
  info->mode = value;
}

void LTxSetResize(info, value)
     LTxInfo *info;
     Bool value;
{ 
  info->resize = value;
}

void LTxSetSmall(info, value)
     LTxInfo *info;
     int value;
{ 
  info->too_small = value;
}

void LTxSetShort(info, value)
     LTxInfo *info;
     char *value;
{ 
  if ((info->short_str = (char *)calloc(strlen(value)+1, sizeof(char))) ==
      (char *)0) 
    printf("Out of Memory! (LTxSetText)");

  info->short_str = value;
}

void LTxSetBackgd(info, value)
     LTxInfo *info;
     unsigned long value;
{ 
  info->backgd = value;
}

void LTxSetForegd(info, value)
     LTxInfo *info;
     unsigned long value;
{ 
  info->foregd = value;
}

void LTxWantStipple(info, value)
     LTxInfo *info;
     Bool value;
{ 
  info->want_stipple = value;
}

void LTxSetChangeTextFlag(info, value)
     LTxInfo *info;
     Bool value;
{ 
  info->change_text_flag = value;
}

void LTxSetXoffset(info, value)
     LTxInfo *info;
     int value;
{
  info->box.x_offset = value;
}

void LTxSetYoffset(info, value)
     LTxInfo *info;
     int value;
{
  info->box.y_offset = value;
}




/***********************************************************************
 * Public Routine: LTxGetFontIdSizeId(fontfamilyname, font_id_ret,
 *                                    size_id_ret) 
 * Requires:
 *  fontfamilyname = a LAS font family name.
 *  font_id_ret = pointer to variable holding font_id.
 *  size_id_ret = pointer to variable holding size_id.
 * Returns:
 *  0  if the fontfamilyname equals to one of the font names in LTxo
 *     and font_id and size_id are set accordingly.
 *  1  if the fontfamilyname does not equal to one of the font names in
 *     LTxo and the font_id and size_id are set to the defaults of
 *     LTxo. 
 * Description:
 *  This routine converts a font family name to the internal font and
 *  size ids for LTxo to manipulate the fonts.
 **********************************************************************/

LTxGetFontIdSizeId(fontfamilyname, font_id_ret, size_id_ret)
     char *fontfamilyname;
     unsigned *font_id_ret, *size_id_ret;
{
  int i;

  for (i = 0; i <= MAXFAMILY; i++) {
    if ( strcmp(fontfamilyname, family_tbl[i].las_name) == 0) {
      *font_id_ret = family_tbl[i].font_id;
      *size_id_ret = family_tbl[i].size_id;
      
      return(0);
    }
  }
  
  /* At this point, "fontfamilyname" does not equal to any of the 
     familynames that LTxo has, so call LErrNotify and return the
     LTxo default font instead. */
  *font_id_ret = HELVET;
  *size_id_ret = MEDIUM;
   
  return(1);
}
