/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2008 OMC Denmark ApS.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#pragma warning(disable : 6211)

#include "IMAPIStatusS_impl.h"
#include "obj_utils.h"
#include "conv_utils.h"
#include "progress_utils.h"

static inline ULONG native_flags(const ::BRUTUS::BDEFINE Flags)
{
	::BRUTUS::BDEFINE flags = Flags;
	ULONG retval = 0;

	if (!Flags)
		return 0;

	if (flags & ::BRUTUS::BRUTUS_MAPI_DIALOG ) {
		retval |= MAPI_DIALOG;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_DIALOG);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_DECLINE_OK ) {
		retval |= MAPI_DECLINE_OK;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_DECLINE_OK);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_MOVE) {
		retval |= MAPI_MOVE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_MOVE);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_NOREPLACE) {
		retval |= MAPI_NOREPLACE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_NOREPLACE);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_MODIFY) {
		retval |= MAPI_MODIFY;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_MODIFY);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_CREATE) {
		retval |= MAPI_CREATE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_CREATE);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_UNICODE) {
		retval |= MAPI_UNICODE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_UNICODE);
	}
	if (flags & ::BRUTUS::BRUTUS_FORCE_SAVE) {
		retval |= FORCE_SAVE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FORCE_SAVE);
	}
	if (flags & ::BRUTUS::BRUTUS_KEEP_OPEN_READONLY) {
		retval |= KEEP_OPEN_READONLY;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_KEEP_OPEN_READONLY);
	}
	if (flags & ::BRUTUS::BRUTUS_KEEP_OPEN_READWRITE) {
		retval |= KEEP_OPEN_READWRITE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_KEEP_OPEN_READWRITE);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_DEFERRED_ERRORS) {
		retval |= MAPI_DEFERRED_ERRORS;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_DEFERRED_ERRORS);
	}
	if (flags & ::BRUTUS::BRUTUS_ABORT_XP_HEADER_OPERATION) {
		retval |= ABORT_XP_HEADER_OPERATION;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_ABORT_XP_HEADER_OPERATION);
	}
	if (flags & ::BRUTUS::BRUTUS_CONFIG_CHANGED) {
		retval |= CONFIG_CHANGED;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_CONFIG_CHANGED);
	}
	if (flags & ::BRUTUS::BRUTUS_FORCE_XP_CONNECT) {
		retval |= FORCE_XP_CONNECT;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FORCE_XP_CONNECT);
	}
	if (flags & ::BRUTUS::BRUTUS_FORCE_XP_DISCONNECT) {
		retval |= FORCE_XP_DISCONNECT;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FORCE_XP_DISCONNECT);
	}
	if (flags & ::BRUTUS::BRUTUS_PROCESS_XP_HEADER_CACHE) {
		retval |= PROCESS_XP_HEADER_CACHE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_PROCESS_XP_HEADER_CACHE);
	}
	if (flags & ::BRUTUS::BRUTUS_REFRESH_XP_HEADER_CACHE) {
		retval |= REFRESH_XP_HEADER_CACHE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_REFRESH_XP_HEADER_CACHE);
	}
	if (flags & ::BRUTUS::BRUTUS_SUPPRESS_UI) {
		retval |= SUPPRESS_UI;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_SUPPRESS_UI);
	}
	if (flags & ::BRUTUS::BRUTUS_SHOW_XP_SESSION_UI) {
		retval |= SHOW_XP_SESSION_UI;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_SHOW_XP_SESSION_UI);
	}
	if (flags & ::BRUTUS::BRUTUS_FLUSH_ASYNC_OK) {
		retval |= FLUSH_ASYNC_OK;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FLUSH_ASYNC_OK);
	}
	if (flags & ::BRUTUS::BRUTUS_FLUSH_DOWNLOAD) {
		retval |= FLUSH_DOWNLOAD;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FLUSH_DOWNLOAD);
	}
	if (flags & ::BRUTUS::BRUTUS_FLUSH_FORCE) {
		retval |= FLUSH_FORCE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FLUSH_FORCE);
	}
	if (flags & ::BRUTUS::BRUTUS_FLUSH_NO_UI) {
		retval |= FLUSH_NO_UI;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FLUSH_NO_UI);
	}
	if (flags & ::BRUTUS::BRUTUS_FLUSH_UPLOAD) {
		retval |= FLUSH_UPLOAD;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_FLUSH_UPLOAD);
	}


	if (flags) {
		char msg[128] = {0};
		sprintf_s(msg, sizeof(msg), "Unknown flag(s) from BRUTUS : %X", flags);
		BRUTUS_LOG_BUG(msg);
	}

	return retval;
}


BRUTUS_IMAPIStatus_i::BRUTUS_IMAPIStatus_i(LPMAPISTATUS Status,
					   LPMAPISESSION MAPISession,
					   ::PortableServer::POA_ptr Poa)
	: mapi_status_(Status),
	  mapi_session_(MAPISession),
	  poa_(::PortableServer::POA::_duplicate(Poa))
{
	mapi_session_->AddRef();
}

::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::ValidateState(::BRUTUS::BDEFINE ulFlags)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::ValidateState()");

	unsigned long flags = native_flags(ulFlags);
	flags |= SUPPRESS_UI;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->ValidateState(0, flags);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::ValidateState()");
	return br;
}

::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::ChangePassword(const char * lpOldPass,
						       const char * lpNewPass,
						       ::BRUTUS::BDEFINE ulFlags)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::ChangePassword()");

	unsigned long flags = native_flags(ulFlags);

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->ChangePassword((char*)lpOldPass,
						  (char*)lpNewPass,
						  flags);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::ChangePassword()");
	return br;
}

::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::FlushQueues(const ::BRUTUS::ENTRYID & lpTargetTransport,
						    ::BRUTUS::BDEFINE ulFlags)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::FlushQueues()");

	ENTRYID *entry_id = NULL;
	unsigned long entry_id_size;
	if (!entryid_brutus_to_mapi(&lpTargetTransport, 0, entry_id_size, entry_id)) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}

	unsigned long flags = native_flags(ulFlags);
	flags |= FLUSH_NO_UI;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->FlushQueues(0,
					       entry_id_size,
					       entry_id,
					       flags);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::FlushQueues()");
	return br;
}

::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::GetLastError(::BRUTUS::BRESULT ReturnCode,
						     ::BRUTUS::BDEFINE ulFlags,
						     ::BRUTUS::MAPIERROR_out lppMAPIError)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::GetLastError()");

	::BRUTUS::MAPIERROR_var error;
	try {
		error = new ::BRUTUS::MAPIERROR;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	error->ulVersion = (::CORBA::ULong)0;
	error->ulLowLevelError = (::CORBA::ULong)0;
	error->ulContext = (::CORBA::ULong)0;

	HRESULT hr;
	if (!bresult_to_hresult(ReturnCode, hr)) {
		lppMAPIError = error._retn();
		BRUTUS_LOG_BUG("Could not convert BRESULT into HRESULT");
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	LPMAPIERROR mapi_error = NULL;
	unsigned long flags = native_flags(ulFlags);

	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetLastError(hr, flags, &mapi_error);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		MAPIFreeBuffer(mapi_error);
		lppMAPIError = error._retn();
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	if (!mapi_error)
		BRUTUS_LOG_INF("No applicable error information from MAPI");

	mapierror_mapi_to_brutus(mapi_error, error, true);

	lppMAPIError = error._retn();

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::GetLastError()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::SaveChanges(::BRUTUS::BDEFINE ulFlags)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::SaveChanges()");

	unsigned long flags = native_flags(ulFlags);

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->SaveChanges(flags);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::SaveChanges()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::GetProps(const ::BRUTUS::SPropTagArray& lpPropTagArray,
						 ::BRUTUS::BDEFINE ulFlags,
						 ::BRUTUS::seq_SPropValue_out lppPropArray)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::GetProps()");

	SPropTagArray *tags = NULL;
	proptag_array_brutus_to_mapi(&lpPropTagArray, 0, tags);
	unsigned long flags = native_flags(ulFlags);
	SPropValue *props = NULL;
	unsigned long count = 0;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetProps(tags, flags, &count, &props);
	}
	MAPIFreeBuffer(tags);

	::BRUTUS::seq_SPropValue_var brutus_props;
	try {
		brutus_props = new ::BRUTUS::seq_SPropValue;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_props->length(0);

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		lppPropArray = brutus_props._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	spropvalue_array_mapi_to_brutus(count, props, brutus_props, mapi_session_, true, poa_);
	lppPropArray = brutus_props._retn();

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::GetProps()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::GetPropList(::BRUTUS::BDEFINE ulFlags,
						    ::BRUTUS::SPropTagArray_out lppPropTagArray)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::GetPropList()");

	SPropTagArray *tags = NULL;
	unsigned long flags = native_flags(ulFlags);

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetPropList(flags, &tags);
	}

	if (!proptag_array_mapi_to_brutus(tags, lppPropTagArray, true)) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::GetPropList()");
	return br;
}

// FIXME. Support for all possible interfaces are misssing
::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::OpenProperty(::CORBA::ULong ulPropTag,
						     const char * lpiid,
						     ::CORBA::ULong ulInterfaceOptions,
						     ::BRUTUS::BDEFINE ulFlags,
						     ::BRUTUS::IUnknown_out lppUnk)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::OpenProperty()");

	lppUnk = ::BRUTUS::IUnknown::_nil();

	unsigned long tag;
	proptag_brutus_to_mapi(ulPropTag, tag);

	GUID id;
	if (!guid_brutus_to_mapi_no_alloc(lpiid, &id)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into MAPI GUID");
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	unsigned long options = (unsigned long)ulInterfaceOptions;
	unsigned long flags = native_flags(ulFlags);

	LPUNKNOWN unknown = NULL;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->OpenProperty(tag, &id, options, flags, &unknown);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	if (S_OK != hr)
		return br;

	bool create_result = false;
	try {
		if (S_OK == hr)
			create_result = create_brutus_object(lpiid, unknown, poa_, lppUnk, mapi_session_);
		else if (unknown) {
			unknown->Release();
			unknown = NULL;
		}
	}
	catch (...) {
		BRUTUS_LOG_BUG("Exception caught");
	}

	if (!create_result) {
		BRUTUS_LOG_BUG("Could not convert MAPI object into BRUTUS object");
		if (unknown)
			unknown->Release();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::OpenProperty()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::SetProps(const ::BRUTUS::seq_SPropValue & lpPropArray,
						 ::CORBA::Boolean ProblemInfo,
						 ::BRUTUS::SPropProblemArray_out lppProblems)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::SetProps()");

	unsigned long count = 0;
	SPropValue *mapi_props = NULL;
	::BRUTUS::SPropProblemArray_var brutus_problems;
	try {
		brutus_problems = new ::BRUTUS::SPropProblemArray;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_problems->length(0);

	if (!spropvalue_array_brutus_to_mapi(&lpPropArray, 0, count, mapi_props)) {
		MAPIFreeBuffer(mapi_props);
		BRUTUS_LOG_ERR("Could not convert Brutus SPropValue to MAPI SPropValue");
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	HRESULT hr;
	LPSPropProblemArray mapi_problems = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->SetProps(count, mapi_props, (ProblemInfo ? &mapi_problems : NULL));
	}
	MAPIFreeBuffer(mapi_props);

	if (!sPropProblem_array_mapi_to_brutus(mapi_problems, brutus_problems.inout(), true)) {
		BRUTUS_LOG_ERR("Could not convert MAPI SPropProblemArray to Brutus SPropProblemArray");
		brutus_problems->length(0);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	lppProblems = brutus_problems._retn();

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::SetProps()");
	return br;
}

::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::DeleteProps(const ::BRUTUS::SPropTagArray & lpPropTagArray,
						    ::CORBA::Boolean ProblemInfo,
						    ::BRUTUS::SPropProblemArray_out lppProblems)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::DeleteProps()");

	::BRUTUS::SPropProblemArray_var brutus_problems;
	try {
		brutus_problems = new ::BRUTUS::SPropProblemArray;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_problems->length(0);

	SPropTagArray *tags = NULL;
	if (!proptag_array_brutus_to_mapi(&lpPropTagArray, 0, tags)) {
		BRUTUS_LOG_ERR("Could not convert Brutus SPropTagArray to MAPI SPropTagArray");
		MAPIFreeBuffer(tags);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	HRESULT hr;
	LPSPropProblemArray mapi_problems = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->DeleteProps(tags, (ProblemInfo ? &mapi_problems : NULL));
	}
	MAPIFreeBuffer(tags);

	if (!sPropProblem_array_mapi_to_brutus(mapi_problems, brutus_problems.inout(), true)) {
		BRUTUS_LOG_ERR("Could not convert MAPI SPropProblemArray to Brutus SPropProblemArray");
		brutus_problems->length(0);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	lppProblems = brutus_problems._retn();

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::DeleteProps()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::CopyTo(const ::BRUTUS::seq_GUID & rgiidExclude,
					       const ::BRUTUS::SPropTagArray & lpExcludeProps,
					       ::BRUTUS::IMAPIProgress_ptr lpProgress,
					       const char * lpInterface,
					       const ::BRUTUS::ENTRYID & lpDestObj,
					       ::BRUTUS::BDEFINE ulFlags,
					       ::CORBA::Boolean ProblemInfo,
					       ::BRUTUS::SPropProblemArray_out lppProblems)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::CopyTo()");

	::BRUTUS::SPropProblemArray_var brutus_problems;
	try {
		brutus_problems = new ::BRUTUS::SPropProblemArray;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_problems->length(0);

	GUID *guids = NULL;
	unsigned long count = rgiidExclude.length();
	if (count) {
		if (!guid_array_brutus_to_mapi(&rgiidExclude, 0, guids)) {
			BRUTUS_LOG_ERR("Could not convert Brutus GUID to MAPI GUID");
			lppProblems = brutus_problems._retn();
			return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
	}

	SPropTagArray *exc_tags = NULL;
	if (lpExcludeProps.length()) {
		if (!proptag_array_brutus_to_mapi(&lpExcludeProps, 0, exc_tags)) {
			BRUTUS_LOG_ERR("Could not convert Brutus SPropTagArray to MAPI SPropTagArray");
			MAPIFreeBuffer(guids);
			lppProblems = brutus_problems._retn();
			return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
	}
	ULONG flags = native_flags(ulFlags);

	CMAPIProgress *progress = NULL;
	if ((flags & MAPI_DIALOG) && !::CORBA::is_nil(lpProgress)) {
		try {
			progress = new CMAPIProgress(lpProgress);
		}
		catch (std::bad_alloc &) {
			BRUTUS_LOG_ERR("No memory");
			MAPIFreeBuffer(exc_tags);
			MAPIFreeBuffer(guids);
			lppProblems = brutus_problems._retn();
			return ::BRUTUS::BRUTUS_MAPI_E_NOT_ENOUGH_MEMORY;
		}
	} else
		FLAGS_OFF(ULONG, flags, MAPI_DIALOG);

	GUID i_id;
	if (!guid_brutus_to_mapi_no_alloc(lpInterface, &i_id)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into GUID");
		if (progress)
			progress->Release();
		MAPIFreeBuffer(exc_tags);
		MAPIFreeBuffer(guids);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	ENTRYID *entry_id = NULL;
	unsigned long entry_id_size;
	if (!entryid_brutus_to_mapi(&lpDestObj, 0, entry_id_size, entry_id)) {
		BRUTUS_LOG_ERR("No memory");
		if (progress)
			progress->Release();
		MAPIFreeBuffer(exc_tags);
		MAPIFreeBuffer(guids);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_MAPI_E_NOT_ENOUGH_MEMORY;
	}

	LPUNKNOWN unk_dest_object = NULL;
	HRESULT hr;
	{
		unsigned long obj_type;
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_session_->OpenEntry(entry_id_size,
					      entry_id,
					      &i_id,
					      MAPI_MODIFY,
					      &obj_type,
					      &unk_dest_object);
	}
	::BRUTUS::BRESULT br;
	if (hr != S_OK) {
		if (!hresult_to_bresult(hr, br)) {
			BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
			br = ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
		BRUTUS_LOG_BUG("Error opening object");
		if (progress)
			progress->Release();
		if (unk_dest_object)
			unk_dest_object->Release();

		MAPIFreeBuffer(exc_tags);
		MAPIFreeBuffer(guids);
		lppProblems = brutus_problems._retn();
		return br;
	}

	LPSPropProblemArray mapi_problems = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->CopyTo(count,
					  guids,
					  exc_tags,
					  0, // UI handle
					  progress,
					  &i_id,
					  unk_dest_object,
					  flags,
					  (ProblemInfo ? &mapi_problems : NULL));
	}
	if (progress)
		progress->Release();
	if (unk_dest_object)
		unk_dest_object->Release();
	MAPIFreeBuffer(exc_tags);
	MAPIFreeBuffer(guids);

	if (!sPropProblem_array_mapi_to_brutus(mapi_problems, brutus_problems.inout(), true)) {
		BRUTUS_LOG_ERR("Could not convert MAPI SPropProblemArray to Brutus SPropProblemArray");
		brutus_problems->length(0);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	lppProblems = brutus_problems._retn();

	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::CopyTo()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::CopyProps(const ::BRUTUS::SPropTagArray & lpIncludeProps,
						  ::BRUTUS::IMAPIProgress_ptr lpProgress,
						  const char * lpInterface,
						  const ::BRUTUS::ENTRYID &lpDestObj,
						  ::BRUTUS::BDEFINE ulFlags,
						  ::CORBA::Boolean ProblemInfo,
						  ::BRUTUS::SPropProblemArray_out lppProblems)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::CopyProps()");

	::BRUTUS::SPropProblemArray_var brutus_problems;
	try {
		brutus_problems = new ::BRUTUS::SPropProblemArray;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_problems->length(0);

	GUID i_id;
	if (!guid_brutus_to_mapi_no_alloc(lpInterface, &i_id)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into GUID");
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	SPropTagArray *inc_tags = NULL;
	if (lpIncludeProps.length()) {
		if (!proptag_array_brutus_to_mapi(&lpIncludeProps, 0, inc_tags)) {
			BRUTUS_LOG_ERR("Could not convert Brutus SPropTagArray to MAPI SPropTagArray");
			lppProblems = brutus_problems._retn();
			return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
	}

	ULONG flags = native_flags(ulFlags);

	CMAPIProgress *progress = NULL;
	if ((flags & MAPI_DIALOG) && !::CORBA::is_nil(lpProgress)) {
		try {
			progress = new CMAPIProgress(lpProgress);
		}
		catch (...) {
			BRUTUS_LOG_ERR("No memory");
			MAPIFreeBuffer(inc_tags);
			lppProblems = brutus_problems._retn();
			return ::BRUTUS::BRUTUS_MAPI_E_NOT_ENOUGH_MEMORY;
		}
	} else
		FLAGS_OFF(ULONG, flags, MAPI_DIALOG);

	ENTRYID *entry_id = NULL;
	unsigned long entry_id_size;
	if (!entryid_brutus_to_mapi(&lpDestObj, 0, entry_id_size, entry_id)) {
		BRUTUS_LOG_ERR("No memory");
		MAPIFreeBuffer(inc_tags);
		if (progress)
			progress->Release();
		throw ::CORBA::NO_MEMORY();
	}

	LPUNKNOWN unk_dest_object = NULL;
	HRESULT hr;
	{
		unsigned long obj_type;
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_session_->OpenEntry(entry_id_size,
					      entry_id,
					      &i_id,
					      MAPI_MODIFY,
					      &obj_type,
					      &unk_dest_object);
	}
	::BRUTUS::BRESULT br;
	if (hr != S_OK) {
		if (!hresult_to_bresult(hr, br)) {
			BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
			br = ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
		BRUTUS_LOG_BUG("Error opening object");
		if (progress)
			progress->Release();
		if (unk_dest_object)
			unk_dest_object->Release();

		MAPIFreeBuffer(inc_tags);
		lppProblems = brutus_problems._retn();

		return br;
	}

	LPSPropProblemArray mapi_problems = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->CopyProps(inc_tags,
					     0, // UI handle
					     progress,
					     &i_id,
					     unk_dest_object,
					     flags,
					     (ProblemInfo ? &mapi_problems : NULL));
	}
	MAPIFreeBuffer(inc_tags);
	if (progress)
		progress->Release();
	if (unk_dest_object)
		unk_dest_object->Release();

	if (!sPropProblem_array_mapi_to_brutus(mapi_problems, brutus_problems.inout(), true)) {
		BRUTUS_LOG_ERR("Could not convert MAPI SPropProblemArray to Brutus SPropProblemArray");
		brutus_problems->length(0);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	lppProblems = brutus_problems._retn();

	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::CopyProps()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::GetNamesFromIDs(::BRUTUS::SPropTagArray & lppPropTags,
							const char * lpPropSetGuid,
							::BRUTUS::BDEFINE ulFlags,
							::BRUTUS::seq_MAPINAMEID_out lpppPropNames)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::GetNamesFromIDs()");

	::BRUTUS::seq_MAPINAMEID_var brutus_names;
	try {
		brutus_names = new ::BRUTUS::seq_MAPINAMEID;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_names->length(0);

	SPropTagArray *tags = NULL;
	if (lppPropTags.length())
		proptag_array_brutus_to_mapi(&lppPropTags, 0, tags);

	GUID *i_id;
	if (!guid_brutus_to_mapi(lpPropSetGuid, NULL, i_id)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into GUID");
		MAPIFreeBuffer(tags);
		lpppPropNames = brutus_names._retn();
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	unsigned long flags = native_flags(ulFlags);

	unsigned long count = 0;
	MAPINAMEID **mapi_names = NULL;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetNamesFromIDs(&tags, i_id, flags, &count, &mapi_names);
	}
	MAPIFreeBuffer(i_id);

	if (!proptag_array_mapi_to_brutus(tags, lppPropTags, true)) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}

	nameid_array_mapi_to_brutus(count, mapi_names, brutus_names, true);
	lpppPropNames = brutus_names._retn();

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::GetNamesFromIDs()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::GetIDsFromNames(const ::BRUTUS::seq_MAPINAMEID & lppPropNames,
							::BRUTUS::BDEFINE ulFlags,
							::BRUTUS::SPropTagArray_out lppPropTags)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::GetIDsFromNames()");

	unsigned long count = lppPropNames.length();
	::LPMAPINAMEID *mapi_names = NULL;
	nameid_array_brutus_to_mapi(&lppPropNames, 0, mapi_names);

	unsigned long flags = native_flags(ulFlags);

	SPropTagArray *tags = NULL;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetIDsFromNames(count,
						   mapi_names,
						   flags,
						   &tags);
	}
	MAPIFreeBuffer(mapi_names);

	::BRUTUS::SPropTagArray_var brutus_tags;
	if (!proptag_array_mapi_to_brutus(tags, brutus_tags.out(), true)) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	lppPropTags = brutus_tags._retn();

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::GetIDsFromNames()");
	return br;
}

::BRUTUS::BRESULT
BRUTUS_IMAPIStatus_i::GetNamedProps(const char *PropSetGuid,
				    const ::BRUTUS::seq_NamedPropertyTag & lpPropTagArray,
				    ::BRUTUS::BDEFINE ulFlags,
				    ::BRUTUS::seq_SPropValue_out lppPropArray)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::GetNamedProps()");

	unsigned long flags = 0;
	SPropValue *props = NULL;
	unsigned long count = 0;

	::BRUTUS::seq_SPropValue_var brutus_props;
	try {
		brutus_props = new ::BRUTUS::seq_SPropValue;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_props->length(0);

	GUID mapi_propset_guid;
	if (!guid_brutus_to_mapi_no_alloc(PropSetGuid, &mapi_propset_guid, false)) {
		lppPropArray = brutus_props._retn();
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	::LPMAPINAMEID *mapi_names = NULL;
	if (S_OK != MAPIAllocateBuffer(lpPropTagArray.length()*sizeof(::MAPINAMEID*),
				       (void**)&mapi_names)) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}

	ULONG n = 0;
	for (n = 0; n < lpPropTagArray.length(); n++) {
		if (S_OK != MAPIAllocateMore(sizeof(::MAPINAMEID),
					     mapi_names,
					     (void**)&(mapi_names[n]))) {
			BRUTUS_LOG_ERR("No memory");
			MAPIFreeBuffer(mapi_names);
			throw ::CORBA::NO_MEMORY();
		}
	}

	for (n = 0; n < lpPropTagArray.length(); n++) {
		mapi_names[n]->lpguid = (LPGUID)&mapi_propset_guid;
		switch (lpPropTagArray[n].name._d()) {
		case ::BRUTUS::BRUTUS_ID_INTEGER :
			mapi_names[n]->ulKind = MNID_ID;
			mapi_names[n]->Kind.lID = lpPropTagArray[n].name.int_name();
			break;
		case ::BRUTUS::BRUTUS_ID_STRING :
			mapi_names[n]->ulKind = MNID_STRING;
			mapi_names[n]->Kind.lpwstrName = ascii_to_unicode((void*)mapi_names, (const char*)lpPropTagArray[n].name.str_name());
			if (!mapi_names[n]->Kind.lpwstrName) {
				BRUTUS_LOG_ERR("No memory");
				MAPIFreeBuffer(mapi_names);
				throw ::CORBA::NO_MEMORY();
			}
			break;
		default :
			BRUTUS_LOG_BUG("Unknown name kind");
			lppPropArray = brutus_props._retn();
			MAPIFreeBuffer(mapi_names);
			return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
		}
	}

	SPropTagArray *tags = NULL;
	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetIDsFromNames(lpPropTagArray.length(),
						   mapi_names,
						   MAPI_CREATE,
						   &tags);
	}
	MAPIFreeBuffer(mapi_names);

	if (!tags || (tags->cValues != lpPropTagArray.length())) {
		BRUTUS_LOG_ERR("Could not get PropTag ID from name");
		BRUTUS_EXIT;
	}
	if ((MAPI_W_ERRORS_RETURNED != hr) && (S_OK != hr))
		BRUTUS_EXIT;

	ULONG prop_id = 0;
	ULONG prop_type = 0;
	for (n = 0; n < tags->cValues; n++) {
		prop_id = PROP_ID(tags->aulPropTag[n]);
		prop_type = proptype_brutus_to_mapi(lpPropTagArray[n].type);
		tags->aulPropTag[n] = PROP_TAG(prop_type, prop_id);
	}

	flags = native_flags(ulFlags);
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetProps(tags, flags, &count, &props);
	}

exit:
	MAPIFreeBuffer(tags);

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		lppPropArray = brutus_props._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	spropvalue_array_mapi_to_brutus(count, props, brutus_props, mapi_session_, true, poa_);
	lppPropArray = brutus_props._retn();

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::GetNamedProps()");
	return br;
}

::BRUTUS::BRESULT
BRUTUS_IMAPIStatus_i::SetNamedProps(const char *PropSetGuid,
				    const ::BRUTUS::seq_NamedPropertyTag & lpPropTagArray,
				    const ::BRUTUS::seq_SPropValue & lpPropArray,
				    ::CORBA::Boolean ProblemInfo,
				    ::BRUTUS::SPropProblemArray_out lppProblems)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::SetNamedProps()");

	unsigned long count = 0;
	SPropValue *mapi_props = NULL;
	::BRUTUS::SPropProblemArray_var brutus_problems;
	try {
		brutus_problems = new ::BRUTUS::SPropProblemArray;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	brutus_problems->length(0);

	if (!spropvalue_array_brutus_to_mapi(&lpPropArray, 0, count, mapi_props)) {
		MAPIFreeBuffer(mapi_props);
		BRUTUS_LOG_ERR("Could not convert Brutus SPropValue to MAPI SPropValue");
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	HRESULT hr;
	LPSPropProblemArray mapi_problems = NULL;

	GUID mapi_propset_guid;
	if (!guid_brutus_to_mapi_no_alloc(PropSetGuid, &mapi_propset_guid, false)) {
		MAPIFreeBuffer(mapi_props);
		MAPIFreeBuffer(mapi_problems);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	::LPMAPINAMEID *mapi_names = NULL;
	if (S_OK != MAPIAllocateBuffer(lpPropArray.length()*sizeof(::MAPINAMEID*),
				       (void**)&mapi_names)) {
		MAPIFreeBuffer(mapi_props);
		MAPIFreeBuffer(mapi_problems);
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}

	ULONG n = 0;
	for (n = 0; n < lpPropArray.length(); n++) {
		if (S_OK != MAPIAllocateMore(sizeof(::MAPINAMEID),
					     mapi_names,
					     (void**)&(mapi_names[n]))) {
			BRUTUS_LOG_ERR("No memory");
			MAPIFreeBuffer(mapi_names);
			throw ::CORBA::NO_MEMORY();
		}
	}

	for (n = 0; n < lpPropTagArray.length(); n++) {
		mapi_names[n]->lpguid = (LPGUID)&mapi_propset_guid;
		switch (lpPropTagArray[n].name._d()) {
		case ::BRUTUS::BRUTUS_ID_INTEGER :
			mapi_names[n]->ulKind = MNID_ID;
			mapi_names[n]->Kind.lID = lpPropTagArray[n].name.int_name();
			break;
		case ::BRUTUS::BRUTUS_ID_STRING :
			mapi_names[n]->ulKind = MNID_STRING;
			mapi_names[n]->Kind.lpwstrName = ascii_to_unicode((void*)mapi_names, (const char*)lpPropTagArray[n].name.str_name());
			if (!mapi_names[n]->Kind.lpwstrName) {
				BRUTUS_LOG_ERR("No memory");
				MAPIFreeBuffer(mapi_names);
				throw ::CORBA::NO_MEMORY();
			}
			break;
		default :
			BRUTUS_LOG_BUG("Unknown name kind");
			MAPIFreeBuffer(mapi_props);
			MAPIFreeBuffer(mapi_problems);
			MAPIFreeBuffer(mapi_names);
			lppProblems = brutus_problems._retn();
			return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
		}
	}

	SPropTagArray *tags = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->GetIDsFromNames(lpPropTagArray.length(),
						   mapi_names,
						   MAPI_CREATE,
						   &tags);
	}
	MAPIFreeBuffer(mapi_names);

	if (!tags || (tags->cValues != lpPropArray.length())) {
		BRUTUS_LOG_ERR("Could not get PropTag ID from name");
		BRUTUS_EXIT;
	}
	if ((MAPI_W_ERRORS_RETURNED != hr) && (S_OK != hr))
		BRUTUS_EXIT;

	ULONG prop_id = 0;
	ULONG prop_type = 0;
	for (n = 0; n < tags->cValues; n++) {
		prop_id = PROP_ID(tags->aulPropTag[n]);
		prop_type = proptype_brutus_to_mapi(lpPropTagArray[n].type);
		mapi_props[n].ulPropTag = PROP_TAG(prop_type, prop_id);
	}

	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->SetProps(lpPropArray.length(), mapi_props, (ProblemInfo ? &mapi_problems : NULL));
	}

exit:
	MAPIFreeBuffer(mapi_props);
	MAPIFreeBuffer(tags);

	if (!sPropProblem_array_mapi_to_brutus(mapi_problems, brutus_problems.inout(), true)) {
		BRUTUS_LOG_ERR("Could not convert MAPI SPropProblemArray to Brutus SPropProblemArray");
		brutus_problems->length(0);
		lppProblems = brutus_problems._retn();
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	lppProblems = brutus_problems._retn();

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::SetNamedProps()");
	return br;
}

::BRUTUS::BRESULT BRUTUS_IMAPIStatus_i::QueryInterface(const char *iid,
						       ::BRUTUS::IUnknown_out ppvObject)
{
	ppvObject = ::BRUTUS::IUnknown::_nil();

	GUID mapi_iid;
	if (!guid_brutus_to_mapi_no_alloc(iid, &mapi_iid)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into MAPI GUID");
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	HRESULT hr;
	void *object = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = mapi_status_->QueryInterface(mapi_iid, &object);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	if (hr == S_OK) {
		if (!create_brutus_object(iid, (LPUNKNOWN)object, poa_, ppvObject, mapi_session_)) {
			BRUTUS_LOG_BUG("Could not create brutus object.");
			((LPUNKNOWN)object)->Release();
			return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
	}

	return br;
}

void BRUTUS_IMAPIStatus_i::Destroy(::CORBA::ULong InstanceID)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IMAPIStatus_i::Destroy()");

	::PortableServer::ObjectId_var oid;
	oid = poa_->servant_to_id(this);

	poa_->deactivate_object(oid);

	BRUTUS_LOG_INF("Leaving BRUTUS_IMAPIStatus_i::Destroy()");
}
