/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2008 OMC Denmark ApS.
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Affero General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Affero General Public License for more details.
 *
 *  You should have received a copy of the GNU Affero General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "IProviderAdminS_impl.h"
#include "obj_utils.h"
#include "conv_utils.h"
#include "IMAPITableS_impl.h"
#include "IProfSectS_impl.h"

#include "templates.i"

static inline ULONG native_flags(const ::BRUTUS::BDEFINE Flags)
{
	::BRUTUS::BDEFINE flags = Flags;
	ULONG retval = 0;

	if (!Flags)
		return 0;

	if (flags & ::BRUTUS::BRUTUS_MAPI_UNICODE) {
		retval |= MAPI_UNICODE;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_UNICODE);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_DIALOG) {
		retval |= MAPI_DIALOG;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_DIALOG);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_DEFERRED_ERRORS) {
		retval |= MAPI_DEFERRED_ERRORS;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_DEFERRED_ERRORS);
	}
	if (flags & ::BRUTUS::BRUTUS_MAPI_MODIFY) {
		retval |= MAPI_MODIFY;
		FLAGS_OFF(::BRUTUS::BDEFINE, flags, ::BRUTUS::BRUTUS_MAPI_MODIFY);
	}

	if (flags) {
		char msg[128] = {0};
		sprintf_s(msg, sizeof(msg), "Unknown flag(s) from BRUTUS : %X", flags);
		BRUTUS_LOG_BUG(msg);
	}

	return retval;
}


BRUTUS_IProviderAdmin_i::BRUTUS_IProviderAdmin_i(LPPROVIDERADMIN ProviderAdmin,
						 LPMAPISESSION MAPISession,
						 ::PortableServer::POA_ptr Poa)
	: provider_admin_(ProviderAdmin),
	  mapi_session_(MAPISession),
	  poa_(::PortableServer::POA::_duplicate(Poa))
{
	mapi_session_->AddRef();
}


::BRUTUS::BRESULT BRUTUS_IProviderAdmin_i::GetLastError(::BRUTUS::BRESULT ReturnCode,
							::CORBA::ULong ulFlags,
							::BRUTUS::MAPIERROR_out lppMAPIError)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IProviderAdmin_i::GetLastError()");

	::BRUTUS::MAPIERROR_var error;
	try {
		error = new ::BRUTUS::MAPIERROR;
	}
	catch (std::bad_alloc &) {
		BRUTUS_LOG_ERR("No memory");
		throw ::CORBA::NO_MEMORY();
	}
	error->ulVersion = (::CORBA::ULong)0;
	error->ulLowLevelError = (::CORBA::ULong)0;
	error->ulContext = (::CORBA::ULong)0;

	HRESULT hr;
	if (!bresult_to_hresult(ReturnCode, hr)) {
		lppMAPIError = error._retn();
		BRUTUS_LOG_BUG("Could not convert BRESULT into HRESULT");
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	LPMAPIERROR mapi_error = NULL;
	unsigned long flags = native_flags(ulFlags);

	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = provider_admin_->GetLastError(hr, flags, &mapi_error);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		MAPIFreeBuffer(mapi_error);
		lppMAPIError = error._retn();
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}
	if (!mapi_error)
		BRUTUS_LOG_INF("No applicable error information from MAPI");

	mapierror_mapi_to_brutus(mapi_error, error, true);

	lppMAPIError = error._retn();

	BRUTUS_LOG_INF("Leaving BRUTUS_IProviderAdmin_i::GetLastError()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IProviderAdmin_i::GetProviderTable(::CORBA::ULong ulFlags,
							    ::BRUTUS::IMAPITable_out lppTable)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IProviderAdmin_i::GetProviderTable()");

	lppTable = ::BRUTUS::IMAPITable::_nil();
	LPMAPITABLE mapi_table = NULL;

	ULONG flags = native_flags(ulFlags);
	FLAGS_OFF(ULONG, flags, MAPI_DIALOG);

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = provider_admin_->GetProviderTable(flags, &mapi_table);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	if (hr == S_OK)
		lppTable = create_object<BRUTUS_IMAPITable_i, ::BRUTUS::IMAPITable, LPMAPITABLE>
			(mapi_table, poa_.in(), mapi_session_);

	BRUTUS_LOG_INF("Leaving BRUTUS_IProviderAdmin_i::GetProviderTable()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IProviderAdmin_i::CreateProvider(const char * lpszProvider,
							  const ::BRUTUS::seq_SPropValue & lpProps,
							  ::CORBA::ULong ulFlags,
							  ::CORBA::String_out lpUID)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IProviderAdmin_i::CreateProvider()");

	::BRUTUS::MAPIUID_var brutus_uid;

	unsigned long count = 0;
	SPropValue *mapi_props = NULL;
	if (!spropvalue_array_brutus_to_mapi(&lpProps, 0, count, mapi_props)) {
		BRUTUS_LOG_ERR("Could not convert Brutus SPropValue to MAPI SPropValue");
		lpUID = brutus_uid._retn();
		MAPIFreeBuffer(mapi_props);
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	ULONG flags = native_flags(ulFlags);
	FLAGS_OFF(ULONG, flags, MAPI_DIALOG);

	MAPIUID u_id;

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = provider_admin_->CreateProvider((char*)lpszProvider,
						     count,
						     mapi_props,
						     0,
						     flags,
						     &u_id);
	}
	MAPIFreeBuffer(mapi_props);

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	mapiuid_mapi_to_brutus(&u_id, brutus_uid.out());
	lpUID = brutus_uid._retn();

	BRUTUS_LOG_INF("Leaving BRUTUS_IProviderAdmin_i::CreateProvider()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IProviderAdmin_i::DeleteProvider(char *& lpUID)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IProviderAdmin_i::DeleteProvider()");

	MAPIUID u_id;
	mapiuid_brutus_to_mapi_no_alloc(lpUID, u_id);

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = provider_admin_->DeleteProvider(&u_id);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	mapiuid_mapi_to_brutus(&u_id, lpUID);

	BRUTUS_LOG_INF("Leaving BRUTUS_IProviderAdmin_i::DeleteProvider()");
	return br;
}


::BRUTUS::BRESULT BRUTUS_IProviderAdmin_i::OpenProfileSection(const char *lpUID,
							      const char *lpInterface,
							      ::CORBA::ULong ulFlags,
							      ::BRUTUS::IProfSect_out lppProfSect)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IProviderAdmin_i::OpenProfileSection()");

	lppProfSect = ::BRUTUS::IProfSect::_nil();
	LPPROFSECT prof_sect = NULL;

	MAPIUID u_id;
	mapiuid_brutus_to_mapi_no_alloc(lpUID, u_id);

	GUID i_id;
	if (!guid_brutus_to_mapi_no_alloc(lpInterface, &i_id)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into MAPI GUID");
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	ULONG flags = native_flags(ulFlags);

	HRESULT hr;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr = provider_admin_->OpenProfileSection(&u_id,
							 &i_id,
							 flags,
							 &prof_sect);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	if (hr == S_OK)
		lppProfSect = create_object<BRUTUS_IProfSect_i, ::BRUTUS::IProfSect, LPPROFSECT>
			(prof_sect, poa_.in(), mapi_session_);

	BRUTUS_LOG_INF("Leaving BRUTUS_IProviderAdmin_i::OpenProfileSection()");
	return br;
}

::BRUTUS::BRESULT BRUTUS_IProviderAdmin_i::QueryInterface(const char *iid,
							  ::BRUTUS::IUnknown_out ppvObject)
{
	ppvObject = ::BRUTUS::IUnknown::_nil();

	GUID mapi_iid;
	if (!guid_brutus_to_mapi_no_alloc(iid, &mapi_iid)) {
		BRUTUS_LOG_BUG("Could not convert ::BRUTUS::GUID into GUID");
		return ::BRUTUS::BRUTUS_MAPI_E_INVALID_PARAMETER;
	}

	HRESULT hr;
	void *object = NULL;
	{
		ACE_Write_Guard<ACE_RW_Mutex> guard(mutex_);
		hr =  provider_admin_->QueryInterface(mapi_iid, &object);
	}

	::BRUTUS::BRESULT br;
	if (!hresult_to_bresult(hr, br)) {
		BRUTUS_LOG_BUG("Could not convert HRESULT into BRESULT");
		return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
	}

	if (hr == S_OK) {
		if (!create_brutus_object(iid, (LPUNKNOWN)object, poa_, ppvObject, mapi_session_)) {
			BRUTUS_LOG_BUG("Could not create brutus object.");
			((LPUNKNOWN)object)->Release();
			return ::BRUTUS::BRUTUS_INTERNAL_ERROR;
		}
	}

	return br;
}


void BRUTUS_IProviderAdmin_i::Destroy(::CORBA::ULong InstanceID)
{
	BRUTUS_LOG_INF("Entering BRUTUS_IProviderAdmin_i::Destroy()");

	::PortableServer::ObjectId_var oid;
	oid = poa_->servant_to_id(this);

	poa_->deactivate_object(oid);

	BRUTUS_LOG_INF("Leaving BRUTUS_IProviderAdmin_i::Destroy()");
}
