/*
 * "Alternative" (SSP-based) ads8746 touchscreen driver.
 *
 * Copyright (c) 2007 Paul Sokolovsky
 * Copyright (c) 2007 Pierre Gaufillet
 * Copyright (C) 2005-2006 Pawel Kolodziejski
 * Copyright (c) 2005 SDG Systems, LLC
 * Copyright (C) 2003 Joshua Wise
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 */

//#define DEBUG

#define DRIVER_NAME "ts-adc-debounce"

#include <linux/module.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/adc.h>
#include <linux/touchscreen-adc.h>
#include <asm/arch/pxa-regs.h>

#include <asm/arch/hardware.h>
#include <asm/irq.h>
#include <asm/mach/irq.h>

#define XY_READINGS 10
#define Z_READINGS 2

#define BIG_VALUE 999999

#define SAMPLE_TIMEOUT 20	/* sample every 20ms */

enum touchscreen_state {
	STATE_WAIT_FOR_TOUCH,   /* Waiting for a PEN interrupt */
	STATE_SAMPLING          /* Actively sampling ADC */
};

struct adc_work {
	struct delayed_work work;
	struct platform_device *pdev;
};

struct ts_adc {
	struct input_dev *input;
	// Is touchscreen active.
	int state;

	struct adc_request req;
	struct adc_sense *pins;

	struct workqueue_struct *wq;
	struct adc_work work;
};

#define abscmpmin(x,y,d) ( ((int)((x) - (y)) < (int)(d)) && ((int)((y) - (x)) < (int)(d)) )

struct ts_pos {
    unsigned long xd;
    unsigned long yd;
    unsigned int pressure;
    int err;
};

static void report_touchpanel(struct ts_adc *ts, int x, int y, int pressure)
{
	input_report_key(ts->input, BTN_TOUCH, pressure != 0);
	input_report_abs(ts->input, ABS_PRESSURE, pressure);
	input_report_abs(ts->input, ABS_X, x);
	input_report_abs(ts->input, ABS_Y, y);
	input_sync(ts->input);
}

void debounce_samples(struct tsadc_platform_data *params, struct adc_sense *pins, struct ts_pos *tp)
{
	unsigned int t, x, y, z1, z2, pressure = 0;
	int i, ptr;
	int d = params->max_jitter;

	/* X-axis */
	ptr = 0;
	x = pins[ptr++].value;
	for (i = XY_READINGS; i > 0; i--) {
		t = x;
		x = pins[ptr++].value;
		if (abscmpmin(t, x, d))
			break;
	}
	if (!i)
		tp->err = 1;

	/* Y-axis */
	ptr = XY_READINGS;
	y = pins[ptr++].value;
	for (i = XY_READINGS; i > 0; i--) {
		t = y;
		y = pins[ptr++].value;
		if (abscmpmin(t, y, d))
			break;
	}
	if (!i)
		tp->err = 1;

	/* Pressure */
	ptr = XY_READINGS * 2;
	ptr++;
	z1 = pins[ptr++].value;

	ptr++;
	z2 = pins[ptr++].value;

	// RTOUCH = (RXPlate) x (XPOSITION /4096) x [(Z2/Z1) - 1]
	// pressure ~ 1 / RTOUCH
	pressure = abs(z2 - z1) * (x ? x : 1); // X just shouldn't be 0
	if (pressure != 0)
		pressure = params->pressure_factor * z1 / pressure;
	else
		//pressure = BIG_VALUE; // this regresses h5000. workaround is below. needs more investigation
		pressure = z1;

	pr_debug(DRIVER_NAME ": x=%d y=%d z1=%d z2=%d P=%d\n", x, y, z1, z2, pressure);

	tp->xd = x;
	tp->yd = y;
	tp->pressure = pressure;
}

static irqreturn_t ts_adc_debounce_isr(int irq, void* data)
{
	struct platform_device *pdev = data;
	struct tsadc_platform_data *params = pdev->dev.platform_data;
	struct ts_adc *ts = platform_get_drvdata(pdev);

	int pen_up = 0;

	if (params->pen_gpio)
		pen_up = GPLR(params->pen_gpio) & GPIO_bit(params->pen_gpio);

	/* If we arrived here by real IRQ, not timer callback */
        if (ts->state == STATE_WAIT_FOR_TOUCH) {
		/* Don't even bother to do anything for spurious bounce */
        	if (pen_up) {
			pr_debug(DRIVER_NAME ": Bouncing TS IRQ\n");
			return IRQ_HANDLED;
		}
        }

	disable_irq_nosync(params->pen_irq);

	pr_debug(DRIVER_NAME ": Started to sample\n");
	cancel_delayed_work(&ts->work.work);
	queue_delayed_work(ts->wq, &ts->work.work, 0);

	return IRQ_HANDLED;
}

static void ts_adc_debounce_work(struct work_struct *work)
{
	struct delayed_work *delayed_work = container_of(work,
	                                         struct delayed_work, work);
	struct adc_work *adc_work = container_of(delayed_work,
	                                         struct adc_work, work);
	struct platform_device *pdev = adc_work->pdev;
	struct tsadc_platform_data *params = pdev->dev.platform_data;
	struct ts_adc *ts = platform_get_drvdata(pdev);

	struct ts_pos ts_pos;
	int pen_up = 0;
	int finish_sample = 0;

	ts_pos.err = 0;
	ts_pos.pressure = -1;

	/* Sample actual state of pen up/down */
	if (params->pen_gpio)
		pen_up = GPLR(params->pen_gpio) & GPIO_bit(params->pen_gpio);

	if (!pen_up) {
		adc_request_sample(&ts->req);
		debounce_samples(params, ts->req.senses, &ts_pos);
	}

	/* Finish sampling when either pen is up or too low pressure was detected */
	finish_sample = pen_up || (ts_pos.pressure <= params->min_pressure);

	if (finish_sample) {
		report_touchpanel(ts, 0, 0, 0);
		ts->state = STATE_WAIT_FOR_TOUCH;
		pr_debug(DRIVER_NAME ": Finished to sample, reason: pen_up=%d, pressure=%d <= %u\n", !!pen_up, ts_pos.pressure, params->min_pressure);
		enable_irq(params->pen_irq);
	} else {
		//printk("%04d %04d\n", (int)ts_pos.xd, (int)ts_pos.yd);
		if (ts_pos.err)
			printk(DRIVER_NAME ": Sample with too much jitter - ignored\n");
		else
			report_touchpanel(ts, ts_pos.xd, ts_pos.yd, 1);

		queue_delayed_work(ts->wq, &ts->work.work, (SAMPLE_TIMEOUT * HZ) / 1000);
	}

	return;
};

inline static void fill_pins(struct adc_sense *pins, int from, int num, char *with)
{
	int i;
	for (i = from + num - 1; i >= from; i--)
		pins[i].name = with;
}

static int ts_adc_debounce_probe(struct platform_device *pdev)
{
	struct tsadc_platform_data *pdata = pdev->dev.platform_data;
	struct ts_adc *ts;
	int ret, i;

	// Initialize ts data structure.
	ts = kzalloc(sizeof(*ts), GFP_KERNEL);
	if (!ts)
		return -ENOMEM;
	platform_set_drvdata(pdev, ts);

	ts->pins = kmalloc(sizeof(*ts->pins) * (XY_READINGS*2 + Z_READINGS*2), GFP_KERNEL);
	ts->req.senses = ts->pins;
	ts->req.num_senses = (XY_READINGS*2 + Z_READINGS*2);
	i = 0;
	fill_pins(ts->pins, i               , XY_READINGS, pdata->x_pin);
	fill_pins(ts->pins, i += XY_READINGS, XY_READINGS, pdata->y_pin);
	fill_pins(ts->pins, i += XY_READINGS,  Z_READINGS, pdata->z1_pin);
	fill_pins(ts->pins, i +=  Z_READINGS,  Z_READINGS, pdata->z2_pin);
	adc_request_register(&ts->req);

	ts->work.pdev = pdev;
	INIT_DELAYED_WORK(&ts->work.work, ts_adc_debounce_work);
	ts->wq = create_workqueue(pdev->dev.bus_id);
	if (!ts->wq)
		return -ESRCH;

        ts->input = input_allocate_device();
        if (!ts->input)
                return -ENOMEM;

	ts->input->name = DRIVER_NAME;
	ts->input->phys = "touchscreen/adc";

	set_bit(EV_ABS, ts->input->evbit);
	set_bit(EV_KEY, ts->input->evbit);
	set_bit(ABS_X, ts->input->absbit);
	set_bit(ABS_Y, ts->input->absbit);
	set_bit(ABS_PRESSURE, ts->input->absbit);
	set_bit(BTN_TOUCH, ts->input->keybit);
	ts->input->absmin[ABS_PRESSURE] = 0;
	ts->input->absmax[ABS_PRESSURE] = 1;
	ts->input->absmin[ABS_X] = 0;
	ts->input->absmax[ABS_X] = 0;
	ts->input->absmin[ABS_Y] = 32767;
	ts->input->absmax[ABS_Y] = 32767;

	ts->state = STATE_WAIT_FOR_TOUCH;
	set_irq_type(pdata->pen_irq, IRQT_FALLING);
	ret = request_irq(pdata->pen_irq, ts_adc_debounce_isr, SA_INTERRUPT, DRIVER_NAME, pdev);
	if (ret < 0)
		goto irqfail;

	ret = input_register_device(ts->input);
	if (ret < 0)
		goto regfail;

	return 0;

regfail:
	free_irq(pdata->pen_irq, pdev);
irqfail:
	input_free_device(ts->input);
	return ret;
}

static int ts_adc_debounce_remove(struct platform_device *pdev)
{
	struct tsadc_platform_data *params = pdev->dev.platform_data;
	struct ts_adc *ts = platform_get_drvdata(pdev);

	free_irq(params->pen_irq, pdev);
	cancel_delayed_work(&ts->work.work);
	destroy_workqueue(ts->wq);
	adc_request_unregister(&ts->req);
	input_unregister_device(ts->input);
	input_free_device(ts->input);

	return 0;
}

#ifdef CONFIG_PM
static int ts_adc_debounce_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct tsadc_platform_data *params = pdev->dev.platform_data;
	struct ts_adc *ts = platform_get_drvdata(pdev);

	if (ts->state != STATE_SAMPLING)
		disable_irq(params->pen_irq);

	return 0;
}

static int ts_adc_debounce_resume(struct platform_device *pdev)
{
	struct tsadc_platform_data *params = pdev->dev.platform_data;
	struct ts_adc *ts = platform_get_drvdata(pdev);

	ts->state = STATE_WAIT_FOR_TOUCH;
	enable_irq(params->pen_irq);

	return 0;
}
#endif

static struct platform_driver ts_adc_debounce_driver = {
	.driver		= {
		.name	= DRIVER_NAME,
	},
	.probe		= ts_adc_debounce_probe,
	.remove         = ts_adc_debounce_remove,
#ifdef CONFIG_PM
	.suspend	= ts_adc_debounce_suspend,
	.resume		= ts_adc_debounce_resume,
#endif
};

static int __init ts_adc_debounce_init(void)
{
	return platform_driver_register(&ts_adc_debounce_driver);
}

static void __exit ts_adc_debounce_exit(void)
{
	platform_driver_unregister(&ts_adc_debounce_driver);
}

module_init(ts_adc_debounce_init)
module_exit(ts_adc_debounce_exit)

MODULE_AUTHOR("Joshua Wise, Pawel Kolodziejski, Paul Sokolovsky, Aric Blumer");
MODULE_DESCRIPTION("Lightweight touchscreen driver with debouncing for ADC chip");
MODULE_LICENSE("GPL");
