#include "mysccs.h"
SCCSID("@(#)contexts.c   %E%   SAP   %I%")

static char * this_File GNU_UNUSED = __FILE__;

/************************************************************************/
/* $Id: platform.c,v 1.1.1.1 1998/11/06 14:33:18 d019080 Exp $
 ************************************************************************/
/*
 *  (C) Copyright 1998  SAP AG Walldorf
 *
 * Author:  Martin Rex
 * 
 * SAP AG DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL SAP AG BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 */

#include "gsstest.h"
#include "non_ansi.h"
#include <time.h>




#ifdef UNIX

#  include <sys/types.h>
#  include <unistd.h>
#  include <sys/utsname.h>
#  include <pwd.h>


/*
 * sys_uts_osname()
 *
 *
 */
int
sys_uts_osname( struct sys_platform_s * p_pls )
{
   struct utsname  uts;
   int             rc;

   memset( &uts, 0, sizeof(uts) );
   rc = uname( &uts );
   if ( rc>=0 ) {
      strncpy( p_pls->os_name, uts.sysname, sizeof(p_pls->os_name) );
      p_pls->os_name[sizeof(p_pls->os_name)-1] = '\0';
      return(0);
   } else {
      XVEB((V_BUG, "uname() failed: %s\n", strerror(errno) ));
      memset( p_pls->os_name, 0, sizeof(p_pls->os_name) );
   }

   return(1);

} /* sys_uts_osname() */


/*
 * sys_uts_osrelease()
 *
 *
 */
int
sys_uts_osrelease( struct sys_platform_s * p_pls )
{
   struct utsname   uts;
   int              rc;

   memset( &uts, 0, sizeof(uts) );
   rc = uname( &uts );
   if ( rc>=0 ) {
      strncpy( p_pls->os_release, uts.release, sizeof(p_pls->os_release) );
      p_pls->os_release[sizeof(p_pls->os_release)-1] = '\0';
      return(0);
   } else {
      XVEB((V_BUG, "uname() failed: %s\n", strerror(errno) ));
      memset( p_pls->os_release, 0, sizeof(p_pls->os_release) );
   }

   return(1);

} /* sys_uts_osrelease() */


/*
 * sys_uts_oshardware()
 *
 *
 */
int
sys_uts_oshardware( struct sys_platform_s * p_pls )
{
   struct utsname  uts;
   int             rc;

   memset( &uts, 0, sizeof(uts) );
   rc = uname( &uts );
   if ( rc>=0 ) {
      strncpy( p_pls->os_hardware, uts.machine, sizeof(p_pls->os_hardware) );
      p_pls->os_hardware[sizeof(p_pls->os_hardware)-1] = '\0';
      return(0);
   } else {
      XVEB((V_BUG, "uname() failed: %s\n", strerror(errno) ));
      memset( p_pls->os_hardware, 0, sizeof(p_pls->os_hardware) );
   }

   return(1);

} /* sys_uts_hardware() */

  
/*
 * sys_gethostname()
 *
 *
 */
int
sys_gethostname( struct sys_platform_s * p_pls )
{
   int    rc;

   memset( p_pls->hostname, 0, sizeof(p_pls->hostname) );
   rc = gethostname( p_pls->hostname, sizeof(p_pls->hostname)-1 );
   if ( rc!=0 ) {
      XVEB((V_BUG, "gethostname() failed: %s\n", strerror(errno) ));
   }

   p_pls->hostname[sizeof(p_pls->hostname)-1] = '\0';

   if ( rc==0 )
      return(rc);

   return(1);

} /* sys_gethostname() */


/*
 * sys_getpwuid()
 *
 *
 */
int
sys_getpwuid( struct sys_platform_s * p_pls )
{
   struct passwd   * result;

   result = getpwuid( getuid() );
   memset( p_pls->user, 0, sizeof(p_pls->user) );
   if ( result==NULL ) {
      XVEB((V_BUG, "getpwent(%lu) failed: %s\n",
	           (unsigned long)getuid(), strerror(errno) ));
   } else {
      strncpy( p_pls->user, result->pw_name, sizeof(p_pls->user) );
      p_pls->user[sizeof(p_pls->user)-1] = '\0';
      return(0);
   }

   return(1);

} /* sys_getpwuid() */



/*
 * sys_popen()
 *
 * read a single line of output from a shell command
 * into a buffer
 *
 */
int
sys_popen( char   * p_command,  char   * p_buf,   size_t  p_buf_max )
{
   FILE   * fp  = NULL;
   size_t   len;
   int	    rc  = 1;

   memset( p_buf, 0, p_buf_max);

   fp = popen( p_command, "r" );
   if ( fp!=NULL ) {
      len = fread( p_buf, 1, p_buf_max-1, fp );
      if ( len>0 ) {
	 /* looks good -- we were able to read from stdout */
	 p_buf[len] = 0;
	 while( len>0 && (p_buf[len-1]=='\r' || p_buf[len-1]=='\n') ) {
	    /* cut end-of-line characters, in case there are any */
	    len--;
	    p_buf[len] = '\0';
	 }
	 if ( len>0 ) {
	    rc = 0; /* success -- we have a non-empty string left */
	 }
      }

      pclose( fp );

   } /* fp!=NULL */

   return(rc);

} /* sys_popen() */


	  
#endif /* UNIX */




#ifdef _WIN32


/*
 * win_osname()
 */
int
win_osname( struct sys_platform_s   * p_pls )
{
   OSVERSIONINFO     osvi;
   char           *  product;

   memset( p_pls->os_name, 0, sizeof(p_pls->os_name) );
   memset( &osvi, 0, sizeof(osvi) );

   osvi.dwOSVersionInfoSize = sizeof(osvi); /* REQUIRED! */
   GetVersionEx( &osvi );

   switch( (int) osvi.dwPlatformId ) {

      case VER_PLATFORM_WIN32s:
		  product = "Microsoft Win32";
		  break;

      case VER_PLATFORM_WIN32_WINDOWS:
		  product = (osvi.dwMinorVersion==0)
			    ? "Microsoft Windows 95" : "Microsoft Windows 98";
		  break;
		     
      case VER_PLATFORM_WIN32_NT:
		  product = "Microsoft Windows NT";
		  break;

      default:	  product = "Windows (?!)";
		  break;

   }

   strncpy( p_pls->os_name, product, sizeof(p_pls->os_name) );
   p_pls->os_name[sizeof(p_pls->os_name)-1] = '\0';

   return(0);

} /* win_osname() */


/*
 * win_osrelease()
 */
int
win_osrelease( struct sys_platform_s   * p_pls )
{
   OSVERSIONINFO     osvi;
   char              tmpbuf[256];

   memset( p_pls->os_release, 0, sizeof(p_pls->os_release) );
   memset( &osvi, 0, sizeof(osvi) );

   osvi.dwOSVersionInfoSize = sizeof(osvi); /* REQUIRED! */
   GetVersionEx( &osvi );

      
   sprintf(tmpbuf, "%d.%d (Build %d)",
		   (int) osvi.dwMajorVersion,
		   (int) osvi.dwMinorVersion,
		   (int) (osvi.dwBuildNumber & 0xFFFF) );

   strncpy( p_pls->os_release, tmpbuf, sizeof(p_pls->os_release) );
   p_pls->os_release[sizeof(p_pls->os_release)-1] = '\0';
   
   return(0);

} /* win_osrelease() */


/*
 * win_hardware()
 */
int
win_hardware( struct sys_platform_s  * p_pls )
{
   SYSTEM_INFO    sinfo;
   size_t	  len;
   char         * cpu  = "(unknown)";
   char         * cpu2 = "";
   char           cpu_level[128];
   char	          cpu_rev[128];

   memset( p_pls->os_hardware, 0, sizeof(p_pls->os_hardware) );
   memset( &sinfo, 0, sizeof(sinfo) );
   GetSystemInfo( &sinfo );

   switch( (int)sinfo.wProcessorArchitecture ) {
      case PROCESSOR_ARCHITECTURE_INTEL:     cpu = "x86";      break;
      case PROCESSOR_ARCHITECTURE_ALPHA:     cpu = "Alpha";    break;
      case PROCESSOR_ARCHITECTURE_PPC:	     cpu = "PowerPC";  break;
      case PROCESSOR_ARCHITECTURE_MIPS:	     cpu = "Mips";     break;
      default:	     break;
   }

#if 0
   if ( sinfo.wProcessorArchitecture == PROCESSOR_ARCHITECTURE_INTEL ) {
      switch( (int)sinfo.dwProcessorType ) {
	 case PROCESSOR_INTEL_386:	     cpu2 = " (386)";  break;
	 case PROCESSOR_INTEL_486:	     cpu2 = " (486)";  break;
	 case PROCESSOR_INTEL_PENTIUM:	     cpu2 = " (Pentium)";  break;
	 default:    break;
      }
   }
#endif

   sprintf( cpu_level, "cpu_level=%ld", (unsigned long)sinfo.wProcessorLevel );
   sprintf( cpu_rev, "cpu_rev=0x%04lx", (unsigned long)sinfo.wProcessorRevision );

   len = strlen(cpu) + strlen(cpu2) + strlen(cpu_level) + strlen(cpu_rev) + 4;
   if ( len < sizeof(p_pls->os_hardware) ) {
      sprintf(p_pls->os_hardware, "%s%s  %s, %s", cpu, cpu2, cpu_level, cpu_rev );
   } else {
      strncpy(p_pls->os_hardware, cpu, sizeof(p_pls->os_hardware));
   }
   p_pls->os_hardware[sizeof(p_pls->os_hardware)-1] = '\0';

   return(0);
   
} /* win_hardware() */


/*
 * win_username()
 */
int
win_username( struct sys_platform_s * p_pls )
{
   BOOL   result;
   DWORD  len;

   len = sizeof(p_pls->user) - 1;
   memset( p_pls->user, 0, sizeof(p_pls->user) );

   result = GetUserName( p_pls->user, &len );
   if ( result==0 ) {
      XVEB((V_BUG, "GetUserName() returned %08lx!\n",
		   (unsigned long) GetLastError() ));
      memset( p_pls->user, 0, sizeof(p_pls->user) ); /* Don't rely on M$ */
      return(1);
   }

   p_pls->user[sizeof(p_pls->user)-1] = '\0';

   return(0);

} /* win_username() */


/*
 * win_computername()
 */
int
win_computername( struct sys_platform_s * p_pls )
{
   BOOL   result;
   DWORD  len;

   len = sizeof(p_pls->hostname) - 1;
   memset( p_pls->hostname, 0, sizeof(p_pls->hostname) );

   result = GetComputerName( p_pls->hostname, &len );
   if (result==0) {
      XVEB((V_BUG, "GetComputerName() returned %08lx!\n",
		   (unsigned long) GetLastError() ));
      memset( p_pls->hostname, 0, sizeof(p_pls->hostname) );
      return(1);
   }

   p_pls->hostname[sizeof(p_pls->hostname)-1] = '\0';


   return(0);

} /* win_computername() */


#endif /* _WIN32 */


static char *days[] =
   { "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };

static char *months[] =
   { "Jan", "Feb", "Mar", "Apr", "May", "Jun",
     "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

/*
 * sys_greeting()
 *
 *
 *
 */
void
sys_greeting( char * p_prefix, struct sys_platform_s * p_pls )
{
   struct tm             * ltime   = NULL;
   char			 * unknown = "(unknown)";
   Ulong		   high, low;
   Ulong		   dbg_speed, opt_speed;
   int			   tzoffset;
   int			   tzhours;
   int			   tzrest;
   int			   tzmins;
   time_t		   now;
   int                     rc      = 0;

   XVEB((V_SHOW, "%s=====================================================================\n", p_prefix));

   now   = time(NULL);
   ltime = localtime(&now);
   if ( ltime==NULL ) {
      XVEB((V_BUG, "HUH? localtime() failed!\n"));
   } else {
      tzoffset = timezone;
      if ( daylight!=0 ) { tzoffset -= 3600; }
      tzhours = (tzoffset)/3600;
      tzrest  = tzoffset - tzhours*3600;
      tzmins  = ((tzrest>=0) ? tzrest : tzrest + 3600) / 60 ;
      XVEB((V_SHOW, "%s  Current Date&Time :  %s, %02d-%s-%4d   %02d:%02d:%02d   GMT %+03d:%02d\n", p_prefix,
		    days[ltime->tm_wday],
		    ltime->tm_mday, months[ltime->tm_mon], 1900+ltime->tm_year,
		    ltime->tm_hour, ltime->tm_min, ltime->tm_sec,
		    tzhours, tzmins ));
   }

   Strmaxcpy( p_pls->os_name,     unknown, sizeof(p_pls->os_name)     );
   Strmaxcpy( p_pls->os_release,  unknown, sizeof(p_pls->os_release)  );
   Strmaxcpy( p_pls->os_hardware, unknown, sizeof(p_pls->os_hardware) );
   Strmaxcpy( p_pls->hostname,    unknown, sizeof(p_pls->hostname)    );
   Strmaxcpy( p_pls->user,        unknown, sizeof(p_pls->user)        );

   rc +=  SYS_GET_OSNAME(     p_pls );
   rc +=  SYS_GET_OSRELEASE(  p_pls );
   rc +=  SYS_GET_OSHARDWARE( p_pls );
   rc +=  SYS_GET_HOSTNAME(   p_pls );
   rc +=  SYS_GET_USERNAME(   p_pls );

   XVEB((V_SHOW, "%s  Operating System  :  %.100s\n", p_prefix, p_pls->os_name     ));
   XVEB((V_SHOW, "%s          -Release  :  %.100s\n", p_prefix, p_pls->os_release  ));
   XVEB((V_SHOW, "%s  Hardware/Machine  :  %.100s\n", p_prefix, p_pls->os_hardware ));

   sys_machine_speed( &dbg_speed, &opt_speed );
   high  = opt_speed / 100;
   low   = opt_speed - (high*100);
   XVEB((V_SHOW, "%s  Perf-Index (p-90) :  opt= %lu.%02lu", p_prefix, high, low ));
   high  = dbg_speed / 100;
   low   = dbg_speed - (high*100);
   XVEB((V_SHOW, "   dbg= %lu.%02lu\n", high, low ));

   if (hrtimer_resolution!=0) {
      XVEB((V_SHOW, "%s  Timer Resolution  :  ", p_prefix));
      high = hrtimer_resolution / 1000;
      low  = hrtimer_resolution - (high * 1000);
      XVEB((V_SHOW, "%lu.%03lu millisec using \"%s\"\n", high, low, hrtimer_name ));
   }

   XVEB((V_SHOW, "%s  Hostname          :  %.100s\n", p_prefix, p_pls->hostname ));
   XVEB((V_SHOW, "%s  Current user      :  %.100s\n", p_prefix, p_pls->user     ));

   XVEB((V_SHOW, "%s=====================================================================\n", p_prefix));

   return;

} /* sys_greeting() */




/*
 * sys_machine_speed()
 *
 * Description:
 *   Get a feeling for the CPU power of the current machine
 *   This is used to scale performance-related limits
 *
 */
void
sys_machine_speed( Ulong * dbg_speed, Ulong * opt_speed )
{
   char           * this_Call = "sys_machine_speed";
   Ulong            vals[16];
   Ulong            avg;
   Ulong            speed;
   unsigned char    digest[16];
   int              i;
   unsigned char  * memory;
   haMD5_CTX        md5_ctx;

   (*opt_speed) = (*dbg_speed) = 0;

   memory = (unsigned char *)malloc(BENCH_MEMSIZE);
   if ( memory==NULL ) {
      XVEB((V_BUG, "%s() error: malloc(%lu) failed!\n", this_Call, (Ulong) BENCH_MEMSIZE));
      return;
   }

   for ( i=0 ; i<16 ; i++ ) {
      start_timer();
      memset( memory, 0x5a, BENCH_MEMSIZE );
      haMD5Init(   &md5_ctx );
      haMD5Update( &md5_ctx, memory, BENCH_MEMSIZE );
      haMD5Final(   digest, &md5_ctx );
      vals[i] = read_timer();
   }

   avg = sample_avg( vals, sizeof(vals)/sizeof(vals[0]) );

   if ( avg>0 ) {
      speed        = (100 * PENTIUM90_SPEED_DBG) / avg;
      (*dbg_speed) = ((speed+5)/10) * 10;

      speed        = (100 * PENTIUM90_SPEED_OPT) / avg;
      (*opt_speed) = ((speed+5)/10) * 10;
   }

   if ( memory!=NULL ) {
      free( memory );
      memory = NULL;
   }

   return;

} /* sys_machine_speed() */









/*
 * Adapted (29-Apr-96) Martin Rex
 *
 * This source was extracted from RFC1321 and "formatted" for SAP-use
 *   - Standard-C function definitions
 *   - indenting,
 *   - Standard-C memcpy() and memset()
 *   - derive 32-bit unsigned integers from Uint32
 *
 * RFC 1321:   Ron Rivest, "The MD5 Message-Digest Algorithm"
 *               MIT Laboratory for Computer Science
 *                                      and RSA Data Security, Inc.
 *               April 1992
 *
 */

/*
 * MD5.C - RSA Data Security, Inc., MD5 message-digest algorithm
 */

/*
 * Copyright (C) 1991-2, RSA Data Security, Inc. Created 1991.
 * All rights reserved.
 *
 * License to copy and use this software is granted provided that it
 * is identified as the "RSA Data Security, Inc. MD5 Message-Digest
 * Algorithm" in all material mentioning or referencing this software
 * or this function.
 *
 * License is also granted to make and use derivative works provided
 * that such works are identified as "derived from the RSA Data
 * Security, Inc. MD5 Message-Digest Algorithm" in all material
 * mentioning or referencing the derived work.
 *
 * RSA Data Security, Inc. makes no representations concerning either
 * the merchantability of this software or the suitability of this
 * software for any particular purpose. It is provided "as is"
 * without express or implied warranty of any kind.
 *
 * These notices must be retained in any copies of any part of this
 * documentation and/or software.
 */



/*
 * Constants for haMD5Transform routine.
 */

#define S11 7
#define S12 12
#define S13 17
#define S14 22
#define S21 5
#define S22 9
#define S23 14
#define S24 20
#define S31 4
#define S32 11
#define S33 16
#define S34 23
#define S41 6
#define S42 10
#define S43 15
#define S44 21

static void haMD5Transform (Uint32 [4], unsigned char [64]);
static void haEncode (unsigned char *, Uint32 *, unsigned int);
static void haDecode (Uint32 *, unsigned char *, unsigned int);

static unsigned char PADDING[64] = {
  0x80, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};

/*
 * F, G, H and I are basic MD5 functions.
 */
#define F(x, y, z) (((x) & (y)) | ((~x) & (z)))
#define G(x, y, z) (((x) & (z)) | ((y) & (~z)))
#define H(x, y, z) ((x) ^ (y) ^ (z))
#define I(x, y, z) ((y) ^ ((x) | (~z)))

/*
 * ROTATE_LEFT rotates x left n bits.
 */
#define ROTATE_LEFT(x, n) (((x) << (n)) | ((x) >> (32-(n))))

/*
 * FF, GG, HH, and II transformations for rounds 1, 2, 3, and 4.
 * Rotation is separate from addition to prevent recomputation.
 */
#define FF(a, b, c, d, x, s, ac) {				 \
		(a) += F ((b), (c), (d)) + (x) + (Uint32)(ac);	 \
		(a) = ROTATE_LEFT ((a), (s));			 \
		(a) += (b);					 \
	}

#define GG(a, b, c, d, x, s, ac) {				 \
		(a) += G ((b), (c), (d)) + (x) + (Uint32)(ac);   \
		(a) = ROTATE_LEFT ((a), (s));			 \
		(a) += (b);					 \
	}

#define HH(a, b, c, d, x, s, ac) {				 \
		(a) += H ((b), (c), (d)) + (x) + (Uint32)(ac);   \
		(a) = ROTATE_LEFT ((a), (s));			 \
		(a) += (b);					 \
	}

#define II(a, b, c, d, x, s, ac) {				 \
		(a) += I ((b), (c), (d)) + (x) + (Uint32)(ac);   \
		(a) = ROTATE_LEFT ((a), (s));			 \
		(a) += (b);					 \
	}



/*
 * MD5 initialization. Begins an MD5 operation, writing a new context.
 */
void
haMD5Init (  haMD5_CTX   * context  )	/* context */
{
  context->count[0] = context->count[1] = 0;
  /* Load magic initialization constants. */
  context->state[0] = 0x67452301;
  context->state[1] = 0xefcdab89;
  context->state[2] = 0x98badcfe;
  context->state[3] = 0x10325476;

} /* haMD5Init() */



/*
 * MD5 block update operation. Continues an MD5 message-digest
 * operation, processing another message block, and updating the
 * context.
 */
void
haMD5Update (  haMD5_CTX      * context,	/* context		 */
	       unsigned char  * input,		/* input block		 */
	       size_t           inputLen  )	/* length of input block */
{
  unsigned int i, index, partLen;

  /* Compute number of bytes mod 64 */
  index = (unsigned int)((context->count[0] >> 3) & 0x3F);

  /* Update number of bits */
  if ( (context->count[0] += ((Uint32)inputLen << 3))
       < ((Uint32)inputLen << 3) )
     context->count[1]++;

  context->count[1] += ((Uint32)inputLen >> 29);

  partLen = 64 - index;

  /*
   * Transform as many times as possible.
   */
  if (inputLen >= partLen) {
     memcpy( &(context->buffer[index]), input, partLen );
     haMD5Transform (context->state, context->buffer);

     for (i = partLen; i + 63 < inputLen; i += 64)
        haMD5Transform (context->state, &input[i]);

     index = 0;
  } else {
     i = 0;
  }

  /* Buffer remaining input */
  memcpy( &(context->buffer[index]), &(input[i]), inputLen-i );

} /* haMD5Update() */



/*
 * MD5 finalization. Ends an MD5 message-digest operation,
 * writing the message digest and zeroizing the context.
 */
void
haMD5Final (  unsigned char     digest[16],	/* message digest */
	      haMD5_CTX       * context  )	/* context	  */
{
  unsigned char bits[8];
  unsigned int index, padLen;

  /* Save number of bits */
  haEncode (bits, context->count, 8);

  /* Pad out to 56 mod 64. */
  index = (unsigned int)((context->count[0] >> 3) & 0x3f);
  padLen = (index < 56) ? (56 - index) : (120 - index);
  haMD5Update (context, PADDING, padLen);

  /* Append length (before padding) */
  haMD5Update (context, bits, 8);

  /* Store state in digest */
  haEncode (digest, context->state, 16);

  /* Zeroize sensitive information. */
  memset ( context, 0, sizeof (*context) );

} /* haMD5Final() */



/*
 * MD5 basic transformation. Transforms state based on block.
 */
static void
haMD5Transform (  Uint32         state[4],
		  unsigned char  block[64]  )
{
  Uint32 a = state[0], b = state[1], c = state[2], d = state[3], x[16];

  haDecode (x, block, 64);

  /* Round 1 */
  FF (a, b, c, d, x[ 0], S11, 0xd76aa478); /* 1 */
  FF (d, a, b, c, x[ 1], S12, 0xe8c7b756); /* 2 */
  FF (c, d, a, b, x[ 2], S13, 0x242070db); /* 3 */
  FF (b, c, d, a, x[ 3], S14, 0xc1bdceee); /* 4 */
  FF (a, b, c, d, x[ 4], S11, 0xf57c0faf); /* 5 */
  FF (d, a, b, c, x[ 5], S12, 0x4787c62a); /* 6 */
  FF (c, d, a, b, x[ 6], S13, 0xa8304613); /* 7 */
  FF (b, c, d, a, x[ 7], S14, 0xfd469501); /* 8 */
  FF (a, b, c, d, x[ 8], S11, 0x698098d8); /* 9 */
  FF (d, a, b, c, x[ 9], S12, 0x8b44f7af); /* 10 */
  FF (c, d, a, b, x[10], S13, 0xffff5bb1); /* 11 */
  FF (b, c, d, a, x[11], S14, 0x895cd7be); /* 12 */
  FF (a, b, c, d, x[12], S11, 0x6b901122); /* 13 */
  FF (d, a, b, c, x[13], S12, 0xfd987193); /* 14 */
  FF (c, d, a, b, x[14], S13, 0xa679438e); /* 15 */
  FF (b, c, d, a, x[15], S14, 0x49b40821); /* 16 */

 /* Round 2 */
  GG (a, b, c, d, x[ 1], S21, 0xf61e2562); /* 17 */
  GG (d, a, b, c, x[ 6], S22, 0xc040b340); /* 18 */
  GG (c, d, a, b, x[11], S23, 0x265e5a51); /* 19 */
  GG (b, c, d, a, x[ 0], S24, 0xe9b6c7aa); /* 20 */
  GG (a, b, c, d, x[ 5], S21, 0xd62f105d); /* 21 */
  GG (d, a, b, c, x[10], S22,  0x2441453); /* 22 */
  GG (c, d, a, b, x[15], S23, 0xd8a1e681); /* 23 */
  GG (b, c, d, a, x[ 4], S24, 0xe7d3fbc8); /* 24 */
  GG (a, b, c, d, x[ 9], S21, 0x21e1cde6); /* 25 */
  GG (d, a, b, c, x[14], S22, 0xc33707d6); /* 26 */
  GG (c, d, a, b, x[ 3], S23, 0xf4d50d87); /* 27 */
  GG (b, c, d, a, x[ 8], S24, 0x455a14ed); /* 28 */
  GG (a, b, c, d, x[13], S21, 0xa9e3e905); /* 29 */
  GG (d, a, b, c, x[ 2], S22, 0xfcefa3f8); /* 30 */
  GG (c, d, a, b, x[ 7], S23, 0x676f02d9); /* 31 */
  GG (b, c, d, a, x[12], S24, 0x8d2a4c8a); /* 32 */

  /* Round 3 */
  HH (a, b, c, d, x[ 5], S31, 0xfffa3942); /* 33 */
  HH (d, a, b, c, x[ 8], S32, 0x8771f681); /* 34 */
  HH (c, d, a, b, x[11], S33, 0x6d9d6122); /* 35 */
  HH (b, c, d, a, x[14], S34, 0xfde5380c); /* 36 */
  HH (a, b, c, d, x[ 1], S31, 0xa4beea44); /* 37 */
  HH (d, a, b, c, x[ 4], S32, 0x4bdecfa9); /* 38 */
  HH (c, d, a, b, x[ 7], S33, 0xf6bb4b60); /* 39 */
  HH (b, c, d, a, x[10], S34, 0xbebfbc70); /* 40 */
  HH (a, b, c, d, x[13], S31, 0x289b7ec6); /* 41 */
  HH (d, a, b, c, x[ 0], S32, 0xeaa127fa); /* 42 */
  HH (c, d, a, b, x[ 3], S33, 0xd4ef3085); /* 43 */
  HH (b, c, d, a, x[ 6], S34,  0x4881d05); /* 44 */
  HH (a, b, c, d, x[ 9], S31, 0xd9d4d039); /* 45 */
  HH (d, a, b, c, x[12], S32, 0xe6db99e5); /* 46 */
  HH (c, d, a, b, x[15], S33, 0x1fa27cf8); /* 47 */
  HH (b, c, d, a, x[ 2], S34, 0xc4ac5665); /* 48 */

  /* Round 4 */
  II (a, b, c, d, x[ 0], S41, 0xf4292244); /* 49 */
  II (d, a, b, c, x[ 7], S42, 0x432aff97); /* 50 */
  II (c, d, a, b, x[14], S43, 0xab9423a7); /* 51 */
  II (b, c, d, a, x[ 5], S44, 0xfc93a039); /* 52 */
  II (a, b, c, d, x[12], S41, 0x655b59c3); /* 53 */
  II (d, a, b, c, x[ 3], S42, 0x8f0ccc92); /* 54 */
  II (c, d, a, b, x[10], S43, 0xffeff47d); /* 55 */
  II (b, c, d, a, x[ 1], S44, 0x85845dd1); /* 56 */
  II (a, b, c, d, x[ 8], S41, 0x6fa87e4f); /* 57 */
  II (d, a, b, c, x[15], S42, 0xfe2ce6e0); /* 58 */
  II (c, d, a, b, x[ 6], S43, 0xa3014314); /* 59 */
  II (b, c, d, a, x[13], S44, 0x4e0811a1); /* 60 */
  II (a, b, c, d, x[ 4], S41, 0xf7537e82); /* 61 */
  II (d, a, b, c, x[11], S42, 0xbd3af235); /* 62 */
  II (c, d, a, b, x[ 2], S43, 0x2ad7d2bb); /* 63 */
  II (b, c, d, a, x[ 9], S44, 0xeb86d391); /* 64 */

  state[0] += a;
  state[1] += b;
  state[2] += c;
  state[3] += d;

  /* Zeroize sensitive information. */
  memset ( x, 0, sizeof (x));

} /* haMD5Transform() */



/*
 * Encodes input (Uint32) into output (unsigned char). Assumes len is
 * a multiple of 4.
 */
static void
haEncode (  unsigned char   * output,
	    Uint32          * input,
	    unsigned int      len  )
{
  unsigned int i, j;

  for (i = 0, j = 0; j < len; i++, j += 4) {
     output[j]   = (unsigned char) (input[i] & 0xff);
     output[j+1] = (unsigned char) ((input[i] >> 8) & 0xff);
     output[j+2] = (unsigned char) ((input[i] >> 16) & 0xff);
     output[j+3] = (unsigned char) ((input[i] >> 24) & 0xff);
  }

  return; /* void */

} /* haEncode() */



/*
 * Decodes input (unsigned char) into output (Uint32). Assumes len is
 * a multiple of 4.
 */
static void
haDecode (  Uint32         * output,
	    unsigned char  * input,
	    unsigned int     len  )
{
  unsigned int i, j;

  for (i = 0, j = 0; j < len; i++, j += 4)
     output[i] = ((Uint32)input[j])
		 | (((Uint32)input[j+1]) << 8)
                 | (((Uint32)input[j+2]) << 16)
		 | (((Uint32)input[j+3]) << 24);

  return; /* void */

} /* haDecode() */






