#include <windows.h>
#include <krb5.h>
#include <k5-int.h>
#include <kadm5/admin.h>
#include <stdio.h>
#include <unistd.h>
#include <syslog.h>
#include "com_err.h"

#define DEFAULT_PRINCNAME "pwsync"
#define PWS_ID "Kerbnet PasswordSync"
#define DEFAULT_SYNC_SRVTAB "\\lib\\krb5kdc\\pwsync.srvtab"

static krb5_context global_context;
static char *global_def_realm = NULL;
static char *global_princstr = NULL;
static char *global_srvtab = 0;
static void *global_handle = 0;

/*
 * Connect to the kadmin daemon.
 */
static int do_kerbnet_kadmin_connect()
{
  int ret;
  kadm5_config_params params;

  if(global_handle == 0) {
    memset(&params, '\0', sizeof(params));
    params.mask |= KADM5_CONFIG_REALM;
    params.realm = global_def_realm;

    if(ret = kadm5_init_with_skey( global_princstr, global_srvtab,
			KADM5_ADMIN_SERVICE,
			&params,
			KADM5_STRUCT_VERSION,
			KADM5_API_VERSION_2,
			&global_handle)) {
      com_err(PWS_ID, ret,
          ":do_kerbnet_kadmin_connect: kadm5_init_with_skey with principal %s, srvtab %s failed.",
          global_princstr, global_srvtab);
      global_handle = 0;
      return -1;
    }
  }
  return 0;
}

/*
 * Disconnect from the kadmin daemon.
 */
static int do_kerbnet_kadmin_disconnect()
{
  if(global_handle != 0) {
    kadm5_destroy(global_handle);
    global_handle = 0;
  }
  return 0;
}

/*
 * Return a default path for the srvtab.
 */

static char *get_default_srvtab()
{
  int size;
  char *kh;
  char *srvtab;
  
  if((kh = getenv("KERBNET_HOME")) == 0) {
    com_err(PWS_ID, 0, ":get_default_srvtab. KERBNET_HOME environment variable not set.");
    return 0;
  }
  size = strlen("FILE:") + strlen(kh) + strlen(DEFAULT_SYNC_SRVTAB) + 1;
  if((srvtab = malloc(size)) == 0) {
    com_err(PWS_ID, ENOMEM, ":get_default_srvtab: malloc failed.");
    return 0;
  }
  strcpy(srvtab, "FILE:");
  strcat(srvtab, kh);
  strcat(srvtab, DEFAULT_SYNC_SRVTAB);
  return srvtab;
}

/*
 * Log a message to the standard log_message function.
 */
 
void log_err( const char *name, long err, const char *fmt, va_list args)
{
  log_message_va(LOG_ERR, fmt, args);  
}

/*
 * Initialize the connection to the kadmin interface.
 */
 
int kerbnet_sync_init(int argc, char **argv)
{
  extern char *optarg;
  int allocated_princstr = FALSE;
  int allocated_def_realm = FALSE;
  int allocated_srvtab = FALSE;
  krb5_principal princ;
  int opt;
  int ret;

  if(ret = krb5_init_context(&global_context)) {
    log_message(LOG_ERR, "kerbnet_sync_init: krb5_init_context failed. Error was %s",
                error_message(ret));
    return -1;
  }
  
  krb5_init_ets(global_context);

  (void)set_com_err_hook( log_err );
  
  while(( opt = getopt( argc, argv, "p:r:s:n")) != EOF) {
    switch(opt) {
    case 'p':
      global_princstr = optarg;
      allocated_princstr = FALSE;
      break;
    case 'r':
      global_def_realm = optarg;
      allocated_def_realm = FALSE;
      break;
    case 's':
      global_srvtab = optarg;
      break;
    case 'n': /* This is the default nofork - ignore it. */
      break;
    default:
      com_err(PWS_ID, 0, ":kerbnet_sync_init: unknown argument. Args are -p<principal> -r<realm> -s<srvtab>");
      return -1;
    }
  }

  if(global_srvtab == NULL) {
    if((global_srvtab = get_default_srvtab()) == 0)
      return -1;
    allocated_srvtab = TRUE;
  }
  
  if(global_princstr == NULL) {
    if(ret = krb5_sname_to_principal( global_context, NULL, DEFAULT_PRINCNAME,
                                      KRB5_NT_SRV_HST, &princ)) {
      krb5_free_context(global_context);
      com_err(PWS_ID, ret, ":kerbnet_sync_init: krb5_sname_to_principal failed. Error was %s.",
              error_message(ret));
      return -1;
    }
    if(ret= krb5_unparse_name( global_context, princ, &global_princstr)) {
      krb5_free_context(global_context);
      krb5_free_principal( global_context, princ);
      com_err(PWS_ID, ret, ":kerbnet_sync_init: krb5_unparse_name failed. Error was %s.",
              error_message(ret));
      return -1;
    }
    allocated_princstr = TRUE;
    krb5_free_principal( global_context, princ);
  }
  
  if(global_def_realm == NULL) {
    if(ret = krb5_get_default_realm(global_context, &global_def_realm)) {
      if(allocated_princstr) {
        free(global_princstr);
        global_princstr = 0;
      }
      krb5_free_context(global_context);
      com_err(PWS_ID, ret, ":kerbnet_sync_init: krb5_get_default_realm failed. Error was %s.",
              error_message(ret));
      global_def_realm = 0;
      return -1;
    }
    allocated_def_realm = TRUE;
  }

  return 0;
}

/*
 * Simple wctombs.
 */
 
static int my_wctombs( const wchar_t *in, char **out)
{
  int size;
  char *buf;
  
  if((size = WideCharToMultiByte( CP_ACP, 0, in, -1, 0, 0, 0, 0)) == 0) {
    com_err(PWS_ID, 0, ":my_wctombs: MultiByteToWideChar failed with NT error %s.",
              str_oserr(GetLastError()));
    return -1;
  }
  /* Allocate size bytes */
  if((*out = malloc(size)) == 0) {
    com_err(PWS_ID, ENOMEM, ":my_wctombs: malloc failed.");
    return -1;
  }
  if(WideCharToMultiByte( CP_ACP, 0, in, -1, *out, size, 0, 0) == FALSE) {
    com_err(PWS_ID, 0, ":my_wctombs: second MultiBytetOWideChar failed with NT error %s.",
              str_oserr(GetLastError()));
    free(*out);
    return -1;
  }
  return 0;
}

/*
 * convert username to asci and then convert to a kerb5 principal.
 */
 
static int setup_name( const wchar_t *wuser, char **user, krb5_principal *princ)
{
  char *tmp_user = 0;
  int ret;
  int size;
  
  *user = 0;
  if(my_wctombs( wuser, &tmp_user))
    return -1;
  size = strlen(tmp_user) + strlen(global_def_realm) + 1;
  if((*user = malloc(size)) == 0) {
    free(tmp_user);
    com_err(PWS_ID, ENOMEM, ":setup_name: malloc failed.");
    return -1;
  }
  strcpy(*user, tmp_user);
  free(tmp_user);
  strcat(*user, "@");
  strcat(*user, global_def_realm);

  if(ret = krb5_parse_name( global_context, *user, princ)) {
    free(*user);
    com_err(PWS_ID, ret, ":setup_name: krb5_parse_name failed. Error was %s.",
            error_message(ret));
    return -1;
  }
  return 0;
}

int do_kerbnet_pw_sync(const wchar_t *username, const wchar_t *password, int uid)
{
  krb5_principal pw_princ;
  char *asci_username = 0;
  char *ascii_password = 0;
  int ret;

  /*
   * Map the wchar_t username into an ascii username and a
   * krb5_principal.
   */
  if(setup_name(username, &asci_username, &pw_princ))
    return -1;

  /*
   * Attempt the connect to the kadmin server.
   */
  if(do_kerbnet_kadmin_connect()) {
    free(asci_username);
    krb5_free_principal( global_context, pw_princ);
    return -1;
  }
  
  /*
   * Map the wchar_t password into an ascii password.
   */
  if(my_wctombs( password, &ascii_password)) {
    free(asci_username);
    krb5_free_principal( global_context, pw_princ);
    return -1;
  }

  /*
   * Request the password change.
   */
  if(ret = kadm5_chpass_principal( global_handle, pw_princ, ascii_password)) {
    memset( ascii_password, '\0', strlen(ascii_password));
    free(asci_username);
    free(ascii_password);
    krb5_free_principal( global_context, pw_princ);
    com_err(PWS_ID, ret, ":PasswordSync: kadm5_chpass_principal failed. Error was %s.",
            error_message(ret));
    do_kerbnet_kadmin_disconnect();
    return -1;
  }

  /*
   * Wipe the stored password, cleanup and return.
   */
  memset( ascii_password, '\0', strlen(ascii_password));
  free(asci_username);
  free(ascii_password);
  krb5_free_principal(global_context, pw_princ);
  return 0;
}



