/*
 * Copyright, OpenVision Technologies, Inc., 1996, All Rights Reserved
 *
 * WARNING:  Retrieving the OpenVision Kerberos Administration system
 * source code, as described below, indicates your acceptance of the
 * following terms.  If you do not agree to the following terms, do not
 * retrieve the OpenVision Kerberos administration system.
 *
 * You may freely use and distribute the Source Code and Object Code
 * compiled from it, but this Source Code is provided to you "AS IS"
 * EXCLUSIVE OF ANY WARRANTY, INCLUDING, WITHOUT LIMITATION, ANY
 * WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR
 * ANY OTHER WARRANTY, WHETHER EXPRESS OR IMPLIED.  IN NO EVENT WILL
 * OPENVISION HAVE ANY LIABILITY FOR ANY LOST PROFITS, LOSS OF DATA OR
 * COSTS OF PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES, OR FOR ANY
 * SPECIAL, INDIRECT, OR CONSEQUENTIAL DAMAGES ARISING OUT OF THIS
 * AGREEMENT, INCLUDING, WITHOUT LIMITATION, THOSE RESULTING FROM THE
 * USE OF THE SOURCE CODE, OR THE FAILURE OF THE SOURCE CODE TO PERFORM,
 * OR FOR ANY OTHER REASON.
 *
 * OpenVision retains all rights, title, and interest in the donated
 * Source Code.  With respect to OpenVision's copyrights in the donated
 * Source Code, OpenVision also retains rights to derivative works of
 * the Source Code whether created by OpenVision or a third party.
 *
 * OpenVision Technologies, Inc. has donated this Kerberos
 * Administration system to MIT for inclusion in the standard Kerberos 5
 * distribution. This donation underscores our commitment to continuing
 * Kerberos technology development and our gratitude for the valuable
 * work which has been performed by MIT and the Kerberos community.
 */

/*
 * $Header: /afs/athena.mit.edu/astaff/project/krbdev/.cvsroot/src/lib/kadm5/srv/svr_iters.c,v 1.1 1996/07/24 22:23:35 tlyu Exp $
 */

#if !defined(lint) && !defined(__CODECENTER__)
static char *rcsid = "$Header: /afs/athena.mit.edu/astaff/project/krbdev/.cvsroot/src/lib/kadm5/srv/svr_iters.c,v 1.1 1996/07/24 22:23:35 tlyu Exp $";
#endif

#if defined(HAVE_COMPILE) && defined(HAVE_STEP)
#define SOLARIS_REGEXPS
#elif defined(HAVE_REGCOMP) && defined(HAVE_REGEXEC)
#define POSIX_REGEXPS
#elif defined(HAVE_RE_COMP) && defined(HAVE_RE_EXEC)
#define BSD_REGEXPS
#else
#error I cannot find any regexp functions
#endif

#include	<sys/types.h>
#include	<string.h>
#include	<kadm5/admin.h>
#include	"adb.h"
#include	<dyn.h>
#ifdef SOLARIS_REGEXPS
#include	<regexpr.h>
#endif
#ifdef POSIX_REGEXPS
#include	<regex.h>
#endif
#include <stdlib.h>

#include	"server_internal.h"

struct iter_data {
     krb5_context context;
     DynObject matches;
     char *exp;
#ifdef SOLARIS_REGEXPS
     char *expbuf;
#endif
#ifdef POSIX_REGEXPS
     regex_t preg;
#endif
};

/*
 * Function: glob_to_regexp
 *
 * Arguments:
 *
 *	glob	(r) the shell-style glob (?*[]) to convert
 *	realm	(r) the default realm to append, or NULL
 *	regexp	(w) the ed-style regexp created from glob
 *
 * Effects:
 *
 * regexp is filled in with allocated memory contained a regular
 * expression to be used with re_comp/compile that matches what the
 * shell-style glob would match.  If glob does not contain an "@"
 * character and realm is not NULL, "@<realm>" is appended to the regexp.
 *
 * Conversion algorithm:
 *
 *	quoted characters are copied quoted
 *	? is converted to .
 *	* is converted to .*
 * 	active characters are quoted: ^, $, .
 *	[ and ] are active but supported and have the same meaning, so
 *		they are copied
 *	other characters are copied
 *	regexp is anchored with ^ and $
 */
kadm5_ret_t glob_to_regexp(char *glob, char *realm, char **regexp)
{
     int append_realm;
     char *p;

     /* validate the glob */
     if (glob[strlen(glob)-1] == '\\')
	  return EINVAL;

     /* A character of glob can turn into two in regexp, plus ^ and $ */
     /* and trailing null.  If glob has no @, also allocate space for */
     /* the realm. */
     append_realm = (realm != NULL) && (strchr(glob, '@') == NULL);
     p = (char *) malloc(strlen(glob)*2+ 3 +
			 (append_realm ? (strlen(realm)+1) : 0));
     if (p == NULL)
	  return ENOMEM;
     *regexp = p;

     *p++ = '^';
     while (*glob) {
	  switch (*glob) {
	  case '?':
	       *p++ = '.';
	       break;
	  case '*':
	       *p++ = '.';
	       *p++ = '*';
	       break;
	  case '.':
	  case '^':
	  case '$':
	       *p++ = '\\';
	       *p++ = *glob;
	       break;
	  case '\\':
	       *p++ = '\\';
	       *p++ = ++*glob;
	       break;
	  default:
	       *p++ = *glob;
	       break;
	  }
	  glob++;
     }

     if (append_realm) {
	  *p++ = '@';
	  strcpy(p, realm);
	  p += strlen(realm);
     }

     *p++ = '$';
     *p++ = '\0';
     return KADM5_OK;
}

void get_either_iter(struct iter_data *data, char *name)
{
     if (
#ifdef SOLARIS_REGEXPS
	 (step(name, data->expbuf) != 0)
#endif
#ifdef POSIX_REGEXPS
	 (regexec(&data->preg, name, 0, NULL, 0) == 0)
#endif
#ifdef BSD_REGEXPS
	 (re_exec(name) != 0)
#endif
	 )
     {
	  (void) DynAdd(data->matches, &name);
     } else
	  free(name);
}

void get_pols_iter(void *data, osa_policy_ent_t entry)
{
     char *name;

     if ((name = strdup(entry->name)) == NULL)
	  return;
     get_either_iter(data, name);
}

void get_princs_iter(void *data, krb5_principal princ)
{
     struct iter_data *id = (struct iter_data *) data;
     char *name;
     
     if (krb5_unparse_name(id->context, princ, &name) != 0)
	  return;
     get_either_iter(data, name);
}

kadm5_ret_t kadm5_get_either(int princ,
				       void *server_handle,
				       char *exp,
				       char ***princs,
				       int *count)
{
     struct iter_data data;
     char *msg, *regexp;
     int ret;
     kadm5_server_handle_t handle = server_handle;
     
     *count = 0;
     if (exp == NULL)
	  exp = "*";

     CHECK_HANDLE(server_handle);

     if ((ret = glob_to_regexp(exp, princ ? handle->params.realm : NULL,
			       &regexp)) != KADM5_OK)
	  return ret;

     if (
#ifdef SOLARIS_REGEXPS
	 ((data.expbuf = compile(regexp, NULL, NULL)) == NULL)
#endif
#ifdef POSIX_REGEXPS
	 ((regcomp(&data.preg, regexp, REG_NOSUB)) != 0)
#endif
#ifdef BSD_REGEXPS
	 ((msg = (char *) re_comp(regexp)) != NULL)
#endif
	 )
     {
	  /* XXX syslog msg or regerr(regerrno) */
	  free(regexp);
	  return EINVAL;
     }

     if ((data.matches = DynCreate(sizeof(char *), -4)) == NULL) {
	  free(regexp);
	  return ENOMEM;
     }

     if (princ) {
	  data.context = handle->context;
	  ret = kdb_iter_entry(handle, get_princs_iter, (void *) &data);
     } else {
	  ret = osa_adb_iter_policy(handle->policy_db, get_pols_iter, (void *)&data);
     }
     
     if (ret != OSA_ADB_OK) {
	  free(regexp);
	  DynDestroy(data.matches);
	  return ret;
     }

     (*princs) = (char **) DynArray(data.matches);
     *count = DynSize(data.matches);
     DynRelease(data.matches);
     free(regexp);
     return KADM5_OK;
}

kadm5_ret_t kadm5_get_principals(void *server_handle,
					   char *exp,
					   char ***princs,
					   int *count)
{
     return kadm5_get_either(1, server_handle, exp, princs, count);
}

kadm5_ret_t kadm5_get_policies(void *server_handle,
					   char *exp,
					   char ***pols,
					   int *count)
{
     return kadm5_get_either(0, server_handle, exp, pols, count);
}

void get_bounded_princs_iter(void *data, krb5_principal princ)
{
     struct iter_data *id = (struct iter_data *) data;
     char *name;
     
     if (krb5_unparse_name(id->context, princ, &name) != 0)
	  return;
     
     (void) DynAdd(id->matches, &name);
}

kadm5_ret_t kadm5_get_next_principals(void *server_handle,
				      char *start_princ,
				      char ***princs,
				      int *count)
{
    /* similar to the kadm5_get_either, but pulled out any glob
       matching information */
     struct iter_data data;
     int ret;
     int bound = *count;
     kadm5_server_handle_t handle = server_handle;
     
     *count = 0;

     CHECK_HANDLE(server_handle);

     if ((data.matches = DynCreate(sizeof(char *), -4)) == NULL) {
	  return ENOMEM;
     }

     data.context = handle->context;
     ret = kdb_bounded_iter_entry(handle, get_bounded_princs_iter, 
				  (void *) &data, start_princ, bound);
     
     if (ret != OSA_ADB_OK) {
	  DynDestroy(data.matches);
	  return ret;
     }

     (*princs) = (char **) DynArray(data.matches);
     *count = DynSize(data.matches);
     DynRelease(data.matches);
     return KADM5_OK;
}


