#include <stdlib.h>
#include <netdb.h>
extern int h_errno;
#include <sys/stat.h>
#include <com_err.h>
#include <krb5.h>
#include <tcl.h>
#include "context.h"

static int bad_args (Tcl_Interp *interp, char *argv0, char *usage) {
    Tcl_AppendResult (interp, "wrong # args: should be \"", argv0,
		      usage ? usage : "", "\"", (char *) NULL);
    return TCL_ERROR;
}

static int
canonicalize_hostname (ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
    struct hostent *h;
    char buf[20];

    if (argc != 2) {
	return bad_args (interp, argv[0], " hostname");
    }
    h_errno = 0;
    h = gethostbyname (argv[1]);
    if (h == 0)
	switch (h_errno) {
	case HOST_NOT_FOUND:
	    Tcl_SetResult (interp, "host not found", TCL_STATIC);
	    return TCL_ERROR;
	case TRY_AGAIN:
	    Tcl_SetResult (interp, "temporary lookup failure", TCL_STATIC);
	    return TCL_ERROR;
	case NO_RECOVERY:
	    Tcl_SetResult (interp, "name resolver failure", TCL_STATIC);
	    return TCL_ERROR;
	case NO_DATA:
	    Tcl_SetResult (interp, "no address for hostname", TCL_STATIC);
	    return TCL_ERROR;
	default:
	    sprintf (buf, "%d", h_errno);
	    Tcl_AppendResult (interp, "unknown error ", buf,
			      " returned from resolver", (char *) NULL);
	    return TCL_ERROR;
	}
    Tcl_SetResult (interp, h->h_name, TCL_VOLATILE);
    return TCL_OK;
}

static int
read_password (ClientData clientData, Tcl_Interp *interp,
	       int argc, char *argv[])
{
    krb5_context context;
    char buf[10240];
    int size = sizeof (buf);
    krb5_error_code code;
    char *prompt1, *prompt2;

    context = VERIFY_CTX (clientData)->ctx;

    if (argc == 2) {
	prompt1 = prompt2 = argv[1];
    } else if (argc == 3) {
	prompt1 = argv[1];
	prompt2 = argv[2];
    } else {
	return bad_args (interp, argv[0], " prompt1 ?prompt2?");
    }

    code = krb5_read_password (context, prompt1, prompt2, buf, &size);
    if (code != 0) {
	Tcl_SetResult (interp, (char *) error_message (code), TCL_VOLATILE);
	return TCL_ERROR;
    } else {
	Tcl_AppendResult (interp, buf, (char *) NULL);
	return TCL_OK;
    }
}

/* These could be replaced with calls out to standard UNIX utilities;
   they're here in case we want to do Windoze or Mac versions.  */
static int
have_system_privs (ClientData clientData, Tcl_Interp *interp,
		   int argc, char *argv[])
{
    if (argc != 1) {
	return bad_args (interp, argv[0], "");
    }
    Tcl_SetResult (interp, getuid() == 0 ? "1" : "0", TCL_STATIC);
    return TCL_OK;
}

static int
system_only_write (ClientData clientData, Tcl_Interp *interp,
		   int argc, char *argv[])
{
    /* Deny write access to specified filesystem object to non-system
       processes.  (In UNIX, this means owner is root, non-owner write
       bits are turned off.)  */
    struct stat statbuf;
    char *label = 0;

    if (argc != 2) {
	return bad_args (interp, argv[0], " pathname");
    }
    if (stat (argv[1], &statbuf) < 0) {
    sys_error:
	Tcl_AppendResult (interp, argv[1], ": ", label ? label : "",
			  error_message (errno), TCL_VOLATILE);
	return TCL_ERROR;
    }
    if (chmod (argv[1], statbuf.st_mode & ~022) < 0) {
	label = "chmod: ";
	goto sys_error;
    }
#if 0 /* disable for testing */
    if (chown (argv[1], 0, 0) < 0) {
	label = "chown: ";
	goto sys_error;
    }
#endif
    return TCL_OK;
}

void
krb5tcl_init_misc (krb5tcl_context *tctx)
{
    krb5tcl_add_proc ("krb5tcl_canonicalize_hostname",
		      canonicalize_hostname, tctx);
    krb5tcl_add_proc ("krb5tcl_read_password",
		      read_password, tctx);
    krb5tcl_add_proc ("krb5tcl_system_only_write",
		      system_only_write, tctx);
    krb5tcl_add_proc ("krb5tcl_have_system_privs",
		      have_system_privs, tctx);
}
