#!/usr/bin/perl

package Debian::DictionariesCommon;

use Data::Dumper;
use base qw(Exporter);

# List all exported symbols here.
our @EXPORT_OK = qw(parseinfo updatedb loaddb emacsen_support jed_support
		    getlibdir getsysdefault setsysdefault
		    getuserdefault setuserdefault
		    build_emacsen_support build_jed_support
                    build_pspell_support);
# Import :all to get everything.
our %EXPORT_TAGS = (all => [@EXPORT_OK]);

my $infodir = "/var/lib/dictionaries-common";
my $cachedir = "/var/cache/dictionaries-common";
my $ispelldefault = "ispell-default";
my $sysdefault = "/etc/dictionaries-common/$ispelldefault";
my $userdefault = "$ENV{HOME}/.$ispelldefault";
my $emacsensupport = "emacsen-ispell-dicts.el";
my $jedsupport = "jed-ispell-dicts.sl";
my $pspellmap = "region-to-spelling.map";

sub getlibdir {
  my $class = shift;
  return "$infodir/$class";
}

sub mydie {
  my $routine = shift;
  my $errmsg = shift;
  die __PACKAGE__, "($routine):E: $errmsg";
}

sub parseinfo {
  my $file = shift;
  open (DICT, "< $file");
  my $old_irs=$/; # Save current value for input record separator
  $/ = "";
  my %dictionaries =
    map {
      s/^([^:]+):/lc ($1) . ":"/meg;  # Lower case field names
      my %hash = /^([^:]+):\s*((?<!\n)[^\n]+)\s*$/mg;
      map { delete $hash{$_} if ($hash{$_} =~ /^\s+$/) } keys %hash;
      mydie ('parseinfo',
	     qq{Record in file $file does not have a "Language" entry})
	if not exists $hash{language};
      mydie ('parseinfo',
	     qq{Record in file $file does not have a "Hash-Name" entry})
	if not exists $hash{"hash-name"};
      my $lang = delete $hash{language};
      ($lang, \%hash);
    } <DICT>;
  $/ = $old_irs; # Reset value of input record separator
  return \%dictionaries;
}

sub updatedb {
  my $class = shift;
  opendir (DIR, "$infodir/$class");
  my @infofiles = grep {/^[^\.]/} readdir DIR;
  closedir DIR;
  my %dictionaries = ();
  foreach my $f (@infofiles) {
    next if $f =~ m/.*~$/;                         # Ignore ~ backup files
    my $dicts = parseinfo ("$infodir/$class/$f");
    %dictionaries = (%dictionaries, %$dicts);
  }
  mkdir $cachedir if (! -d $cachedir);
  my $dumper=Data::Dumper->new([\%dictionaries],
                               [qw{*dictionaries}]);
  $dumper->Indent(1);
  open (DB, "> $cachedir/$class.db");
  print DB $dumper->Dump;
  print DB "\n1;\n"; # Return a true value so require works.
  close DB;
}

sub loaddb {
  my $class = shift;
  my $dbfile = "$cachedir/$class.db";
  if (-e $dbfile) {
    do $dbfile;
  }
  return \%dictionaries;
}

sub getdefault {
  $file = shift;
  if (-f $file) {
    my $lang = `cat $file`;
    chomp $lang;
    return $lang;
  }
  else {
    return undef;
  }
}

sub getuserdefault {
  getdefault ($userdefault);
}

sub getsysdefault {
  getdefault ($sysdefault);
}

sub setsysdefault {
  $value = shift;
  open (DEFAULT, "> $sysdefault");
  print DEFAULT $value;
  close DEFAULT;
}

sub setuserdefault {

  my $default = getuserdefault ();

  my $dictionaries = loaddb ("ispell");

  my @choices = sort keys %$dictionaries;

  if (scalar @choices == 0) {
    warn "Sorry, no ispell dictionary is installed in your system.\n";
    return;
  }

  my $initial = -1;
  if (defined $default) {
    for (my $i = 0; $i < scalar @choices; $i++) {
      if ($default eq $choices[$i]) {
	$initial = $i;
	last;
      }
    }
  }

  open (TTY, "/dev/tty");
  while (1) {
    $| = 1;
    print
      "\nSelect your personal ispell dictionary for use with ispell-wrapper\n\n";
    for ($i = 0; $i < scalar @choices; $i++) {
      print "  " . ($i == $initial ? "*" : " ")
	     . " [" . ($i+1) . "] $choices[$i]\n";
    }
    print qq(\nSelect number or "q" for quit)
      . ($initial != -1 ? " (* is the current default): " : ": ");
    my $sel = <TTY>;
    chomp $sel;
    last if $sel eq "q";
    if ($sel < 1 or $sel > scalar @choices) {
      print qq{\nInvalid choice "$sel".\n\n};
      next;
    }
    else {
      $sel--;
      open (DEFAULT, "> $userdefault");
      print DEFAULT $choices[$sel];
      close DEFAULT;
      last;
    }
  }
  close TTY;
}

sub generate_comment {
  my $commstr = shift;
  my $comment = "This file is part of the dictionaries-common package.
It has been automatically generated.
DO NOT EDIT!";
  $comment =~ s{^}{$commstr}mg;
  return "$comment\n";
}

sub build_emacsen_support {

  my $elisp = '';
  my $availability = '';
  my @classes=("aspell","ispell");
  my %entries = ();
  my %aspell_locales = ();
  my %emacsen_ispell = ();
  my %emacsen_aspell = ();
  
  foreach $class ( @classes ){
      my $dictionaries = loaddb ($class);

      foreach $k (keys %$dictionaries) {

	  my $lang = $dictionaries->{$k};
	  next if (exists $lang->{'emacs-display'} 
		   && $lang->{'emacs-display'} eq "no");
	  
	  my $hashname = $lang->{"hash-name"};
	  my $casechars = exists $lang->{casechars} ?
	      $lang->{casechars} : "[a-zA-Z]";
	  my $notcasechars = exists $lang->{"not-casechars"} ?
	      $lang->{"not-casechars"} : "[^a-zA-Z]";
	  my $otherchars = exists $lang->{otherchars} ?
	      $lang->{otherchars} : "[']";
	  my $manyothercharsp = exists $lang->{"many-otherchars"} ?
	      ($lang->{"many-otherchars"} eq "yes" ? "t" : "nil") : "nil";
	  my $ispellargs = exists $lang->{"ispell-args"} ?
	      ('("' . join ('" "', split (/\s+/, $lang->{"ispell-args"}))
	       . '")') : (qq/("-d" "/ . $lang->{"hash-name"} . qq/")/) ;
	  my $extendedcharactermode = exists $lang->{"extended-character-mode"} ?
	      ('"' . $lang->{"extended-character-mode"} . '"') : "nil";
	  my $codingsystem = exists $lang->{"coding-system"} ?
	      $lang->{"coding-system"} : "nil";
	  my $emacsenname = exists $lang->{"emacsen-name"} ?
	      $lang->{"emacsen-name"} : $hashname;

	  if ( $class eq "ispell" ){
	      $emacsen_ispell{$emacsenname}++;
	  } elsif ( $class eq "aspell" ){
	      $emacsen_aspell{$emacsenname}++;	
	      if ( exists $lang->{"aspell-locales"} ){
		  foreach ( split(/\s*,\s*/,$lang->{"aspell-locales"}) ){
		      $aspell_locales{$_}=$emacsenname;
		  }
	      }    
	  }
	  
	  if ( exists $emacsen_ispell{$emacsenname} and $emacsen_aspell{$emacsenname} ){
	      $availability = "all";
	  } elsif ( exists $emacsen_ispell{$emacsenname} ){
	      $availability = "ispell";
	  } elsif ( exists $emacsen_aspell{$emacsenname} ){
	      $availability = "aspell";
	  } else {
	      $availability = "none"; # This should not happen
	  }
	  
	  $entries{$emacsenname} =  qq{
(debian-ispell-add-dictionary-entry
  \'("$emacsenname"
    "$casechars"
    "$notcasechars"
    "$otherchars"
    $manyothercharsp
    $ispellargs
    $extendedcharactermode
    $codingsystem)
  "$availability")};
    }
  }

  open (ELISP, "> $cachedir/$emacsensupport")
    or die "Cannot open emacsen cache file";

  print ELISP generate_comment (";;; ");
  $elisp .= join ("\n", map {"$entries{$_}"} reverse sort keys %entries);

  if ( scalar %aspell_locales ){
      $elisp .= "\n\n;; An assoc list that will try to map locales to emacsen names";
      $elisp .= "\n\n(setq debian-aspell-equivs-alist \'(\n";
      foreach ( sort keys %aspell_locales ){
	  $elisp .= "     (\"$_\" \"$aspell_locales{$_}\")\n";
      }
      $elisp .= "))\n";
  } else {
      $elisp .= "\n\n;; No emacsen-aspell-equivs entries were found\n";
  }

  print ELISP $elisp;
  close ELISP;

}

sub build_jed_support {

  my $dictionaries = loaddb ("ispell");
  my $slang = generate_comment ("%%% ");

  foreach $k (keys %$dictionaries) {

    my $lang = $dictionaries->{$k};
    next if (exists $lang->{'jed-display'} 
	     && $lang->{'jed-display'} eq "no");

    my $hashname = $lang->{"hash-name"};
    my $additionalchars = exists $lang->{additionalchars} ?
      $lang->{additionalchars} : "";
    my $otherchars = exists $lang->{otherchars} ?
      $lang->{otherchars} : "'";
    my $emacsenname = exists $lang->{"emacsen-name"} ?
      $lang->{"emacsen-name"} : $hashname;
    my $extendedcharmode = exists $lang->{"extended-character-mode"} ?
      $lang->{"extended-character-mode"} : "";
    my $ispellargs = exists $lang->{"ispell-args"} ?
      $lang->{"ispell-args"} : "";

    $slang .= qq{
ispell_add_dictionary (
  "$emacsenname",
  "$hashname",
  "$additionalchars",
  "$otherchars",
  "$extendedcharmode",
  "$ispellargs");
};
  }

  open (SLANG, "> $cachedir/$jedsupport")
    or die "Cannot open jed cache file";
  print SLANG $slang;
  close SLANG;
}


sub build_pspell_support {

  my $mapping = generate_comment ("### ");

  # The following entries will be always included in the
  # regions-to-spelling.map file.  These entries are present in the original
  # file provided by the libpspell4 package, and they are probalby necessary
  # to avoid breakage of the aspell package.
  #
  # N.B.: These entries will be superseded by the registered ispell
  # dictionaries following the dictionaries-common Policy.
  my %entries = ("en_us" => "american",
		 "en_gb" => "british",
		 "en_ca" => "canadian");
  
  my @classes=("aspell","ispell");
  
  foreach $class ( @classes ) {
      my $dictionaries = loaddb ($class);
      foreach $k (keys %$dictionaries) {
	  
	  my $lang = $dictionaries->{$k};
	  
	  if ( exists $lang->{"locale"} and exists $lang->{"pspell-$class"} ) {
	      if ( $lang->{"pspell-$class"} =~ /^[^-]+-([^- ]+)/ ) {
		  $entries{lc ($lang->{"locale"})} = $1;
	      }
	  }
      }
  }
  
  $mapping .= join ("\n", map {"$_ $entries{$_}"} keys %entries);

  open (PSPELL, "> $cachedir/$pspellmap")
    or die "Cannot open pspell cache file";
  print PSPELL $mapping;
  close PSPELL;
}

# Ensure we evaluate to true.
1;

__END__

=head1 NAME

Debian::DictionariesCommon.pm - dictionaries-common library

=head1 SYNOPSIS

    use Debian::DictionariesCommon q(:all)
    $dictionaries = parseinfo ('/var/lib/dictionaries-common/ispell/iwolof');
    loaddb ('ispell')
    updatedb ('wordlist')

=head1 DESCRIPTION

(To be written)

=head1 SEE ALSO

(To be written)

=head1 AUTHORS

Rafael Laboissiere

=cut
