/*
 * arch/i386/mm/boot_ioremap.c
 * 
 * Re-map functions for early boot-time before paging_init() when the 
 * boot-time pagetables are still in use
 *
 * Written by Dave Hansen <haveblue@us.ibm.com>
 */


#include <linux/config.h>
#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/tlbflush.h>
#include <linux/init.h>
#include <linux/stddef.h>


/*
 * This is only for a caller who is clever enough to page-align
 * phys_addr and virtual_source, and who also has a preference
 * about which virtual address from which to steal ptes
 */
static void __init __boot_ioremap(unsigned long phys_addr, unsigned int nrpages,
		    char* virtual_source)
{
	pgd_t *pgd;
	pud_t *pud;
	pmd_t *pmd;
	pte_t* pte;
	unsigned int i;
	unsigned long vaddr = (unsigned long)virtual_source;

	pgd = pgd_offset_k(vaddr);
	pud = pud_offset(pgd, vaddr);
	pmd = pmd_offset(pud, vaddr);
	pte = pte_offset_kernel(pmd, vaddr);

	for (i=0; i < nrpages; i++, phys_addr += PAGE_SIZE, pte++) {
		set_pte(pte, pfn_pte(phys_addr>>PAGE_SHIFT, PAGE_KERNEL));
		__flush_tlb_one(&virtual_source[i*PAGE_SIZE]);
	}
}

/* the virtual space we're going to remap comes from this array */
#define BOOT_IOREMAP_PAGES 4
#define BOOT_IOREMAP_SIZE (BOOT_IOREMAP_PAGES*PAGE_SIZE)
static __initdata char boot_ioremap_space[BOOT_IOREMAP_SIZE]
		       __attribute__ ((aligned (PAGE_SIZE)));

/*
 * This only applies to things which need to ioremap before paging_init()
 * bt_ioremap() and plain ioremap() are both useless at this point.
 * 
 * When used, we're still using the boot-time pagetables, which only
 * have 2 PTE pages mapping the first 8MB
 *
 * There is no unmap.  The boot-time PTE pages aren't used after boot.
 * If you really want the space back, just remap it yourself.
 * boot_ioremap(&ioremap_space-PAGE_OFFSET, BOOT_IOREMAP_SIZE)
 */
__init void* boot_ioremap(unsigned long phys_addr, unsigned long size)
{
	unsigned long last_addr, offset;
	unsigned int nrpages;
	
	last_addr = phys_addr + size - 1;

	/* page align the requested address */
	offset = phys_addr & ~PAGE_MASK;
	phys_addr &= PAGE_MASK;
	size = PAGE_ALIGN(last_addr) - phys_addr;
	
	nrpages = size >> PAGE_SHIFT;
	if (nrpages > BOOT_IOREMAP_PAGES)
		return NULL;
	
	__boot_ioremap(phys_addr, nrpages, boot_ioremap_space);

	return &boot_ioremap_space[offset];
}
