/* $Id: fault.c,v 1.59 2002/02/09 19:49:31 davem Exp $
 * arch/sparc64/mm/fault.c: Page fault handlers for the 64-bit Sparc.
 *
 * Copyright (C) 1996 David S. Miller (davem@caip.rutgers.edu)
 * Copyright (C) 1997, 1999 Jakub Jelinek (jj@ultra.linux.cz)
 */

#include <asm/head.h>

#include <linux/string.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/mman.h>
#include <linux/signal.h>
#include <linux/mm.h>
#include <linux/module.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <linux/interrupt.h>
#include <linux/kprobes.h>
#include <linux/slab.h>
#include <linux/pagemap.h>
#include <linux/compiler.h>
#include <linux/binfmts.h>

#include <asm/page.h>
#include <asm/pgtable.h>
#include <asm/openprom.h>
#include <asm/oplib.h>
#include <asm/uaccess.h>
#include <asm/asi.h>
#include <asm/lsu.h>
#include <asm/sections.h>
#include <asm/kdebug.h>

/*
 * To debug kernel to catch accesses to certain virtual/physical addresses.
 * Mode = 0 selects physical watchpoints, mode = 1 selects virtual watchpoints.
 * flags = VM_READ watches memread accesses, flags = VM_WRITE watches memwrite accesses.
 * Caller passes in a 64bit aligned addr, with mask set to the bytes that need to be
 * watched. This is only useful on a single cpu machine for now. After the watchpoint
 * is detected, the process causing it will be killed, thus preventing an infinite loop.
 */
void set_brkpt(unsigned long addr, unsigned char mask, int flags, int mode)
{
	unsigned long lsubits;

	__asm__ __volatile__("ldxa [%%g0] %1, %0"
			     : "=r" (lsubits)
			     : "i" (ASI_LSU_CONTROL));
	lsubits &= ~(LSU_CONTROL_PM | LSU_CONTROL_VM |
		     LSU_CONTROL_PR | LSU_CONTROL_VR |
		     LSU_CONTROL_PW | LSU_CONTROL_VW);

	__asm__ __volatile__("stxa	%0, [%1] %2\n\t"
			     "membar	#Sync"
			     : /* no outputs */
			     : "r" (addr), "r" (mode ? VIRT_WATCHPOINT : PHYS_WATCHPOINT),
			       "i" (ASI_DMMU));

	lsubits |= ((unsigned long)mask << (mode ? 25 : 33));
	if (flags & VM_READ)
		lsubits |= (mode ? LSU_CONTROL_VR : LSU_CONTROL_PR);
	if (flags & VM_WRITE)
		lsubits |= (mode ? LSU_CONTROL_VW : LSU_CONTROL_PW);
	__asm__ __volatile__("stxa %0, [%%g0] %1\n\t"
			     "membar #Sync"
			     : /* no outputs */
			     : "r" (lsubits), "i" (ASI_LSU_CONTROL)
			     : "memory");
}

static void __kprobes unhandled_fault(unsigned long address,
				      struct task_struct *tsk,
				      struct pt_regs *regs)
{
	if ((unsigned long) address < PAGE_SIZE) {
		printk(KERN_ALERT "Unable to handle kernel NULL "
		       "pointer dereference\n");
	} else {
		printk(KERN_ALERT "Unable to handle kernel paging request "
		       "at virtual address %016lx\n", (unsigned long)address);
	}
	printk(KERN_ALERT "tsk->{mm,active_mm}->context = %016lx\n",
	       (tsk->mm ?
		CTX_HWBITS(tsk->mm->context) :
		CTX_HWBITS(tsk->active_mm->context)));
	printk(KERN_ALERT "tsk->{mm,active_mm}->pgd = %016lx\n",
	       (tsk->mm ? (unsigned long) tsk->mm->pgd :
		          (unsigned long) tsk->active_mm->pgd));
	if (notify_die(DIE_GPF, "general protection fault", regs,
		       0, 0, SIGSEGV) == NOTIFY_STOP)
		return;
	die_if_kernel("Oops", regs);
}

static void bad_kernel_pc(struct pt_regs *regs)
{
	unsigned long *ksp;

	printk(KERN_CRIT "OOPS: Bogus kernel PC [%016lx] in fault handler\n",
	       regs->tpc);
	__asm__("mov %%sp, %0" : "=r" (ksp));
	show_stack(current, ksp);
	unhandled_fault(regs->tpc, current, regs);
}

/*
 * We now make sure that mmap_sem is held in all paths that call 
 * this. Additionally, to prevent kswapd from ripping ptes from
 * under us, raise interrupts around the time that we look at the
 * pte, kswapd will have to wait to get his smp ipi response from
 * us. vmtruncate likewise. This saves us having to get pte lock.
 */
static unsigned int get_user_insn(unsigned long tpc)
{
	pgd_t *pgdp = pgd_offset(current->mm, tpc);
	pud_t *pudp;
	pmd_t *pmdp;
	pte_t *ptep, pte;
	unsigned long pa;
	u32 insn = 0;
	unsigned long pstate;

	if (pgd_none(*pgdp))
		goto outret;
	pudp = pud_offset(pgdp, tpc);
	if (pud_none(*pudp))
		goto outret;
	pmdp = pmd_offset(pudp, tpc);
	if (pmd_none(*pmdp))
		goto outret;

	/* This disables preemption for us as well. */
	__asm__ __volatile__("rdpr %%pstate, %0" : "=r" (pstate));
	__asm__ __volatile__("wrpr %0, %1, %%pstate"
				: : "r" (pstate), "i" (PSTATE_IE));
	ptep = pte_offset_map(pmdp, tpc);
	pte = *ptep;
	if (!pte_present(pte))
		goto out;

	pa  = (pte_val(pte) & _PAGE_PADDR);
	pa += (tpc & ~PAGE_MASK);

	/* Use phys bypass so we don't pollute dtlb/dcache. */
	__asm__ __volatile__("lduwa [%1] %2, %0"
			     : "=r" (insn)
			     : "r" (pa), "i" (ASI_PHYS_USE_EC));

out:
	pte_unmap(ptep);
	__asm__ __volatile__("wrpr %0, 0x0, %%pstate" : : "r" (pstate));
outret:
	return insn;
}

extern unsigned long compute_effective_address(struct pt_regs *, unsigned int, unsigned int);

static void do_fault_siginfo(int code, int sig, struct pt_regs *regs,
			     unsigned int insn, int fault_code)
{
	siginfo_t info;

	info.si_code = code;
	info.si_signo = sig;
	info.si_errno = 0;
	if (fault_code & FAULT_CODE_ITLB)
		info.si_addr = (void __user *) regs->tpc;
	else
		info.si_addr = (void __user *)
			compute_effective_address(regs, insn, 0);
	info.si_trapno = 0;
	force_sig_info(sig, &info, current);
}

extern int handle_ldf_stq(u32, struct pt_regs *);
extern int handle_ld_nf(u32, struct pt_regs *);

static unsigned int get_fault_insn(struct pt_regs *regs, unsigned int insn)
{
	if (!insn) {
		if (!regs->tpc || (regs->tpc & 0x3))
			return 0;
		if (regs->tstate & TSTATE_PRIV) {
			insn = *(unsigned int *) regs->tpc;
		} else {
			insn = get_user_insn(regs->tpc);
		}
	}
	return insn;
}

static void do_kernel_fault(struct pt_regs *regs, int si_code, int fault_code,
			    unsigned int insn, unsigned long address)
{
	unsigned char asi = ASI_P;
 
	if ((!insn) && (regs->tstate & TSTATE_PRIV))
		goto cannot_handle;

	/* If user insn could be read (thus insn is zero), that
	 * is fine.  We will just gun down the process with a signal
	 * in that case.
	 */

	if (!(fault_code & (FAULT_CODE_WRITE|FAULT_CODE_ITLB)) &&
	    (insn & 0xc0800000) == 0xc0800000) {
		if (insn & 0x2000)
			asi = (regs->tstate >> 24);
		else
			asi = (insn >> 5);
		if ((asi & 0xf2) == 0x82) {
			if (insn & 0x1000000) {
				handle_ldf_stq(insn, regs);
			} else {
				/* This was a non-faulting load. Just clear the
				 * destination register(s) and continue with the next
				 * instruction. -jj
				 */
				handle_ld_nf(insn, regs);
			}
			return;
		}
	}
		
	/* Is this in ex_table? */
	if (regs->tstate & TSTATE_PRIV) {
		const struct exception_table_entry *entry;

		if (asi == ASI_P && (insn & 0xc0800000) == 0xc0800000) {
			if (insn & 0x2000)
				asi = (regs->tstate >> 24);
			else
				asi = (insn >> 5);
		}
	
		/* Look in asi.h: All _S asis have LS bit set */
		if ((asi & 0x1) &&
		    (entry = search_exception_tables(regs->tpc))) {
			regs->tpc = entry->fixup;
			regs->tnpc = regs->tpc + 4;
			return;
		}
	} else {
		/* The si_code was set to make clear whether
		 * this was a SEGV_MAPERR or SEGV_ACCERR fault.
		 */
		do_fault_siginfo(si_code, SIGSEGV, regs, insn, fault_code);
		return;
	}

cannot_handle:
	unhandled_fault (address, current, regs);
}

#ifdef CONFIG_PAX_PAGEEXEC
#ifdef CONFIG_PAX_EMUPLT
static void pax_emuplt_close(struct vm_area_struct * vma)
{
	vma->vm_mm->call_dl_resolve = 0UL;
}

static struct page* pax_emuplt_nopage(struct vm_area_struct *vma, unsigned long address, int *type)
{
	struct page* page;
	unsigned int *kaddr;

	page = alloc_page(GFP_HIGHUSER);
	if (!page)
		return NOPAGE_OOM;

	kaddr = kmap(page);
	memset(kaddr, 0, PAGE_SIZE);
	kaddr[0] = 0x9DE3BFA8U; /* save */
	flush_dcache_page(page);
	kunmap(page);
	if (type)
		*type = VM_FAULT_MAJOR;
	return page;
}

static struct vm_operations_struct pax_vm_ops = {
	.close = pax_emuplt_close,
	.nopage = pax_emuplt_nopage,
};

static int pax_insert_vma(struct vm_area_struct *vma, unsigned long addr)
{
	int ret;

	memset(vma, 0, sizeof(*vma));
	vma->vm_mm = current->mm;
	vma->vm_start = addr;
	vma->vm_end = addr + PAGE_SIZE;
	vma->vm_flags = VM_READ | VM_EXEC | VM_MAYREAD | VM_MAYEXEC;
	vma->vm_page_prot = protection_map[vma->vm_flags & 0x0f];
	vma->vm_ops = &pax_vm_ops;

	ret = insert_vm_struct(current->mm, vma);
	if (ret)
		return ret;

	++current->mm->total_vm;
	return 0;
}
#endif

/*
 * PaX: decide what to do with offenders (regs->tpc = fault address)
 *
 * returns 1 when task should be killed
 *         2 when patched PLT trampoline was detected
 *         3 when unpatched PLT trampoline was detected
 */
static int pax_handle_fetch_fault(struct pt_regs *regs)
{

#ifdef CONFIG_PAX_EMUPLT
	int err;

	do { /* PaX: patched PLT emulation #1 */
		unsigned int sethi1, sethi2, jmpl;

		err = get_user(sethi1, (unsigned int*)regs->tpc);
		err |= get_user(sethi2, (unsigned int*)(regs->tpc+4));
		err |= get_user(jmpl, (unsigned int*)(regs->tpc+8));

		if (err)
			break;

		if ((sethi1 & 0xFFC00000U) == 0x03000000U &&
		    (sethi2 & 0xFFC00000U) == 0x03000000U &&
		    (jmpl & 0xFFFFE000U) == 0x81C06000U)
		{
			unsigned long addr;

			regs->u_regs[UREG_G1] = (sethi2 & 0x003FFFFFU) << 10;
			addr = regs->u_regs[UREG_G1];
			addr += (((jmpl | 0xFFFFFFFFFFFFE000UL) ^ 0x00001000UL) + 0x00001000UL);
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	} while (0);

	{ /* PaX: patched PLT emulation #2 */
		unsigned int ba;

		err = get_user(ba, (unsigned int*)regs->tpc);

		if (!err && (ba & 0xFFC00000U) == 0x30800000U) {
			unsigned long addr;

			addr = regs->tpc + ((((ba | 0xFFFFFFFFFFC00000UL) ^ 0x00200000UL) + 0x00200000UL) << 2);
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	}

	do { /* PaX: patched PLT emulation #3 */
		unsigned int sethi, jmpl, nop;

		err = get_user(sethi, (unsigned int*)regs->tpc);
		err |= get_user(jmpl, (unsigned int*)(regs->tpc+4));
		err |= get_user(nop, (unsigned int*)(regs->tpc+8));

		if (err)
			break;

		if ((sethi & 0xFFC00000U) == 0x03000000U &&
		    (jmpl & 0xFFFFE000U) == 0x81C06000U &&
		    nop == 0x01000000U)
		{
			unsigned long addr;

			addr = (sethi & 0x003FFFFFU) << 10;
			regs->u_regs[UREG_G1] = addr;
			addr += (((jmpl | 0xFFFFFFFFFFFFE000UL) ^ 0x00001000UL) + 0x00001000UL);
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	} while (0);

	do { /* PaX: patched PLT emulation #4 */
		unsigned int mov1, call, mov2;

		err = get_user(mov1, (unsigned int*)regs->tpc);
		err |= get_user(call, (unsigned int*)(regs->tpc+4));
		err |= get_user(mov2, (unsigned int*)(regs->tpc+8));

		if (err)
			break;

		if (mov1 == 0x8210000FU &&
		    (call & 0xC0000000U) == 0x40000000U &&
		    mov2 == 0x9E100001U)
		{
			unsigned long addr;

			regs->u_regs[UREG_G1] = regs->u_regs[UREG_RETPC];
			addr = regs->tpc + 4 + ((((call | 0xFFFFFFFFC0000000UL) ^ 0x20000000UL) + 0x20000000UL) << 2);
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	} while (0);

	do { /* PaX: patched PLT emulation #5 */
		unsigned int sethi1, sethi2, or1, or2, sllx, jmpl, nop;

		err = get_user(sethi1, (unsigned int*)regs->tpc);
		err |= get_user(sethi2, (unsigned int*)(regs->tpc+4));
		err |= get_user(or1, (unsigned int*)(regs->tpc+8));
		err |= get_user(or2, (unsigned int*)(regs->tpc+12));
		err |= get_user(sllx, (unsigned int*)(regs->tpc+16));
		err |= get_user(jmpl, (unsigned int*)(regs->tpc+20));
		err |= get_user(nop, (unsigned int*)(regs->tpc+24));

		if (err)
			break;

		if ((sethi1 & 0xFFC00000U) == 0x03000000U &&
		    (sethi2 & 0xFFC00000U) == 0x0B000000U &&
		    (or1 & 0xFFFFE000U) == 0x82106000U &&
		    (or2 & 0xFFFFE000U) == 0x8A116000U &&
		    sllx == 0x83287020 &&
		    jmpl == 0x81C04005U &&
		    nop == 0x01000000U)
		{
			unsigned long addr;

			regs->u_regs[UREG_G1] = ((sethi1 & 0x003FFFFFU) << 10) | (or1 & 0x000003FFU);
			regs->u_regs[UREG_G1] <<= 32;
			regs->u_regs[UREG_G5] = ((sethi2 & 0x003FFFFFU) << 10) | (or2 & 0x000003FFU);
			addr = regs->u_regs[UREG_G1] + regs->u_regs[UREG_G5];
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	} while (0);

	do { /* PaX: patched PLT emulation #6 */
		unsigned int sethi1, sethi2, sllx, or,  jmpl, nop;

		err = get_user(sethi1, (unsigned int*)regs->tpc);
		err |= get_user(sethi2, (unsigned int*)(regs->tpc+4));
		err |= get_user(sllx, (unsigned int*)(regs->tpc+8));
		err |= get_user(or, (unsigned int*)(regs->tpc+12));
		err |= get_user(jmpl, (unsigned int*)(regs->tpc+16));
		err |= get_user(nop, (unsigned int*)(regs->tpc+20));

		if (err)
			break;

		if ((sethi1 & 0xFFC00000U) == 0x03000000U &&
		    (sethi2 & 0xFFC00000U) == 0x0B000000U &&
		    sllx == 0x83287020 &&
		    (or & 0xFFFFE000U) == 0x8A116000U &&
		    jmpl == 0x81C04005U &&
		    nop == 0x01000000U)
		{
			unsigned long addr;

			regs->u_regs[UREG_G1] = (sethi1 & 0x003FFFFFU) << 10;
			regs->u_regs[UREG_G1] <<= 32;
			regs->u_regs[UREG_G5] = ((sethi2 & 0x003FFFFFU) << 10) | (or & 0x3FFU);
			addr = regs->u_regs[UREG_G1] + regs->u_regs[UREG_G5];
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	} while (0);

	do { /* PaX: patched PLT emulation #7 */
		unsigned int sethi, ba, nop;

		err = get_user(sethi, (unsigned int*)regs->tpc);
		err |= get_user(ba, (unsigned int*)(regs->tpc+4));
		err |= get_user(nop, (unsigned int*)(regs->tpc+8));

		if (err)
			break;

		if ((sethi & 0xFFC00000U) == 0x03000000U &&
		    (ba & 0xFFF00000U) == 0x30600000U &&
		    nop == 0x01000000U)
		{
			unsigned long addr;

			addr = (sethi & 0x003FFFFFU) << 10;
			regs->u_regs[UREG_G1] = addr;
			addr = regs->tpc + ((((ba | 0xFFFFFFFFFFF80000UL) ^ 0x00040000UL) + 0x00040000UL) << 2);
			regs->tpc = addr;
			regs->tnpc = addr+4;
			return 2;
		}
	} while (0);

	do { /* PaX: unpatched PLT emulation step 1 */
		unsigned int sethi, ba, nop;

		err = get_user(sethi, (unsigned int*)regs->tpc);
		err |= get_user(ba, (unsigned int*)(regs->tpc+4));
		err |= get_user(nop, (unsigned int*)(regs->tpc+8));

		if (err)
			break;

		if ((sethi & 0xFFC00000U) == 0x03000000U &&
		    ((ba & 0xFFC00000U) == 0x30800000U || (ba & 0xFFF80000U) == 0x30680000U) &&
		    nop == 0x01000000U)
		{
			unsigned long addr;
			unsigned int save, call;

			if ((ba & 0xFFC00000U) == 0x30800000U)
				addr = regs->tpc + 4 + ((((ba | 0xFFFFFFFFFFC00000UL) ^ 0x00200000UL) + 0x00200000UL) << 2);
			else
				addr = regs->tpc + 4 + ((((ba | 0xFFFFFFFFFFF80000UL) ^ 0x00040000UL) + 0x00040000UL) << 2);

			err = get_user(save, (unsigned int*)addr);
			err |= get_user(call, (unsigned int*)(addr+4));
			err |= get_user(nop, (unsigned int*)(addr+8));
			if (err)
				break;

			if (save == 0x9DE3BFA8U &&
			    (call & 0xC0000000U) == 0x40000000U &&
			    nop == 0x01000000U)
			{
				struct vm_area_struct *vma;
				unsigned long call_dl_resolve;

				down_read(&current->mm->mmap_sem);
				call_dl_resolve = current->mm->call_dl_resolve;
				up_read(&current->mm->mmap_sem);
				if (likely(call_dl_resolve))
					goto emulate;

				vma = kmem_cache_alloc(vm_area_cachep, SLAB_KERNEL);

				down_write(&current->mm->mmap_sem);
				if (current->mm->call_dl_resolve) {
					call_dl_resolve = current->mm->call_dl_resolve;
					up_write(&current->mm->mmap_sem);
					if (vma) kmem_cache_free(vm_area_cachep, vma);
					goto emulate;
				}

				call_dl_resolve = get_unmapped_area(NULL, 0UL, PAGE_SIZE, 0UL, MAP_PRIVATE);
				if (!vma || (call_dl_resolve & ~PAGE_MASK)) {
					up_write(&current->mm->mmap_sem);
					if (vma) kmem_cache_free(vm_area_cachep, vma);
					return 1;
				}

				if (pax_insert_vma(vma, call_dl_resolve)) {
					up_write(&current->mm->mmap_sem);
					kmem_cache_free(vm_area_cachep, vma);
					return 1;
				}

				current->mm->call_dl_resolve = call_dl_resolve;
				up_write(&current->mm->mmap_sem);

emulate:
				regs->u_regs[UREG_G1] = (sethi & 0x003FFFFFU) << 10;
				regs->tpc = call_dl_resolve;
				regs->tnpc = addr+4;
				return 3;
			}
		}
	} while (0);

	do { /* PaX: unpatched PLT emulation step 2 */
		unsigned int save, call, nop;

		err = get_user(save, (unsigned int*)(regs->tpc-4));
		err |= get_user(call, (unsigned int*)regs->tpc);
		err |= get_user(nop, (unsigned int*)(regs->tpc+4));
		if (err)
			break;

		if (save == 0x9DE3BFA8U &&
		    (call & 0xC0000000U) == 0x40000000U &&
		    nop == 0x01000000U)
		{
			unsigned long dl_resolve = regs->tpc + ((((call | 0xFFFFFFFFC0000000UL) ^ 0x20000000UL) + 0x20000000UL) << 2);

			regs->u_regs[UREG_RETPC] = regs->tpc;
			regs->tpc = dl_resolve;
			regs->tnpc = dl_resolve+4;
			return 3;
		}
	} while (0);
#endif

	return 1;
}

void pax_report_insns(void *pc, void *sp)
{
	unsigned long i;

	printk(KERN_ERR "PAX: bytes at PC: ");
	for (i = 0; i < 5; i++) {
		unsigned int c;
		if (get_user(c, (unsigned int*)pc+i))
			printk("???????? ");
		else
			printk("%08x ", c);
	}
	printk("\n");
}
#endif

asmlinkage void __kprobes do_sparc64_fault(struct pt_regs *regs)
{
	struct mm_struct *mm = current->mm;
	struct vm_area_struct *vma;
	unsigned int insn = 0;
	int si_code, fault_code;
	unsigned long address;

	fault_code = get_thread_fault_code();

	if (notify_die(DIE_PAGE_FAULT, "page_fault", regs,
		       fault_code, 0, SIGSEGV) == NOTIFY_STOP)
		return;

	si_code = SEGV_MAPERR;
	address = current_thread_info()->fault_address;

	if ((fault_code & FAULT_CODE_ITLB) &&
	    (fault_code & FAULT_CODE_DTLB))
		BUG();

	if (regs->tstate & TSTATE_PRIV) {
		unsigned long tpc = regs->tpc;

		/* Sanity check the PC. */
		if ((tpc >= KERNBASE && tpc < (unsigned long) _etext) ||
		    (tpc >= MODULES_VADDR && tpc < MODULES_END)) {
			/* Valid, no problems... */
		} else {
			bad_kernel_pc(regs);
			return;
		}
	}

	/*
	 * If we're in an interrupt or have no user
	 * context, we must not take the fault..
	 */
	if (in_atomic() || !mm)
		goto intr_or_no_mm;

	if (test_thread_flag(TIF_32BIT)) {
		if (!(regs->tstate & TSTATE_PRIV)) {
			regs->tpc &= 0xffffffff;
			regs->tnpc &= 0xffffffff;
		}
		address &= 0xffffffff;
	}

	if (!down_read_trylock(&mm->mmap_sem)) {
		if ((regs->tstate & TSTATE_PRIV) &&
		    !search_exception_tables(regs->tpc)) {
			insn = get_fault_insn(regs, insn);
			goto handle_kernel_fault;
		}
		down_read(&mm->mmap_sem);
	}

	vma = find_vma(mm, address);
	if (!vma)
		goto bad_area;

#ifdef CONFIG_PAX_PAGEEXEC
	/* PaX: detect ITLB misses on non-exec pages */
	if ((mm->pax_flags & MF_PAX_PAGEEXEC) && vma->vm_start <= address &&
	    !(vma->vm_flags & VM_EXEC) && (fault_code & FAULT_CODE_ITLB))
	{
		if (address != regs->tpc)
			goto good_area;

		up_read(&mm->mmap_sem);
		switch (pax_handle_fetch_fault(regs)) {

#ifdef CONFIG_PAX_EMUPLT
		case 2:
		case 3:
			return;
#endif

		}
		pax_report_fault(regs, (void*)regs->tpc, (void*)(regs->u_regs[UREG_FP] + STACK_BIAS));
		do_exit(SIGKILL);
	}
#endif

	/* Pure DTLB misses do not tell us whether the fault causing
	 * load/store/atomic was a write or not, it only says that there
	 * was no match.  So in such a case we (carefully) read the
	 * instruction to try and figure this out.  It's an optimization
	 * so it's ok if we can't do this.
	 *
	 * Special hack, window spill/fill knows the exact fault type.
	 */
	if (((fault_code &
	      (FAULT_CODE_DTLB | FAULT_CODE_WRITE | FAULT_CODE_WINFIXUP)) == FAULT_CODE_DTLB) &&
	    (vma->vm_flags & VM_WRITE) != 0) {
		insn = get_fault_insn(regs, 0);
		if (!insn)
			goto continue_fault;
		if ((insn & 0xc0200000) == 0xc0200000 &&
		    (insn & 0x1780000) != 0x1680000) {
			/* Don't bother updating thread struct value,
			 * because update_mmu_cache only cares which tlb
			 * the access came from.
			 */
			fault_code |= FAULT_CODE_WRITE;
		}
	}
continue_fault:

	if (vma->vm_start <= address)
		goto good_area;
	if (!(vma->vm_flags & VM_GROWSDOWN))
		goto bad_area;
	if (!(fault_code & FAULT_CODE_WRITE)) {
		/* Non-faulting loads shouldn't expand stack. */
		insn = get_fault_insn(regs, insn);
		if ((insn & 0xc0800000) == 0xc0800000) {
			unsigned char asi;

			if (insn & 0x2000)
				asi = (regs->tstate >> 24);
			else
				asi = (insn >> 5);
			if ((asi & 0xf2) == 0x82)
				goto bad_area;
		}
	}
	if (expand_stack(vma, address))
		goto bad_area;
	/*
	 * Ok, we have a good vm_area for this memory access, so
	 * we can handle it..
	 */
good_area:
	si_code = SEGV_ACCERR;

	/* If we took a ITLB miss on a non-executable page, catch
	 * that here.
	 */
	if ((fault_code & FAULT_CODE_ITLB) && !(vma->vm_flags & VM_EXEC)) {
		BUG_ON(address != regs->tpc);
		BUG_ON(regs->tstate & TSTATE_PRIV);
		goto bad_area;
	}

	if (fault_code & FAULT_CODE_WRITE) {
		if (!(vma->vm_flags & VM_WRITE))
			goto bad_area;

		/* Spitfire has an icache which does not snoop
		 * processor stores.  Later processors do...
		 */
		if (tlb_type == spitfire &&
		    (vma->vm_flags & VM_EXEC) != 0 &&
		    vma->vm_file != NULL)
			set_thread_fault_code(fault_code |
					      FAULT_CODE_BLKCOMMIT);
	} else {
		/* Allow reads even for write-only mappings */
		if (!(vma->vm_flags & (VM_READ | VM_EXEC)))
			goto bad_area;
	}

	switch (handle_mm_fault(mm, vma, address, (fault_code & FAULT_CODE_WRITE))) {
	case VM_FAULT_MINOR:
		current->min_flt++;
		break;
	case VM_FAULT_MAJOR:
		current->maj_flt++;
		break;
	case VM_FAULT_SIGBUS:
		goto do_sigbus;
	case VM_FAULT_OOM:
		goto out_of_memory;
	default:
		BUG();
	}

	up_read(&mm->mmap_sem);
	return;

	/*
	 * Something tried to access memory that isn't in our memory map..
	 * Fix it, but check if it's kernel or user first..
	 */
bad_area:
	insn = get_fault_insn(regs, insn);
	up_read(&mm->mmap_sem);

handle_kernel_fault:
	do_kernel_fault(regs, si_code, fault_code, insn, address);
	return;

/*
 * We ran out of memory, or some other thing happened to us that made
 * us unable to handle the page fault gracefully.
 */
out_of_memory:
	insn = get_fault_insn(regs, insn);
	up_read(&mm->mmap_sem);
	printk("VM: killing process %s\n", current->comm);
	if (!(regs->tstate & TSTATE_PRIV))
		do_exit(SIGKILL);
	goto handle_kernel_fault;

intr_or_no_mm:
	insn = get_fault_insn(regs, 0);
	goto handle_kernel_fault;

do_sigbus:
	insn = get_fault_insn(regs, insn);
	up_read(&mm->mmap_sem);

	/*
	 * Send a sigbus, regardless of whether we were in kernel
	 * or user mode.
	 */
	do_fault_siginfo(BUS_ADRERR, SIGBUS, regs, insn, fault_code);

	/* Kernel mode? Handle exceptions or die */
	if (regs->tstate & TSTATE_PRIV)
		goto handle_kernel_fault;
}
