package maslab.camera;

import maslab.util.*;
import maslab.orc.*;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.*;
import javax.swing.*;
import javax.swing.event.*;
import java.io.*;
import javax.imageio.*;
import java.util.*;
import maslab.orcspy.*;

public class CameraViewer
{
    class CaptureSettings
    {
	int width, height, fps;

	public CaptureSettings(int width, int height, int fps)
	{
	    this.width=width;
	    this.height=height;
	    this.fps=fps;
	}

	public String toString()
	{
	    return ""+width+"x"+height+"@"+fps;
	}
    }

    GetOpt opts=new GetOpt();
    Logger log=new Logger(this);

    JFrame frame;
    JLoggerPublisher jloggerpublisher=new JLoggerPublisher();

    Camera cam;
    JImage jimage;

    JCheckBox antiFlickerCheckBox=new JCheckBox();
    JCheckBox backlightCompCheckBox=new JCheckBox();
    JSlider   noiseReductionSlider=new JSlider(JSlider.HORIZONTAL,0,3,0);
    JSlider   qualitySlider=new JSlider(JSlider.HORIZONTAL,0,3,0);
    JCheckBox autoGainCheckBox=new JCheckBox();
    JSlider   gainSlider=new JSlider(JSlider.HORIZONTAL,0,65535,0);
    JCheckBox autoShutterCheckBox=new JCheckBox();
    JSlider   shutterSlider=new JSlider(JSlider.HORIZONTAL, 0, 65535, 65535);
    JCheckBox autoContourCheckBox=new JCheckBox();
    JSlider   contourSlider=new JSlider(JSlider.HORIZONTAL, 0, 65535, 0);
    JComboBox wbModeBox=new JComboBox(new Object[] {"Indoor","Outdoor","Flourescent","Manual","Auto"});
    JSlider   wbRedSlider=new JSlider(JSlider.HORIZONTAL, 0, 65535, 0);
    JSlider   wbBlueSlider=new JSlider(JSlider.HORIZONTAL, 0, 65535, 0);

    // Dynamic range of these LED sliders artifically reduced by factor of 10
    // Who needs blink rates that are minutes long?
    JSlider   ledOnSlider=new JSlider(JSlider.HORIZONTAL,0,2500,0);
    JSlider   ledOffSlider=new JSlider(JSlider.HORIZONTAL,0,2500,0);

    JComboBox resolutionBox=new JComboBox(new Object[] {new CaptureSettings(128,96,15),
							new CaptureSettings(160,120,5),
							new CaptureSettings(160,120,15),
							new CaptureSettings(160,120,30),
							new CaptureSettings(176,144,30),
							new CaptureSettings(320,240,5),
							new CaptureSettings(320,240,15),
							new CaptureSettings(320,240,30),
							new CaptureSettings(352,288,30),
							new CaptureSettings(640,480,5),
							new CaptureSettings(640,480,15)});

    JComboBox colorspaceBox=new JComboBox(new Object[] {"RGB","HSV","RG-Chromaticity (RGV)"});
    JComboBox channelsBox=new JComboBox(new Object[] {"All channels","Channel 0", "Channel 1", "Channel 2", "Channel 3"});

    JCheckBox scaleImageCheckBox=new JCheckBox();
    JLabel    frameRateLabel=new JLabel("FPS: 0");

    BufferedImage camimage=null;

    JButton   snapshotButton=new JButton("Write Image");
    JLabel    colorSampleLabel=new JLabel("");
    Point     colorSamplePoint=new Point(0,0);

    public void doHelp()
    {
        if (opts.getReason()!=null)
            System.out.println("Invocation error: "+opts.getReason());
	
        System.out.println("\nUsage: "+this.getClass().getName()+
			   " [options]\n");
        opts.doHelp();
        System.out.println("\n");
    }


    public static void main(String[] args)
    {
        boolean forcequit=true;
	CameraViewer app;

        try {
	    app=new CameraViewer();
            forcequit=app.run(args);
        } catch (Exception ex) {
	    Logger l=new Logger("Startup");
            l.error("Uncaught exception ",ex);
        }

        if (forcequit)
            System.exit(0);
    }

    public synchronized void updateImage()
    {
	camimage=cam.capture(null);
	switch (colorspaceBox.getSelectedIndex())
	    {
	    case 0: // do nothing. RGB.
		break;
	    case 1:
		Camera.rgbToHsv(camimage);
		break;
	    case 2:
		Camera.rgbToRgv(camimage);
		break;
	    }

	if (channelsBox.getSelectedIndex()!=0)
	    Camera.channelSelect(camimage, channelsBox.getSelectedIndex()-1);
	jimage.setImage(camimage);

	updateColorSample();
    }

    public void updateColorSample()
    {
	if (colorSamplePoint==null)
	    {
		colorSampleLabel.setText("(no sample)");
		return;
	    }

	int rgb;

	// this is an admittedly ungraceful way of handling a race
	// condition: our sample point might be bogus if they just
	// changed the resolution. So we attempt the operation, but
	// fail silently if we're in our error case.
	try {
	    rgb=camimage.getRGB((int) colorSamplePoint.getX(),(int) colorSamplePoint.getY());
	} catch (Exception ex) {
	    colorSampleLabel.setText("(no sample)");
	    return;
	}

	int b=rgb&0x00ff;
	int g=(rgb>>8)&0x00ff;
	int r=(rgb>>16)&0x00ff;
	
	colorSampleLabel.setText(String.format("(%4d,%4d ): %08X %02X %02X %02X",
					       (int) colorSamplePoint.getX(),
					       (int) colorSamplePoint.getY(),
					       rgb,
					       r,g,b));
    }

    public boolean run(String[] args) throws Exception
    {
        opts.addBoolean('h',"help",false,
                        "Show this help");

        if (!opts.parse(args) | opts.getBoolean("help"))
            {
                doHelp();
                return true;
            }

	frame.setVisible(true);
	frame.setMinimumSize(new Dimension(400,300));

	updateImage();
	frame.setSize(frame.getPreferredSize());//cam.getWidth(),cam.getHeight());

	log.output("Started...");

	int nframes=0;
	long starttime=System.currentTimeMillis();
	while(true)
	    {
		updateImage();

		nframes++;
		long thistime=System.currentTimeMillis();
		if (thistime-starttime>1000)
		    {
			double framerate=nframes/((thistime-starttime)/1000.0);
			frameRateLabel.setText("FPS: "+StringUtil.formatDouble(framerate,2));
			nframes=0;
			starttime=thistime;
		    }
	    }
	    
	//return false;
    }

    public CameraViewer()
    {
	frame = new JFrame("CameraViewer");
	BorderLayout bl=new BorderLayout();
	bl.setHgap(5);

	frame.getContentPane().setLayout(bl);

	try {
	    cam = new Camera();
	} catch (IOException ex) {
	    log.error("Couldn't open camera",ex);
	    return;
	}

	jimage = new JImage(cam.getWidth(), cam.getHeight());
	jimage.setScale(true);
	
	JPanel panel=new JPanel(new GridBagLayout());

	GridBagConstraints gbc=new GridBagConstraints();
	gbc.weightx=1;	gbc.weighty=1;
	gbc.anchor=GridBagConstraints.CENTER; gbc.fill=GridBagConstraints.BOTH;

	gbc.gridx=0; gbc.gridy=0; 
	panel.add(new JLabel("Resolution"),gbc);
	gbc.gridx=1;
	panel.add(resolutionBox,gbc);

	gbc.gridx=0; gbc.gridy++;
	panel.add(new JLabel("Anti-Flicker"),gbc);
	gbc.gridx=1;
	panel.add(antiFlickerCheckBox,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Noise Reduction"),gbc);
	gbc.gridx=1;
	panel.add(noiseReductionSlider,gbc);
	noiseReductionSlider.setSnapToTicks(true);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Compression"),gbc);
	gbc.gridx=1;
	panel.add(qualitySlider,gbc);
	qualitySlider.setSnapToTicks(true);

	gbc.gridx=0; gbc.gridy++; gbc.gridwidth=2;
	panel.add(new JSeparator(JSeparator.HORIZONTAL), gbc);
	gbc.gridwidth=1;

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Auto Gain"),gbc);
	gbc.gridx=1;
	panel.add(autoGainCheckBox,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Backlight Comp"),gbc);
	gbc.gridx=1;
	panel.add(backlightCompCheckBox,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Manual Gain"),gbc);
	gbc.gridx=1;
	panel.add(gainSlider,gbc);

	gbc.gridx=0; gbc.gridy++; gbc.gridwidth=2;
	panel.add(new JSeparator(JSeparator.HORIZONTAL), gbc);
	gbc.gridwidth=1;

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Auto Shutter"),gbc);
	gbc.gridx=1;
	panel.add(autoShutterCheckBox,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Manual Shutter"),gbc);
	gbc.gridx=1;
	panel.add(shutterSlider,gbc);

	gbc.gridx=0; gbc.gridy++; gbc.gridwidth=2;
	panel.add(new JSeparator(JSeparator.HORIZONTAL), gbc);
	gbc.gridwidth=1;

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Auto Contour"),gbc);
	gbc.gridx=1;
	panel.add(autoContourCheckBox,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Manual Contour"),gbc);
	gbc.gridx=1;
	panel.add(contourSlider,gbc);

	gbc.gridx=0; gbc.gridy++; gbc.gridwidth=2;
	panel.add(new JSeparator(JSeparator.HORIZONTAL), gbc);
	gbc.gridwidth=1;

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("WhiteBalance Mode"),gbc);
	gbc.gridx=1;
	panel.add(wbModeBox,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Manual WB Red"),gbc);
	gbc.gridx=1;
	panel.add(wbRedSlider,gbc);

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("Manual WB Blue"),gbc);
	gbc.gridx=1;
	panel.add(wbBlueSlider,gbc);
	
	gbc.gridx=0; gbc.gridy++; gbc.gridwidth=2;
	panel.add(new JSeparator(JSeparator.HORIZONTAL), gbc);
	gbc.gridwidth=1;

	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("LED on time"),gbc);
	gbc.gridx=1;
	panel.add(ledOnSlider,gbc);
	
	gbc.gridx=0; gbc.gridy++; 
	panel.add(new JLabel("LED off time"),gbc);
	gbc.gridx=1;
	panel.add(ledOffSlider,gbc);

	frame.getContentPane().add(panel, BorderLayout.EAST);

	JPanel ippanel=new JPanel(new FlowLayout());
	ippanel.add(frameRateLabel);
	ippanel.add(colorspaceBox);
	ippanel.add(channelsBox);
	ippanel.add(new JLabel("Scale"));
	ippanel.add(scaleImageCheckBox);
	scaleImageCheckBox.setSelected(true);

	//	frame.getContentPane().add(ippanel, BorderLayout.NORTH);
	frame.getContentPane().add(jloggerpublisher, BorderLayout.SOUTH);

	colorSampleLabel.setFont(new Font("Monospaced", Font.PLAIN, 12));

	JPanel buttonpanel=new JPanel(new BorderLayout());
	buttonpanel.add(colorSampleLabel,BorderLayout.WEST);
	buttonpanel.add(snapshotButton,BorderLayout.EAST);

	JPanel picpanel=new JPanel(new BorderLayout());
	picpanel.add(jimage, BorderLayout.CENTER);
	picpanel.add(ippanel, BorderLayout.NORTH);
	picpanel.add(buttonpanel, BorderLayout.SOUTH);

	frame.getContentPane().add(picpanel, BorderLayout.CENTER);

	updateControls();

	antiFlickerCheckBox.addActionListener(new FlickerActionListener());
	backlightCompCheckBox.addActionListener(new BacklightActionListener());
	noiseReductionSlider.addChangeListener(new NoiseActionListener());
	qualitySlider.addChangeListener(new QualityActionListener());

	autoGainCheckBox.addActionListener(new GainActionListener());
	gainSlider.addChangeListener(new GainActionListener());

	autoShutterCheckBox.addActionListener(new ShutterActionListener());
	shutterSlider.addChangeListener(new ShutterActionListener());
	
	autoContourCheckBox.addActionListener(new ContourActionListener());
	contourSlider.addChangeListener(new ContourActionListener());

	wbModeBox.addActionListener(new WhiteBalanceActionListener());
	wbRedSlider.addChangeListener(new WhiteBalanceActionListener());
	wbBlueSlider.addChangeListener(new WhiteBalanceActionListener());

	ledOnSlider.addChangeListener(new LEDActionListener());
	ledOffSlider.addChangeListener(new LEDActionListener());

	resolutionBox.addActionListener(new ResolutionListener());
	scaleImageCheckBox.addActionListener(new ScaleActionListener());

	snapshotButton.addActionListener(new SnapshotListener());

	jimage.addMouseMotionListener(new MouseListener());
    }

    public void updateControls()
    {
	int v;
	int width=cam.getWidth();
	int height=cam.getHeight();
	int fps=cam.getFps();
	
	int bestidx=0;
	int bestscore=0;
	for (v=0;v<resolutionBox.getItemCount();v++)
	    {
		CaptureSettings cs=(CaptureSettings) resolutionBox.getItemAt(v);
		if (cs.width==width && cs.height==height && cs.fps==fps)
		    {
			bestidx=v;
			break;
		    }
		if (cs.width==width && cs.height==height)
		    {
			bestidx=v;
		    }
	    }
	resolutionBox.setSelectedIndex(bestidx);

	antiFlickerCheckBox.setSelected(cam.getFlicker());
	backlightCompCheckBox.setSelected(cam.getBacklightCompensation());
	noiseReductionSlider.setValue(cam.getNoiseReduction());
 	qualitySlider.setValue(cam.getQuality());

	v=cam.getGain();
	gainSlider.setValue(v>=0 ? v : 0);
	autoGainCheckBox.setSelected(v==-1);

	/* getShutter not really supported; fake it. */
	//	  v=cam.getShutter();
	v=-1;
	shutterSlider.setValue(v>=0 ? v : 0);
	autoShutterCheckBox.setSelected(v==-1);

	v=cam.getContour();
	contourSlider.setValue(v>=0 ? v : 0);
	autoContourCheckBox.setSelected(v==-1);

	// wrap these in trys in case the driver gives us something
	// odd.
	try {
	    wbModeBox.setSelectedIndex(cam.getWhiteBalanceMode());
	} catch (Exception ex) {}
	try {
	    wbRedSlider.setValue(cam.getWhiteBalanceRed());
	} catch (Exception ex) {}
	try {
	    wbBlueSlider.setValue(cam.getWhiteBalanceBlue());
	} catch (Exception ex) {}
	try {
	    ledOnSlider.setValue(cam.getLedOn());
	} catch (Exception ex) {}
	try {
	    ledOffSlider.setValue(cam.getLedOff());
	} catch (Exception ex) {}
    }

    class ScaleActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    boolean v=scaleImageCheckBox.isSelected();
	    jimage.setScale(v);
	}
    }
    class FlickerActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    boolean v=antiFlickerCheckBox.isSelected();
	    log.output("Setting flicker="+v);
	    cam.setFlicker(v);
	}
    }
    class BacklightActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    boolean v=backlightCompCheckBox.isSelected();
	    log.output("Setting backlight="+v);
	    cam.setBacklightCompensation(v);
	}
    }
    class NoiseActionListener implements ActionListener, ChangeListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v=noiseReductionSlider.getValue();
	    log.output("Setting noise reduction="+v);
	    cam.setNoiseReduction(v);
	}
	public void stateChanged(ChangeEvent e)
	{
	    actionPerformed(null);
	}
    }
    class QualityActionListener implements ActionListener, ChangeListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v=qualitySlider.getValue();
	    log.output("Setting quality="+v);
	    cam.setQuality(v);
	}
	public void stateChanged(ChangeEvent e)
	{
	    actionPerformed(null);
	}
    }
    class GainActionListener implements ActionListener, ChangeListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v=gainSlider.getValue();
	    if (autoGainCheckBox.isSelected())
		v=-1;

	    log.output("Setting gain="+v);
	    cam.setGain(v);
	}
	public void stateChanged(ChangeEvent e)
	{
	    actionPerformed(null);
	}
    }
    class ShutterActionListener implements ActionListener, ChangeListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v=shutterSlider.getValue();

	    if (autoShutterCheckBox.isSelected())
		v=-1;

	    log.output("Setting shutter="+v);
	    cam.setShutter(v);
	}
	public void stateChanged(ChangeEvent e)
	{
	    actionPerformed(null);
	}
    }
    class ContourActionListener implements ActionListener, ChangeListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v=contourSlider.getValue();
	    if (autoContourCheckBox.isSelected())
		v=-1;

	    log.output("Setting contour="+v);
	    cam.setContour(v);
	}
	public void stateChanged(ChangeEvent e)
	{
	    actionPerformed(null);
	}
    }
    class WhiteBalanceActionListener implements ActionListener, ChangeListener
    {
	public void stateChanged(ChangeEvent e)
	{
	    int v;
	    int mode=wbModeBox.getSelectedIndex();

	    log.output("Setting white balance mode="+mode);
	    if (mode!=3) // auto
		cam.setWhiteBalanceMode(mode);
	    else
		{
		    log.output("red="+wbRedSlider.getValue()+" blue="+wbBlueSlider.getValue());
		    cam.setWhiteBalanceManual(wbRedSlider.getValue(), wbBlueSlider.getValue());
		}
	}
	public void actionPerformed(ActionEvent e)
	{
	    stateChanged(null);
	}
    }  
    class LEDActionListener implements ActionListener, ChangeListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v1=ledOnSlider.getValue();
	    int v2=ledOffSlider.getValue();

	    log.output("Setting LED: "+v1+","+v2);
	    cam.setLed(v1, v2);
	}
	public void stateChanged(ChangeEvent e)
	{
	    actionPerformed(null);
	}
    }
    class ResolutionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    int v=resolutionBox.getSelectedIndex();
	    CaptureSettings cs=(CaptureSettings) resolutionBox.getItemAt(v);
	    cam.setCaptureSettings(cs.width,cs.height,cs.fps);
	    updateImage();
	    if (!scaleImageCheckBox.isSelected())
		frame.setSize(frame.getPreferredSize());
	}	

    }
    class SnapshotListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    Calendar cal=Calendar.getInstance();
	    int year=cal.get(Calendar.YEAR);
	    int month=cal.get(Calendar.MONTH);
	    int day=cal.get(Calendar.DAY_OF_MONTH);
	    int hour=cal.get(Calendar.HOUR);
	    int min=cal.get(Calendar.MINUTE);
	    int sec=cal.get(Calendar.SECOND);
	    int ms=cal.get(Calendar.MILLISECOND);

	    BufferedImage im=camimage;
	    String path=String.format("snap%04d%02d%02d_%02d%02d%02d.%03d.png",year,month,day,hour,min,sec,ms);

	    try {
		ImageIO.write(im, "png",new File(path));
		log.output("Image written to: "+path);
	    } catch (IOException ex) {
		log.error("Couldn't write image.",ex);
	    }
	}
    }
    class MouseListener extends MouseMotionAdapter
    {
	public void mouseMoved(MouseEvent e)
	{
	    colorSamplePoint=jimage.componentToImage(e.getPoint());
	    updateColorSample();
	}
    }
}
