package maslab.laser;

import java.awt.*;
import java.io.*;
import javax.swing.*;
import javax.swing.event.*;

import java.util.*;
import maslab.geom.*;

public class LaserView extends JFrame implements LaserListener
{
    static final long serialVersionUID=1001;

    LaserScanner ls;
    LaserCanvas canvas;
    JPanel controls;
    JSlider scaleSlider;
    JLabel scaleLabel;
    double scale=4.0;
    JLabel hertzLabel;
    int hertzCount=Integer.MAX_VALUE;
    long hertzLastMillis=0;

    public LaserView(LaserScanner ls)
    {
	this.ls=ls;
	canvas=new LaserCanvas();

	getContentPane().setLayout(new BorderLayout());
	getContentPane().add(canvas, BorderLayout.CENTER);
	
	controls=new JPanel();
	getContentPane().add(controls, BorderLayout.SOUTH);

	scaleSlider=new JSlider(1,32,4);
	scaleSlider.setPaintLabels(true);
	scaleSlider.addChangeListener(new ScaleSliderChangeListener());
	controls.add(scaleSlider);

	scaleLabel=new JLabel("Scale: 4m");
	controls.add(scaleLabel);

	hertzLabel=new JLabel("0 Hz");
	controls.add(hertzLabel);

	setSize(400,300);
	setVisible(true);

	ls.addLaserListener(this);
    }

    class ScaleSliderChangeListener implements ChangeListener
    {
	public void stateChanged(ChangeEvent e)
	{
	    scale=scaleSlider.getValue();
	    //	    scale=Math.pow(2,scaleSlider.getValue());
	    scaleLabel.setText("Scale: "+((int) scale)+"m");
	}

    }

    public void processLaser(LaserScanData scan)
    {
	canvas.setData(scan);
	canvas.repaint();

	hertzCount++;
	long currentTime=System.currentTimeMillis();
	long elapsedTime=currentTime-hertzLastMillis;

	if (hertzCount>60 || elapsedTime>2000)
	    {
		double fps=(1000.0*hertzCount)/elapsedTime;
		fps=((int) (10.0*fps))/10.0;

		hertzLabel.setText(""+fps+" Hz");

		hertzCount=0;
		hertzLastMillis=currentTime;
	    }

    }

    public class LaserCanvas extends JPanel
    {
	LaserScanData scan;
	double fieldOfView=Math.PI;

	static final long serialVersionUID=1001;

	public synchronized void paint(Graphics g)
	{
	    int width=getWidth();
	    int height=getHeight();

	    g.setColor(Color.black);
	    g.fillRect(0,0,width,height);

	    if (scan==null)
		return;

	    g.drawOval(10,10,20,20);
	    double theta=Math.PI/2;

	    int cx=width/2;
	    int cy=height-10;

	    double maxradius=Math.min(width/2,height); // in pixels
	    double dtheta=fieldOfView/(scan.ranges.length-1);

	    double radiusscale=maxradius/scale;

	    int lastx=0,lasty=0;
	    boolean firsttime=true;

	    g.setFont(new Font("Helvetica",Font.PLAIN,10));

	    int maxscale=(int) (Math.sqrt(Math.pow(width/2,2)
					  + Math.pow(height,2))/radiusscale);

	    for (int i=0;i<=maxscale;i++)
		{
		    int radius=(int) (radiusscale*i);
		    if (i%5==0)
			g.setColor(Color.blue);
		    else
			g.setColor(Color.darkGray);
		    g.drawOval(cx-radius,cy-radius,2*radius,2*radius);

		    g.setColor(Color.blue);
		    g.drawString(""+((int) i),cx+radius,cy);
		    g.drawString(""+((int) i),cx-radius,cy);
		}

	    g.setColor(Color.yellow);
	    for (int i=0;i<scan.points.size();i++)
		{
		    // ignore out-of-range values
		    /*
		    if (ranges[i]>scale*2)
			continue;
		    */

		    GPoint2D p=(GPoint2D) scan.points.get(i);
		    double rot=Math.PI/2;

		    p=p.transform(0,0,rot);

		    int x=cx+(int) (radiusscale*p.getX());
		    int y=cy-(int) (radiusscale*p.getY());

		    if (firsttime)
			firsttime=false;
		    else
			g.drawLine(x, y, lastx, lasty);

		    lastx=x;
		    lasty=y;

		    theta+=dtheta;
		}
	}

	public void setData(LaserScanData scan)
	{
	    this.scan=scan;
	    invalidate();
	}
    }

    public static void main(String[] args)
    {
	int medianfilter=3;

	for (int i=0;i<args.length;i++)
	    {
		if (args[i].startsWith("--filter="))
		    medianfilter=Integer.parseInt(args[i].substring(9));
		if (args[i].equals("--help") || args[i].equals("-h"))
		    {
			doHelp();
			return;
		    }
	    }

	System.out.println("Using filter="+medianfilter);
	
	try {
	    Sick s=new Sick();
	    LaserView lv=new LaserView(new LaserFilter(s,medianfilter));
	} catch (Exception ex) {
	    System.out.println("Couldn't open laserview, "+ex);
	}
    }

    protected static void doHelp()
    {
	System.out.println(" --help          Show this");
	System.out.println(" --filter=n      Use a temporal low pass filter of order n (1,3,5)");
    }

}

