package maslab.odom;

import maslab.geom.*;
import maslab.util.*;
import maslab.orc.*;

public class OdomIntegrator
{
    public double leftScaleFactor=-1.0;
    public double rightScaleFactor=1.0;
    public boolean switchLeftRight=false;

    // For Splinter II
    public double metersPerTick=0.0001717/4;
    public double baselineMeters=0.4475; //0.451;

    public double theta=0;
    public double x=0;
    public double y=0;

    public int maximumTickValue=1000000;

    public Logger log=new Logger(this);

    public long lastLeft=-1, lastRight=-1;

    public OdomIntegrator()
    {
    }

    /** handles the wrap-around business */
    public OdomData update(Orc orc)
    {
	int left=orc.quadphaseRead(16);
	int right=orc.quadphaseRead(19);

	// first call?
	if (lastLeft<0)
	    {
		lastLeft=left;
		lastRight=right;
		return new OdomData(System.currentTimeMillis()/1000.0, 0, 0);
	    }

	int dleft=(int) Orc.diff16(left, lastLeft);
	int dright=(int) Orc.diff16(right, lastRight);
	lastLeft=left;
	lastRight=right;

	OdomData od=new OdomData(System.currentTimeMillis()/1000.0, dleft, dright);

	update(od);

	return od;
    }

    public void update(OdomData od)
    {
	update(od.leftTicks, od.rightTicks);
    }

    public void update(int leftTicks, int rightTicks)
    {
	if (Math.abs(leftTicks)>maximumTickValue || Math.abs(rightTicks)>maximumTickValue)
	    {
		log.warn("Warning: dropping suspicious odometry data: "+leftTicks+" "+rightTicks);
		return;
	    }

	double left=leftTicks*leftScaleFactor;
	double right=rightTicks*rightScaleFactor;

	if (switchLeftRight)
	    {
		double temp=right;
		right=left;
		left=temp;
	    }

	double dL=metersPerTick*(left+right)/2.0;
	double dT=metersPerTick*(right-left)/baselineMeters;
	
	theta+=dT;
	x+=dL*Math.cos(theta);
	y+=dL*Math.sin(theta);
    }

    public OdomState getState()
    {
	return new OdomState(x, y, theta);
    }

}

