package maslab.telemetry.channel;

import maslab.telemetry.*;
import maslab.util.*;

import java.io.*;
import java.awt.*;
import java.util.zip.*;


/** Channel for sending vector graphics to the BotClient. **/
public class VectorChannel extends Channel
{
    public static Logger log = new Logger("VectorChannel");

    public static final int LINE = 0;
    public static final int DOT = 1;
    public static final int DELETE = 3;
    public static final int CIRCLE = 4;

    ByteArrayOutputStream baos;
    DataOutputStream dos;
    Color currentColor;

    /** Create a new VectorChannel 
	@param name The channel name. **/
    public VectorChannel(String name)
    {
	assertInitialized();
	nametype = name + "$" + "vectorgraphics";
	jc.advertise(nametype);
	baos = new ByteArrayOutputStream();
	dos = new DataOutputStream(baos);
	currentColor = new Color(0,255,0);
    }

    /** Draw a line.  
	@param x1 First x coordinate.
	@param y1 First y coordinate.
	@param x2 Second x coordinate. 
	@param y2 Second y coordinate
	@return A handle that may be used to delete the line later.
    **/
    public synchronized int drawLine(int x1, int y1, int x2, int y2)
    {
	try{
	    dos.writeInt(LINE);
	    writeColor(currentColor,dos);
	    dos.writeInt(x1);
	    dos.writeInt(y1);
	    dos.writeInt(x2);
	    dos.writeInt(y2);
	    jc.publish(nametype,baos.toByteArray());
	}
	catch(IOException ioe){
	    log.log(Logger.WARN,"Unexpected IOException in drawLine",ioe);
	}
	finally{
	    baos.reset();
	}
	CRC32 crc = new CRC32();
	crc.update(currentColor.getRGB());
	crc.update(x1);
	crc.update(y1);
	crc.update(x2);
	crc.update(y2);
	return (int)crc.getValue();
    }

    /** Set the current drawing color.  Subsequent objects will be
     * drawn with this color.
     @param c A color.
    **/
    public synchronized void setColor(Color c)
    {
	currentColor = c;
    }


    /** Draw a circle.
     @param x Center x coordinate. 
     @param y Center y coordinate. 
     @param r Radius. 
     @return A handle that may be used to delete this circle later. **/
    public synchronized int drawCircle(int x, int y, int r)
    {
	try{
	    dos.writeInt(CIRCLE);
	    writeColor(currentColor,dos);
	    dos.writeInt(x);
	    dos.writeInt(y);
	    dos.writeInt(r);
	    jc.publish(nametype,baos.toByteArray());
	}
	catch(IOException ioe){
	    log.log(Logger.WARN,"Unexpected IOException in drawLine",ioe);
	}
	finally{
	    baos.reset();
	}
	CRC32 crc = new CRC32();
	crc.update(currentColor.getRGB());
	crc.update(x);
	crc.update(y);
	crc.update(r);
	return (int)crc.getValue();
    }

    void writeColor(Color c, DataOutputStream dos) throws IOException
    {
	dos.writeInt(c.getRed());
	dos.writeInt(c.getGreen());
	dos.writeInt(c.getBlue());
    }

    /** Delete a previously drawn object. 
     @param handle The handle returned from a previous drawing
     operation. **/
    public synchronized void delete(int handle)
    {
	try{
	    dos.writeInt(DELETE);
	    dos.writeInt(handle);
	    jc.publish(nametype,baos.toByteArray());
	}
	catch(IOException ioe){
	    log.log(Logger.WARN,"Unexpected IOException in delete",ioe);
	}
	finally{
	    baos.reset();
	}
    }
}
