package maslab.geom;
import Jama.*; 

import java.io.*;

/** A 2D line **/
public class GLine2D implements Cloneable, Serializable
{
    protected static GLine2D YAXIS=new GLine2D(0,1,new GPoint2D(0,0));
    protected static GLine2D XAXIS=new GLine2D(1,0,new GPoint2D(0,0));

    final static long serialVersionUID=1001;

    /** A vector representing the slope of the line **/
    protected double dx, dy;
    
    /** A point that the line passes through **/
    GPoint2D p;

    // is the slope a vector of length 1?
    boolean normalizedSlope=false;

    // is the point p the point that is on a perpendicular line from the origin?
    boolean normalizedp=false;

    public GLine2D()
    {
	dx=0;
	dy=1;
	p=new GPoint2D(0,0);
    }

    /** Hint that the object should be optimized for future computation by putting
	the internal representation into a more convenient format. This never affects
	correctness, only performance.
    **/
    public void optimize()
    {
	normalizeSlope();
	normalizeP();
    }

    protected void normalizeSlope()
    {
	if (normalizedSlope)
	    return;

	double mag=Math.sqrt(dx*dx+dy*dy);
	dx/=mag;
	dy/=mag;
	normalizedSlope=true;
    }

    /** Create a new line
     * @param dx A change in X corresponding to dy
     * @param dy A change in Y corresponding to dx
     * @param p A point that the line passes through
     **/
    public GLine2D(double dx, double dy, GPoint2D p)
    {
	this.dx=dx;
	this.dy=dy;
	this.p=p;
    }

    /** Create a new line
     * @param M the slope
     * @param b the y intercept
     **/
    public GLine2D(double M, double b)
    {
	this.dx=1;
	this.dy=M;
	this.p=new GPoint2D(0,b);
    }

    /** Create a new line through two points. **/
    public GLine2D(GPoint2D p1, GPoint2D p2)
    {
	dx=p2.x-p1.x;
	dy=p2.y-p1.y;

	this.p=p1;
    }

    /** Get the slope of the line **/
    public double getM()
    {
	return dy/dx;
    }

    /** Get the y intercept of the line **/
    public double getB()
    {
	GPoint2D p=intersectionWith(YAXIS);
	return p.y;
    }

    /** Get the coordinate of a point (on this line), with 0
     * corresponding to the point on the line that is perpendicular to
     * a line passing through the origin and the line.  This allows
     * easy computation if one point is between two other points on
     * the line: compute the line coordinate of all three points and
     * test if a<=b<=c. This is implemented by computing the dot product
     * of the vector p with the line's direction unit vector.
     **/
    public double getLineCoordinate(GPoint2D p)
    {
	normalizeSlope();
	return (p.x*dx+p.y*dy);
    }

    /** The inverse of getLineCoordinate. **/
    public GPoint2D getPointOfCoordinate(double coord)
    {
	Matrix R=getR();

	GPoint2D p=new GPoint2D();

	normalizeSlope();
	p.x=R.get(0,0) + coord*dx;
	p.y=R.get(1,0) + coord*dy;

	return p;
    }

    /** Create a line from the vector from the origin to the line that
     * is perpendicular to the line. 
     * @param R a 2x1 matrix [x y]'
     **/
    public static GLine2D fromRmatrix(Matrix R)
    {
	GPoint2D p=new GPoint2D(R.get(0,0),R.get(1,0));
	double dx=-R.get(1,0);
	double dy=R.get(0,0);

	return new GLine2D(dx, dy, p);
    }

    /** Create a new line given a distance and angle from the origin
     * that is perpendicular to the line.
     **/
    public static GLine2D fromRTheta(double r, double theta)
    {
	double x=r*Math.cos(theta);
	double y=r*Math.sin(theta);

	double M=-1/(y/x);
	double b=y-M*x;

	return new GLine2D(M,b);
    }

    /** Create a line that is at angle theta from the x axis and passes
     * through point p
     **/
    public static GLine2D fromThetaPoint(double theta, GPoint2D p)
    {
	return new GLine2D(Math.cos(theta), Math.sin(theta), p);
    }

    /** A line perpendicular to this line.
     **/
    public GLine2D perpendicularLine()
    {
	return new GLine2D(-dy, dx, p);
    }

    protected void normalizeP()
    {
	if (normalizedp)
	    return;

	normalizeSlope();
	
	// we already have a point (P) on the line, and we know the line vector U and it's perpendicular vector V:
	// so, P'=P.*V *V
	double dotprod=-dy*p.x + dx*p.y;
	
	p=new GPoint2D(-dy*dotprod, dx*dotprod);
	/*	GLine2D l=perpendicularLineThrough(GPoint2D.ORIGIN);
	p=intersectionWith(l);
	assert(p!=null);
	*/
	normalizedp=true;

    }

    /** The 2x1 vector from the origin to the line that is
     * perpendicular to the line.
     **/
    public Matrix getR()
    {
	normalizeP();

	Matrix m=new Matrix(2,1);
	m.set(0,0,p.x);
	m.set(1,0,p.y);

	return m;
    }

    /** The 2x1 unit vector corresponding to the slope of the line. 
     **/
    public Matrix getU()
    {
	Matrix m=new Matrix(2,1);
	m.set(0,0,dx);
	m.set(1,0,dy);

	return m.times(1/m.normF());
    }
    
    /** The line perpendicular to this line that passes through point
     * p
     **/
    public GLine2D perpendicularLineThrough(GPoint2D pin)
    {
	return new GLine2D(-dy, dx, pin);
    }

    /** Return a line parallel to this line that passes through the
     * specified point.
     **/
    public GLine2D parallelLineThrough(GPoint2D pin)
    {
	return new GLine2D(dx, dy, pin);
    }

    /** Compute the point where two lines intersect, or null if the lines are parallel. **/
    public GPoint2D intersectionWith(GLine2D l)
    {
	// this implementation is many times faster than the original,
	// mostly due to avoiding a general-purpose LU decomposition in
	// Matrix.inverse().
	double m00, m01, m10, m11;
	double i00, i01, i10, i11;
	double b00, b10;

	m00=this.dx;
	m01=-l.dx;
	m10=this.dy;
	m11=-l.dy;

	// determinant of m
	double det=m00*m11-m01*m10;

	// parallel lines?
	if (Math.abs(det)<0.0000000001)
	    return null;

	// inverse of m
	i00=m11/det;
	i11=m00/det;
	i01=-m01/det;
	i10=-m10/det;

	b00=l.p.getX()-p.getX();
	b10=l.p.getY()-p.getY();

	double x00, x10;
	x00=i00*b00+i01*b10;
	//	x10=i10*b00+i11*b10;

	return new GPoint2D(dx*x00+p.getX(), dy*x00+p.getY());

	/*
	  // This is the slow, original, but easier-to read/understand
	  // implementation.

	  Matrix A=new Matrix(2,2);
	  A.set(0,0,this.dx);
	  A.set(0,1,-l.dx);
	  A.set(1,0,this.dy);
	  A.set(1,1,-l.dy);
	  
	  if (Math.abs(A.get(0,1)*A.get(1,0)-A.get(0,0)*A.get(1,1))<0.00000000001)
	  return null;
	  
	  Matrix b=new Matrix(2,1);
	  b.set(0,0,l.p.getX()-p.getX());
	  b.set(1,0,l.p.getY()-p.getY());
	  
	  Matrix x=A.inverse().times(b);
	  
	  return new GPoint2D(dx*x.get(0,0)+p.getX(),dy*x.get(0,0)+p.getY());
	*/
    }

    public GPoint2D pointOnLineClosestTo(GPoint2D pin)
    {
	normalizeSlope();
	normalizeP();

	double dotprod=pin.x*dx + pin.y*dy;

	return new GPoint2D(p.x+dx*dotprod, p.y+dy*dotprod);
	/*

	GLine2D pLine=perpendicularLineThrough(pin);
	if (pLine==null)
	    return p;

	GPoint2D pIntersect=intersectionWith(pLine);
	if (pIntersect==null)
	    return new GPoint2D(Double.MAX_VALUE, Double.MAX_VALUE);
	return pIntersect;
	*/
    }

    /** Compute the perpendicular distance between a point and the
     * line
     **/
    public double perpendicularDistanceTo(GPoint2D pin)
    {
	GPoint2D pClosest=pointOnLineClosestTo(pin);
	return pClosest.distanceTo(pin);
    }

    public String toString()
    {
	return "{Line through "+p+", ["+dx+","+dy+"]}";
    }

    /** Self tests **/
    public static void main(String[] args)
    {
	GLine2D l=new GLine2D(4, 10); // y=4x+10
	GLine2D lperp=l.perpendicularLineThrough(GPoint2D.ORIGIN);

	double EPSILON=0.00001;

	assert(Math.abs(lperp.getM()+0.25)<EPSILON);
	assert(Math.abs(lperp.getB())<EPSILON);

	System.out.println(""+l.p+" "+lperp+" "+l.perpendicularDistanceTo(new GPoint2D(4,9)));

	assert(Math.abs(l.perpendicularDistanceTo(new GPoint2D(4,9))-4.1231056256)<EPSILON);

	double c=l.getLineCoordinate(GPoint2D.ORIGIN);
	assert(Math.abs(c)<EPSILON);

	c=l.getLineCoordinate(new GPoint2D(2,18));
	GPoint2D p=l.getPointOfCoordinate(c);
	assert(Math.abs(p.x-2)<EPSILON);
	assert(Math.abs(p.y-18)<EPSILON);

	System.out.println("\nAll tests passed. The next assertion should fail.\n");
	assert(false);
    }
}
