package maslab.laser;

import java.util.*;

public class LaserFilter implements LaserScanner, LaserListener
{
    LaserScanner ls; // our source of data
    int samples; // number of scans we'll remember
    Vector<double[]> data; // where we store our data
    int incount; // number of scans we currently are remembering
    int points; // points in each laser scan
    double[] timeStamps;

    ArrayList<LaserListener> listeners=new ArrayList<LaserListener>();

    public LaserFilter(LaserScanner ls, int samples)
    {
	this.ls=ls;
	this.samples=samples;

	data=new Vector<double[]>(samples);
	timeStamps=new double[samples];

	incount=0;

	for (int i=0;i<samples;i++)
	    data.add(new double[1]);

	ls.addLaserListener(this);
    }

    // this gets called by our external laser provider.
    public void processLaser(LaserScanData scan)
    {
	if (scan==null)
	    {
		System.out.println("got null ranges!");
		return;
	    }

	// how many points in this scan?
	if (incount==0)
	    points=scan.ranges.length;

	// make sure each scan has the same # of points
	if (points!=scan.ranges.length) 
	    {
		// if it doesn't, start over.
		points=scan.ranges.length;
		incount=0;
	    }
		   
	// remember this scan.
	timeStamps[incount]=scan.timeStamp;
	data.set(incount++, scan.ranges);

	if (incount<samples)
	    return;

	// prepare for median filtering.
	double[] filteredranges=new double[scan.ranges.length];
	double[] t=new double[samples];

	for (int i=0;i<points;i++) // e.g. 180 points in a scan
	    {
		for (int j=0;j<samples;j++) // e.g. 5 sample median filter
		    {
			double[] d=data.get(j);
			t[j]=d[i];
		    }

		Arrays.sort(t);
		filteredranges[i]=t[samples/2];
	    }

	LaserScanData filteredScan=new LaserScanData(timeStamps[samples/2], filteredranges);
	notifyListeners(filteredScan);

	// start over
	incount=0;
    }

    protected synchronized void notifyListeners(LaserScanData scan)
    {
	for (LaserListener ll : listeners)
	    ll.processLaser(scan);
    }

    public synchronized void addLaserListener(LaserListener ll)
    {
	listeners.add(ll);
    }

    public synchronized void removeLaserListener(LaserListener ll)
    {
	if (listeners.contains(ll))
	    listeners.remove(ll);
    }

}
