package maslab.telemetry.channel;

import maslab.telemetry.*;
import java.awt.image.*;
import java.io.*;
import maslab.util.*;

/** Channel for publishing images to the BotClient.  **/
public class ImageChannel extends Channel
{
    ByteArrayOutputStream imageBuffer;
    int downsampling = 1;
    boolean compression = false;
    Logger log=new Logger(this);

    protected void init(String name, boolean compression, int downsampling)
    {
	assertInitialized();
	this.downsampling = downsampling;

	//Compression is disabled pending imageio support
	this.compression = false;

	if (compression)
	    {
		nametype = name + "$" + "jpegimage";
	    }
	else
	    {
		nametype = name + "$" + "rawimage";
	    }

	jc.advertise(nametype);
	jc.setMaximumQueueSize(nametype,32768);
	imageBuffer = new ByteArrayOutputStream();
    }
    
    /** Create a new ImageChannel with compression off and
     downsampling=1.
     @param name The channel name. **/
    public ImageChannel(String name)
    {
	init(name,false,1);
    }


    /** Create a new ImageChannel. 
	@param name The channel name.
	@param compression If true, JPEG compression will be used.  If
	false, no compression will be used.  JPEG compression can be
	computationally expensive, but reduces bandwidth usage.
	Actually I'm lying, JPEG compression is completely disabled to
	make this code gcj-compatible.
     **/
    public ImageChannel(String name, boolean compression)
    {
	init(name,compression,1);
    }

    /** Create a new ImageChannel. 
	@param name The channel name.
	@param compression If true, JPEG compression will be used.  If
	false, no compression will be used.  JPEG compression can be
	computationally expensive, but reduces bandwidth usage.
	Actually I'm lying, JPEG compression is completely disabled to
	make this code gcj-compatible.  
	@param downsampling Amount of downsampling to perform.  1
	means no downsampling.  See maslab.util.ImageUtil.downSample()
	for more info.
     **/
    public ImageChannel(String name, boolean compression, int downsampling)
    {
	init(name,compression,downsampling);
    }


    /** Send an image to the BotClient, if any BotClient is connected
     * and subscribing to this channel. 
     @param img The image to send. **/
    public synchronized void publish(BufferedImage img)
    {
	//if (compression)
	//    publishJPEG(img);
	//else
	if (!jc.publishing(nametype))
	    return;

	if (downsampling != 1)
	    img = ImageUtil.downSample(img,downsampling);

	img=ImageUtil.convertImage(img, BufferedImage.TYPE_INT_RGB);
	publishRaw(img);
    }


    /*
    void publishJPEG(BufferedImage img)
    {
	if (!jc.publishing(nametype))
	    return;

	try
	    {
		ImageIO.write(img,"jpg",imageBuffer);
	    }
	catch(IOException foo)
	    {
		return;
	    }

	jc.publish(nametype,imageBuffer.toByteArray());
	imageBuffer.reset();
    }
    */

    void publishRaw(BufferedImage img)
    {
	DataBuffer db=img.getRaster().getDataBuffer();

	if (db instanceof DataBufferInt)
	    publishRawInt(img, (DataBufferInt) db);
	else if (db instanceof DataBufferByte)
	    publishRawByte(img, (DataBufferByte) db);
	else
	    log.error("Unknown image type!");
    }

    void publishRawInt(BufferedImage img, DataBufferInt dbuf)
    {
	int buf[] = dbuf.getData();
	ByteArrayOutputStream baos = new ByteArrayOutputStream(128+4*buf.length);
	DataOutputStream dos = new DataOutputStream(baos);

	try{
	    dos.writeInt(img.getWidth());
	    dos.writeInt(img.getHeight());
	    dos.writeInt(3);
	    dos.writeInt(downsampling);

	    for (int i=0;i<buf.length;i++)
		{
		    dos.writeByte(buf[i]&0xff);
		    dos.writeByte((buf[i]>>8)&0xff);
		    dos.writeByte((buf[i]>>16)&0xff);
		}
	}
	catch(IOException ioe){
	    return;
	}

	jc.publish(nametype,baos.toByteArray());
    }

    void publishRawByte(BufferedImage img, DataBufferByte dbuf)
    {
	byte buf[] = dbuf.getData();
	ByteArrayOutputStream baos = new ByteArrayOutputStream(16+buf.length);
	DataOutputStream dos = new DataOutputStream(baos);

	try{
	    dos.writeInt(img.getWidth());
	    dos.writeInt(img.getHeight());
	    dos.writeInt(3);
	    dos.writeInt(downsampling);
	    dos.write(buf);
	}
	catch(IOException ioe){
	    return;
	}

	jc.publish(nametype,baos.toByteArray());
    }

}
