package maslab.vis;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.awt.image.*;
import java.io.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.imageio.*;
import java.util.*;

import maslab.util.*;

/** Draws VisObjects and implements navigational controls **/
public class VisCanvas extends JComponent
{
    public static double DEFAULTSCALE=30;

    public double cx=0, cy=0; // in meters
    public double scale=DEFAULTSCALE; // meters to pixels
    public double theta=0; // rotation in radians

    // use for screen viewing
    public Color gridColor=Color.black;
    public Color backgroundColor=Color.darkGray;

    // use for generating printable graphs
    //    public Color gridColor=new Color(230,230,230); //Color.lightGray;
    //    public Color backgroundColor=Color.white;

    public double gridLineWidthPx=1;
    public double gridLineSpacing=5; // meters

    public enum ZOOM { IN, OUT };

    AffineTransform transform=new AffineTransform();

    ArrayList<VisObject> immutobjs=new ArrayList<VisObject>();
    ArrayList<VisObject> mutobjs=new ArrayList<VisObject>();
    ArrayList<VisObject> foreobjs=new ArrayList<VisObject>();
    ArrayList<VisObject> backobjs=new ArrayList<VisObject>();

    BufferedImage cachedImage=null;
    double cachedImageCx=0;
    double cachedImageCy=0;
    double cachedImageScale=1;
    double cachedImageTheta=0;

    public static final double MAXZOOM=800;
    public static final double MINZOOM=2;
    public static final double FASTZOOMSPEED=1.3;

    public static final double PANAMOUNT=20; // pixels, roughly.
    public static final double PANAMOUNTFAST=100; // pixels.

    public static final double ROTATEAMOUNT=10.0/180.0*Math.PI; // radians

    /** Determine whether to add an object to the top or bottom. **/
    public enum POSITION { TOP, BOTTOM };

    protected boolean fullRedraw=true;

    protected boolean dirty=true;
    protected Object dirtyObject=new Object();
    protected int repaintDelay=500; // i.e., 2fps

    protected RepaintThread repaintThread;

    protected Logger log=new Logger("VisCanvas");

    final static long serialVersionUID=1001;
    
    boolean greedyFocus=true;
    
    Object drawSyncObject=new Object();

    ArrayList<VisCanvasViewChangeListener> viewChangeListeners=new ArrayList<VisCanvasViewChangeListener>();
    ArrayList<VisCanvasEventListener> eventListeners=new ArrayList<VisCanvasEventListener>();

    JPopupMenu popupMenu;

    static HelpActionListener helpActionListener=new HelpActionListener();

    JCheckBoxMenuItem contrastMenuItem=new JCheckBoxMenuItem("High contrast", false);
    JCheckBoxMenuItem showPositionMenuItem=new JCheckBoxMenuItem("Show position", false);
    JCheckBoxMenuItem showGridMenuItem=new JCheckBoxMenuItem("Show grid", true);

    boolean antialias=true;

    /** Create a new canvas
     */
    public VisCanvas()
    {
	//	add(new VisLabelledPoint(0,0,"(0,0)"),TOP);
	VisMouseMotionListener vmml=new VisMouseMotionListener();

	addMouseMotionListener(vmml);
	addMouseListener(vmml);
	addMouseWheelListener(vmml);

	repaintThread=new RepaintThread();
	repaintThread.start();


	addKeyListener(new MyKeyListener());
	setFocusable(true);
	enableInputMethods(true);

	/*** Setup the menu ****/
	popupMenu=new JPopupMenu("VisCanvas Options");

	contrastMenuItem.addActionListener(new ContrastActionListener());
	popupMenu.add(contrastMenuItem);

	showPositionMenuItem.addActionListener(new RedrawActionListener());
	popupMenu.add(showPositionMenuItem);

	showGridMenuItem.addActionListener(new RedrawActionListener());
	popupMenu.add(showGridMenuItem);
	
	popupMenu.add(new JPopupMenu.Separator());

	JMenuItem jmi;
	jmi=new JMenuItem("Screenshot...");
	jmi.addActionListener(new ScreenshotActionListener());
	popupMenu.add(jmi);

	jmi=new JMenuItem("Make movie...");
	popupMenu.add(jmi);

	popupMenu.add(new JPopupMenu.Separator());

	jmi=new JMenuItem("Help");
	jmi.addActionListener(helpActionListener);

	popupMenu.add(jmi);

	/*************************/
    }

    // nothing to do but schedule a redraw
    class RedrawActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    fullRedraw=true;
	    draw();
	}
    }

    class ContrastActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    if (contrastMenuItem.getState())
		{
		    backgroundColor=Color.darkGray;
		    gridColor=Color.black;
		}
	    else
		{
		    backgroundColor=Color.white;
		    gridColor=new Color(230,230,230);
		}

	    fullRedraw=true;
	    draw();
	}
    }

    static class HelpActionListener implements ActionListener
    {
	JFrame f;

	public HelpActionListener()
	{
	    f=new JFrame("VisCanvas Help");
	    f.getContentPane().setLayout(new BorderLayout());
	    Box b=new Box(BoxLayout.Y_AXIS);
	    f.getContentPane().add(b, BorderLayout.CENTER);
	    JButton okButton=new JButton("Okay");
	    okButton.addActionListener(new ActionListener() 
		{ public void actionPerformed(ActionEvent e) 
		    { f.setVisible(false); }
		});
	    f.getContentPane().add(okButton, BorderLayout.SOUTH);

	    String helpString=
		"<h2>VisCanvas commands</h2>"+
		"<table>" +

		"<tr><td colspan=2><B>Keyboard commands</b>" +

		"<tr><td>h<td>Reset display settings to default" +
		"<tr><td>t<td>Set theta=0" +
		"<tr><td>z<td>Zoom in" +
		"<tr><td>x<td>Zoom out" +
		"<tr><td>(control+arrows)<td>Rotate display" +
		"<tr><td>(arrows)<td>Pan display" +

		"<tr><td colspan=2><B>Mouse commands</b>" +

		"<tr><td>(left drag)<td>Pan display" +	
		"<tr><td>(mouse wheel)<td>Zoom display" +	
		"<tr><td>(control+mouse wheel)<td>Change grid size" +	
		"<tr><td>(middle drag)<td>Rotate display" +	
		"<tr><td>(right mouse)<td>Display option menu" +	

		"<tr><td colspan=2>&nbsp;" +
		"<tr><td colspan=2><I>Hold down shift key to increase speed of virtually all commands above. </I>" +

		"</table>" +
		"<P>VisCanvas by Edwin Olson (eolson@mit.edu), May 2004.";

	    JEditorPane jep=new JEditorPane("text/html",helpString);
	    jep.setEditable(false);

	    f.getContentPane().add(new JScrollPane(jep), BorderLayout.CENTER);

	    f.setSize(450,500);
	}

	public void actionPerformed(ActionEvent e)
	{
	    f.setVisible(true);
	}
    }

    class ScreenshotActionListener implements ActionListener
    {
	JFrame frame=new JFrame("Screenshot");
	Box box =new Box(BoxLayout.X_AXIS);;
	JTextField widthField=new JTextField("800",5);
	JTextField heightField=new JTextField("600",5);
	JButton saveButton=new JButton("Save as...");
	JButton redrawButton=new JButton("Redraw");
	BufferedImageComponent bicomp=new BufferedImageComponent();
	JScrollPane scrollPane=new JScrollPane();

	JFileChooser fileChooser=new JFileChooser();

	public ScreenshotActionListener()
	{
	    frame.getContentPane().setLayout(new BorderLayout());
	    frame.getContentPane().add(box,BorderLayout.SOUTH);
	    frame.getContentPane().add(scrollPane,BorderLayout.CENTER);

	    box.add(new JLabel("Width"));
	    box.add(widthField);
	    box.add(Box.createHorizontalStrut(50));
	    box.add(new JLabel("Height"));
	    box.add(heightField);
	    box.add(redrawButton);

	    box.add(Box.createHorizontalStrut(50));
	    box.add(saveButton);

	    update();

	    redrawButton.addActionListener(new ActionListener() 
		{ public void actionPerformed(ActionEvent e) 
		    { update(); }
		});

	    saveButton.addActionListener(new ActionListener()
		{ public void actionPerformed(ActionEvent e)
		    { save(); }
		});

	    frame.setSize(600,400);
	}

	protected void save()
	{
	    if (JFileChooser.APPROVE_OPTION!=fileChooser.showSaveDialog(VisCanvas.this))
		return;
	    
	    try {
		javax.imageio.ImageIO.write(bicomp.bi, "png", fileChooser.getSelectedFile());
	    } catch (IOException ex) {
		log.error("Couldn't write screen shot.",ex);
	    }
	}

	public void actionPerformed(ActionEvent e)
	{
	    update();
	    frame.setVisible(true);
	}

	protected void update()
	{
	    BufferedImage bi=paintIntoImageScale(Integer.parseInt(widthField.getText()),
						 Integer.parseInt(heightField.getText()));
	    bicomp.update(bi);
	    scrollPane.setViewportView(bicomp);
	}

	class BufferedImageComponent extends JComponent
	{
	    final static long serialVersionUID=1001;
	    BufferedImage bi;

	    public void paint(Graphics g)
	    {
		if (bi==null)
		    return;

		g.drawImage(bi, 0, 0, bi.getWidth(), bi.getHeight(),null);
	    }

	    public void update(BufferedImage bi)
	    {
		this.bi=bi;
		invalidate();
		repaint();
	    }

	    public Dimension getPreferredSize()
	    {
		return new Dimension(bi.getWidth(), bi.getHeight());
	    }
	}
    }

    void showPopup(MouseEvent e) 
    {
	if (e.isPopupTrigger())
	    {
		popupMenu.show(e.getComponent(),
			       e.getX(), e.getY());
	    }
    }

    /** Add a listener to receive notification whenever the view of
     * this canvas is changed.
     **/
    public void addViewChangeListener(VisCanvasViewChangeListener vcvcl)
    {
	if (!viewChangeListeners.contains(vcvcl))
	    viewChangeListeners.add(vcvcl);
    }

    /** Add a listener to receive notification whenever an unhandled
     * user interface event occurs.
     **/
    public void addEventListener(VisCanvasEventListener el)
    {
	if (!eventListeners.contains(el))
	    eventListeners.add(el);
    }

    protected synchronized void notifyChangeListeners()
    {
	for (VisCanvasViewChangeListener vcvcl : viewChangeListeners)
	    {
		vcvcl.viewChanged(this);
	    }
    }

    protected synchronized void notifyEventListeners(MouseEvent e)
    {
	double clickx=(e.getPoint().getX()-getWidth()/2)/scale;
	double clicky=-(e.getPoint().getY()-getHeight()/2)/scale;
	double r=Math.sqrt(clickx*clickx+clicky*clicky);
	double clicktheta=Math.atan2(clicky,clickx)+theta;
	clickx=r*Math.cos(clicktheta);
	clicky=r*Math.sin(clicktheta);
	clickx+=cx;
	clicky+=cy;

	for (VisCanvasEventListener el : eventListeners)
	    {
		el.visCanvasMouseEvent(this, clickx, clicky, e);
	    }
    }


    /** Cause this VisCanvas's view to follow another VisCanvas's view. It is safe
     * to form circular loops.
     **/
    public void synchronizeView(VisCanvas vc)
    {
	vc.addViewChangeListener(new InternalVisCanvasViewChangeListener());
    }

    class InternalVisCanvasViewChangeListener implements VisCanvasViewChangeListener
    {
	public void viewChanged(VisCanvas vc)
	{
	    // don't do anything if we're already set this way!
	    if (VisCanvas.this.cx==vc.cx && 
		VisCanvas.this.cy==vc.cy && 
		VisCanvas.this.scale==vc.scale &&
		VisCanvas.this.theta==vc.theta &&
		VisCanvas.this.gridLineSpacing==vc.gridLineSpacing)
		return;

	    VisCanvas.this.cx=vc.cx;
	    VisCanvas.this.cy=vc.cy;
	    VisCanvas.this.scale=vc.scale;
	    VisCanvas.this.theta=vc.theta;
	    VisCanvas.this.gridLineSpacing=vc.gridLineSpacing;

	    fullRedraw=true;
	    drawNow();

	    // notify our listeners that we changed.
	    notifyChangeListeners();
	}
    }

    /** Set whether the canvas should steal the focus whenver the
     * mouse hovers over it. **/
    public void setGreedyFocus(boolean enable)
    {
	greedyFocus=enable;
    }


    /** Add an object to the back buffer. Will not become visible until switchBuffer() is called.
     **/
    public void addBuffered(VisObject vo)
    {
	backobjs.add(vo);
    }
    
    public void addBuffered(VisObject vo, POSITION pos)
    {
	if (pos==POSITION.TOP)
	    backobjs.add(vo);
	else
	    backobjs.add(0,vo);
    }

    /** Make all objects added via addBuffered since last call to switchBuffer() visible, and
     * discard the objects that were added before. i.e., this is used for double buffering an 
     * animation.
     **/
    public void switchBuffer()
    {
	foreobjs=backobjs;
	backobjs=new ArrayList<VisObject>();

	draw();
    }

    public void clear()
    {
	immutobjs=new ArrayList<VisObject>();
	mutobjs=new ArrayList<VisObject>();
	foreobjs=new ArrayList<VisObject>();
	backobjs=new ArrayList<VisObject>();

	fullRedraw=true;
	draw();
    }

    /** Add a new mutable object on top of existing objects. 
     */
    public void add(VisObject vo)
    {
	addMutable(vo, POSITION.TOP);
    }

    /** Call this method, rather than repaint(), to schedule a redraw. This method
    * enforces a maximum frame rate. **/
    public void draw()
    {
	synchronized(dirtyObject) {
	    dirty=true;
	    dirtyObject.notify();
	}
    }

    /** Cause a repaint() ASAP, without regard for the maximum frame rate **/
    public void drawNow()
    {
	repaint();
	dirty=false;
    }

    public void drawSync()
    {
	synchronized(drawSyncObject)
	    {
		repaint();
		dirty=false;

		try {
		    drawSyncObject.wait();
		} catch (InterruptedException ex) {
		}
	    }
    }

    /** Add a new mutable object at the specified layer 
     */
    public void add(VisObject vo, POSITION pos)
    {
	addMutable(vo, pos);
    }

    /** Add a new mutable object at the specified layer
     */
    public void addMutable(VisObject vo, POSITION pos)
    {
	if (pos==POSITION.TOP)
	    mutobjs.add(vo);
	else
	    mutobjs.add(0,vo);

	draw();
    }

    public void addImmutable(VisObject vo)
    {
	immutobjs.add(vo);
	fullRedraw=true;
    }

    /** Add a new immutable object (one which does not have to be redrawn) at
	the specified layer.
    */
    public void addImmutable(VisObject vo, POSITION pos)
    {
	if (pos==POSITION.BOTTOM)
	    immutobjs.add(vo);
	else
	    immutobjs.add(0,vo);

	draw();
    }

    /** Move the center of the display to the specified coordinates. **/
    public void setCenter(double cx, double cy)
    {
	this.cx=cx;
	this.cy=cy;

	draw();
    }

    public void zoom(ZOOM dir, boolean fast)
    {
	zoom(dir, -1, -1, fast);
    }

    public void zoom(ZOOM dir, int mousex, int mousey, boolean fast)
    {
	double oldscale=scale;

	if (dir==ZOOM.IN)
	    {
		if (fast)
		    scale=Math.min(MAXZOOM, scale*FASTZOOMSPEED);
		else if (scale<MAXZOOM)
		    scale++;
	    }
	else
	    {
		if (fast)
		    scale=Math.max(MINZOOM, scale/FASTZOOMSPEED);
		else if (scale>MINZOOM)
		    scale--;
	    }

	// if we got a mouse coordinate, zoom so that the mouse position is over the same
	// coordinate. (We now have to translate in order to accomplish this!)
	if (mousex>=0 && mousey>=0)
	    {
		cx-=(mousex-getWidth()/2)*(1/scale - 1/oldscale);
		cy+=(mousey-getHeight()/2)*(1/scale - 1/oldscale);
	    }

	if (scale!=oldscale)
	    {
		drawNow();
		notifyChangeListeners();
	    }
    }

    /** Restore the view to the start-up default. **/ 
    public void goHome()
    {
	cx=0;
	cy=0;
	theta=0;
	scale=DEFAULTSCALE;
	drawNow();
	notifyChangeListeners();
    }


    public void pan(double right, double down, boolean fast)
    {
	double dx=right*Math.cos(theta)+down*Math.sin(theta);
	double dy=-right*Math.sin(theta)+down*Math.cos(theta);
	
	cx+=dx*(fast ? PANAMOUNTFAST : PANAMOUNT)/scale;
	cy-=dy*(fast ? PANAMOUNTFAST : PANAMOUNT)/scale;

	drawNow();
	notifyChangeListeners();
    }

    public void rotateRight()
    {
	theta+=ROTATEAMOUNT;
	drawNow();
	notifyChangeListeners();
    }

    public void rotateLeft()
    {
	theta-=ROTATEAMOUNT;
	drawNow();
	notifyChangeListeners();
    }

    class VisMouseMotionListener implements MouseMotionListener, MouseListener, MouseWheelListener
    {
	Point dragBegin=null;
	boolean dragDetected=false;


	public void mouseWheelMoved(MouseWheelEvent e)
	{
	    int amount=e.getWheelRotation();
	    int mods=e.getModifiersEx();
	    boolean shiftDown=(mods&MouseEvent.SHIFT_DOWN_MASK)>0;
	    boolean controlDown=(mods&MouseEvent.CTRL_DOWN_MASK)>0;

	    if (controlDown)
		{
		    double[] gridLineSpacings=new double[] {1,2,5,10,25,50,100,250,500};

		    int bestidx=4;
		    double err=Double.MAX_VALUE;
		    for (int i=0;i<gridLineSpacings.length;i++)
			{
			    if (Math.abs(gridLineSpacings[i]-gridLineSpacing)<err)
				{
				    err=Math.abs(gridLineSpacings[i]-gridLineSpacing);
				    bestidx=i;
				}
			}

		    if (amount>0)
			bestidx--;
		    else
			bestidx++;

		    bestidx=Math.max(0,bestidx);
		    bestidx=Math.min(gridLineSpacings.length-1, bestidx);

		    gridLineSpacing=gridLineSpacings[bestidx];

		    fullRedraw=true;
		    drawNow();
		    notifyChangeListeners();
		}
	    else
		{
		    if (amount>0)
			zoom(ZOOM.OUT, e.getX(), e.getY(), shiftDown);
		    else if (amount<0)
			zoom(ZOOM.IN, e.getX(), e.getY(), shiftDown);
		}
	}

	public void mouseDragged(MouseEvent e)
	{
	    int mods=e.getModifiersEx();

	    dragDetected=true;

	    // left button pans
	    if ((mods&InputEvent.BUTTON1_DOWN_MASK)>0)
		{
		    setCursor(new Cursor(Cursor.MOVE_CURSOR));

		    Point p=e.getPoint();
		    double dx=(e.getPoint().getX()-dragBegin.getX())/scale;
		    double dy=(e.getPoint().getY()-dragBegin.getY())/scale;
		    cx-=dx*Math.cos(theta)+dy*Math.sin(theta);
		    cy+=-dx*Math.sin(theta)+dy*Math.cos(theta);

		    dragBegin=e.getPoint();
		    drawNow();
		    notifyChangeListeners();
		}

	    // right button rotates
	    if ((mods&InputEvent.BUTTON2_DOWN_MASK)>0 && dragBegin!=null)
		{
		    setCursor(new Cursor(Cursor.HAND_CURSOR));

		    Point p=e.getPoint();

		    double dtheta=Math.atan2(dragBegin.getY()-getHeight()/2, 
					     dragBegin.getX()-getWidth()/2) -
			Math.atan2(p.getY()-getHeight()/2,p.getX()-getWidth()/2);

		    theta-=dtheta;
		    dragBegin=e.getPoint();
		    drawNow();
		    notifyChangeListeners();
		}
	}

	public void mouseMoved(MouseEvent e)
	{
	    dragBegin=null;
	}

	public void mousePressed(MouseEvent e)
	{
	    int mods=e.getModifiersEx();

	    dragDetected=false;

	    dragBegin=e.getPoint();

	    if ((mods&InputEvent.BUTTON3_DOWN_MASK)>0)
		{
		    showPopup(e);
		}
	    requestFocusInWindow();
	}

	public void mouseReleased(MouseEvent e)
	{
	    setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
	}

	public void mouseClicked(MouseEvent e)
	{
	    if (dragDetected)
		{
		    dragDetected=false;
		    return;
		}

	    notifyEventListeners(e);
	}

	public void mouseEntered(MouseEvent e)
	{
	    if (greedyFocus)
		requestFocus();
	}

	public void mouseExited(MouseEvent e)
	{
	}

    }

    class MyKeyListener extends KeyAdapter
    {
	public void keyPressed(KeyEvent e)
	{
	    handleKeyPress(e);
	}
    }

    // returns true if the keypress was handled
    public boolean handleKeyPress(KeyEvent e)
    {
	boolean ok=false;
	int mods=e.getModifiersEx();

	switch(e.getKeyChar())
	    {
	    case 'Z':
		zoom(ZOOM.IN, true);
		ok=true;
		break;

	    case 'z':
		zoom(ZOOM.IN, false);
		ok=true;
		break;
	
	    case 'X':
		zoom(ZOOM.OUT, true);
		ok=true;
		break;

	    case 'x':
		zoom(ZOOM.OUT, false);
		ok=true;
		break;

	    case 'h':
	    case 'H':
		goHome();
		ok=true;
		break;

	    case 't':
	    case 'T':
		theta=0;
		ok=true;
		drawNow();
		notifyChangeListeners();
		break;

	    case 'r':
	    case 'R':
		fullRedraw=true;
		draw();
		ok=true;
		break;

	    case '?':
		showHotKeys();
		ok=true;
		break;
	    }

	if (!ok)
	    {
		switch (e.getKeyCode())
		    {
		    case KeyEvent.VK_UP:
			pan(0, -1, (mods&InputEvent.SHIFT_DOWN_MASK)>0);
			ok=true;
			break;
			
		    case KeyEvent.VK_DOWN:
			pan(0, 1, (mods&InputEvent.SHIFT_DOWN_MASK)>0);
			ok=true;
			break;
			
		    case KeyEvent.VK_LEFT:
			if ((mods&InputEvent.CTRL_DOWN_MASK)>0)
			    rotateLeft();
			else
			    pan(-1, 0, (mods&InputEvent.SHIFT_DOWN_MASK)>0);
			ok=true;
			break;
			
		    case KeyEvent.VK_RIGHT:
			if ((mods&InputEvent.CTRL_DOWN_MASK)>0)
			    rotateRight();
			else
			    pan(1, 0, (mods&InputEvent.SHIFT_DOWN_MASK)>0);
			ok=true;
			break;
			
		    default:
			//			System.out.println("key code: "+e.getKeyCode());
		    }
	    }

	return ok;
    }

    public void showHotKeys()
    {
	System.out.println("Vis Help:");
	System.out.println("------------------------------------");
	System.out.println("h             Center window on (0,0)");
	System.out.println("t             Set theta=0");
	System.out.println("r             Repaint window");
	System.out.println("z             Zoom in");
	System.out.println("Z             Zoom in fast");
	System.out.println("x             Zoom out");
	System.out.println("X             Zoom out fast");

	System.out.println("(left drag)   Move center of window");
	System.out.println("(right drag)  Rotate window");
	System.out.println("arrows        Move center of window");
	System.out.println("shift+arrows  Move center of window fast");
	System.out.println("ctrl-arrows   Rotate window");

    }

    protected void paintBackground(BufferedImage bi, Graphics2D g)
    {
	int width=bi.getWidth();
	int height=bi.getHeight();

	double sqrt2=1.5;

	int maxd=(int) Math.max(width,height);

// 	double widthm=sqrt2*width/scale;
// 	double heightm=sqrt2*height/scale;

	double widthm=sqrt2*maxd/scale;
	double heightm=sqrt2*maxd/scale;

	double leftm=(cx-widthm/2);
	double rightm=(cx+widthm/2);
	double topm=(cy-heightm/2);
	double bottomm=(cy+heightm/2);

	g.setColor(backgroundColor);
	g.fill(new Rectangle2D.Double(leftm, topm, widthm, heightm));
	
	g.setColor(gridColor);

	g.setStroke(new BasicStroke((float) (gridLineWidthPx/scale)));

	if (showGridMenuItem.getState())
	    {
		double margin=Math.max(widthm,heightm);

		int left=(int)Math.round((cx-margin-1));
		int right=(int) Math.round((cx+margin+1));

		left=(int) (Math.round((left-gridLineSpacing)/gridLineSpacing)*gridLineSpacing);
		right=(int) (Math.round((right-gridLineSpacing)/gridLineSpacing)*gridLineSpacing);

		for (int i=left;i<=right;i+=gridLineSpacing)
		    g.draw(new Line2D.Double(i,topm,i,bottomm));
		
		int top=(int)Math.round((cy-margin-1));
		int bottom=(int)Math.round((cy+margin+1));

		top=(int) (Math.round((top-gridLineSpacing)/gridLineSpacing)*gridLineSpacing);
		bottom=(int) (Math.round((bottom-gridLineSpacing)/gridLineSpacing)*gridLineSpacing);

		for (int i=top;i<=bottom;i+=gridLineSpacing)
		    g.draw(new Line2D.Double(leftm,i,rightm,i));
	    }

    }

    public synchronized void paint(Graphics gin)
    {
	int width=getWidth();
	int height=getHeight();

	BufferedImage bi=paintIntoImage(width, height);

	gin.drawImage(bi, 0, 0, width, height, null);

	synchronized(drawSyncObject)
	    {
		drawSyncObject.notifyAll();
	    }
    }

    /** Render the current display into an image of width and height,
     * zooming in as much as possible while displaying an area at
     * least as large as is currently displayed.
     **/
    public synchronized BufferedImage paintIntoImageScale(int width, int height)
    {
	double oldscale=scale;

	double scalefactor=Math.min(width/((double) getWidth()), height/((double) getHeight()));
	scale=scalefactor*oldscale;

	BufferedImage bi=paintIntoImage(width,height);
	scale=oldscale;

	return bi;
    }

    /** Render the display into a new BufferedImage object of the requested
     * width and height. The current cx, cy, theta are used. 
     **/
    public synchronized BufferedImage paintIntoImage(int width, int height)
    {
	BufferedImage im=new BufferedImage(width,height,BufferedImage.TYPE_4BYTE_ABGR);
	Graphics2D g=im.createGraphics();

	if (antialias)
	    g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

	// determine if we must redraw everything from scratch.
	if (cachedImage==null || cachedImage.getWidth()!=width || 
	    cachedImage.getHeight()!=height ||
	    cachedImageCx!=cx || cachedImageCy!=cy || cachedImageTheta!=theta || 
	    cachedImageScale!=scale || fullRedraw)
	    {
		// start over!
		cachedImage=new BufferedImage(width, height, BufferedImage.TYPE_4BYTE_ABGR);
		Graphics2D g2=cachedImage.createGraphics();
		if (antialias)
		    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		g2.setTransform(g.getDeviceConfiguration().getDefaultTransform());
		g2.translate(width/2,height/2);
		g2.rotate(theta);
		g2.scale(scale,-scale);
		g2.translate(-cx,-cy);

		paintBackground(cachedImage, g2);
		paintObjects(cachedImage, g2, immutobjs);

		g2.dispose();

		cachedImageCx=cx;
		cachedImageCy=cy;
		cachedImageScale=scale;
		cachedImageTheta=theta;
		fullRedraw=false;
	    }

	// draw the immutable data.
	g.drawImage(cachedImage, 0, 0, width, height, null);

	g.setTransform(g.getDeviceConfiguration().getDefaultTransform());
	g.translate(width/2,height/2);
	g.rotate(theta);
	g.scale(scale,-scale);
	g.translate(-cx,-cy);

	paintObjects(im, g,mutobjs);
	paintObjects(im, g,foreobjs);

	if (showPositionMenuItem.getState())
	    {
		VisCaption position=new VisCaption(Color.magenta, 
						   VisCaption.BOTTOMRIGHT, 
						   "("+StringUtil.formatDouble(cx,3)+","+
						   StringUtil.formatDouble(cy,3)+")"+"\n" +
						   "theta="+StringUtil.formatDouble(-theta,3)+"\n"+
						   "scale="+StringUtil.formatDouble(scale,3)+" px/unit\n"+
						   "["+im.getWidth()+"x"+im.getHeight()+"]"+"\n"+
						   "grid="+gridLineSpacing);

		ArrayList<VisObject> al=new ArrayList<VisObject>();
		al.add(position);

		paintObjects(im, g, al);
	    }

	g.dispose();

	return im;
    }

    /** Paint objects into a graphics context.
     **/
    public void paintObjects(BufferedImage bi, Graphics2D g, 
			     ArrayList<VisObject> objects)
    {
	int l=objects.size();
	AffineTransform t=g.getTransform();
	for (int i=0;i<l;i++)
	    {
		VisObject vo=objects.get(i);

		if (vo.coordinateSpace==VisObject.SCREENSPACE)
		    g.setTransform(g.getDeviceConfiguration().getDefaultTransform());
		else
		    g.setTransform(t);

		g.setColor(Color.white);
		g.setStroke(new BasicStroke((float) (1/scale)));

		vo.paint(this,g, bi);
	    }
    }

    /** Create a font with the appropriate affine transform for the
     * canvas. This is intended to be used by VisObjects that want to
     * draw text. 
     * @param family e.g. "Helvetica"
     * @param type e.g. Font.PLAIN
     * @param heightm The height of the font, in meters.
     * @return A font of the specified family/type with the correct
     * transformation matrix.
     **/
    public Font getFont(String family, int type, double heightm)
    {
	Font f=new Font(family, type, 1);
	f=f.deriveFont((float) heightm);
	AffineTransform t=f.getTransform();
	t.scale(1,-1);
	f=f.deriveFont(t);
	
	return f;
    }

    /** Set the maximum redraw rate. It is typically limited to ensure
     * that we don't needlessly redraw the screen.
     * @param fps The maximum frame rate.
     **/
    public void setMaximumFrameRate(double fps)
    {
	repaintDelay=(int) (1000.0/fps);
    }
    
    /** The current scale of the display, in pixels per unit (meter). **/
    public double getScale()
    {
	return scale;
    }
    
    class RepaintThread extends Thread
    {
	public RepaintThread()
	{
	    setDaemon(true);
	}

	public void run()
	{
	    while (true)
		{
		    synchronized(VisCanvas.this.dirtyObject) {
			
			while (!VisCanvas.this.dirty) {
			    try {
				dirtyObject.wait();
			    } catch (InterruptedException ex) {
			    }
			}

			VisCanvas.this.dirty=false;
		    }
		    
		    VisCanvas.this.repaint();
		    
		    try {
			Thread.sleep(repaintDelay);
		    } catch (InterruptedException ex) {
		    }
		}
	}
    }
}
