package maslab.camera;

import maslab.util.*;

import java.awt.image.*;
import javax.imageio.*;
import java.io.*;

public class Camera
{
    Logger log=new Logger(this);

    public static final int WB_INDOOR = 0;
    public static final int WB_OUTDOOR = 1;
    public static final int WB_FLOURESCENT = 2;
    public static final int WB_MANUAL = 3;
    public static final int WB_AUTO = 4;

    protected static native int camera_open(String devname);
    protected static native void camera_close(int ref);
    protected static native void camera_capture(int ref, int[] image);
    protected static native void camera_set_capture_settings(int ref, int width, int height, int fps);
    protected static native int camera_get_width(int ref);
    protected static native int camera_get_height(int ref);
    protected static native int camera_get_fps(int ref);
    protected static native void camera_set_white_balance(int ref, int mode, int red, int blue);
    protected static native int camera_get_white_balance(int ref, int field);
    protected static native void camera_set_led(int ref, int on_time, int off_time);
    protected static native int camera_get_led_on(int ref);
    protected static native int camera_get_led_off(int ref);
    protected static native void camera_set_quality(int ref, int value);
    protected static native int camera_get_quality(int ref);
    protected static native void camera_set_contour(int ref, int value);
    protected static native int camera_get_contour(int ref);
    protected static native void camera_set_backlight(int ref, int value);
    protected static native int camera_get_backlight(int ref);
    protected static native void camera_set_flicker(int ref, int value);
    protected static native int camera_get_flicker(int ref);
    protected static native void camera_set_noisereduction(int ref, int value);
    protected static native int camera_get_noisereduction(int ref);
    protected static native void camera_set_gain(int ref, int value);
    protected static native int camera_get_gain(int ref);
    protected static native void camera_set_shutter(int ref, int value);
    protected static native void rgb_to_hsv(int[] image, int width, int height);
    protected static native void rgb_to_rgv(int[] image, int width, int height);
    protected static native void channel_select(int[] image, int width, int height, int channel);

    static
    {
	String loadedPath="";

	// look for the camera library in a list of typical places.
	String[] candidatepaths={ "/opt/maslab/lib/libcamera.so",
				  "maslab/camera/native/libcamera.so",
				  "src/maslab/camera/native/libcamera.so",
				  "/tmp/libcamera.so"
	};
	
	boolean loaded=false;
	
	// if we didn't find it, look inside the jar
	if (!loaded)
	    {
		//The jar should be in the classpath.  Look for it there.
		String cp = System.getProperty("java.class.path");
		String[] items = StringUtil.split(cp,":");
		for (int i=0; i<items.length; i++)
		    {
			if (items[i].endsWith("maslab.jar") && (new File(items[i])).exists())
			    {
				try{
				    ZipExtractor ze = new ZipExtractor(items[i]);
				    ze.saveResource("maslab/camera/native/libcamera.so","/tmp/libcamera.so");
				    System.load("/tmp/libcamera.so");
				    loaded = true;
				    loadedPath = "/tmp/libcamera.so";
				    break;
				}
				catch(IOException ioe){}
			    }
		    }
	    }

	if (!loaded)
	    {
		for (int i=0;i<candidatepaths.length;i++)
		    {
			File f=new File(candidatepaths[i]);
			if (f.exists())
			    {
				String p=f.getAbsolutePath();
				System.load(p);
				loadedPath=p;
				loaded=true;
				break;
			    }
		    }
	    }

	if (!loaded)
	    throw new RuntimeException("Unable to load native camera library");

	System.out.println("Using camera library: "+loadedPath);
    };

    protected int ref; // this is actually used to store a pointer...

    /** Open a camera, searching /dev/videoX and /dev/video. **/
    public Camera() throws IOException
    {
	if (ref==0)
	    this.ref=camera_open("/dev/video");
	if (ref==0)
	    this.ref=camera_open("/dev/video0");
	if (ref==0)
	    this.ref=camera_open("/dev/video1");

	if (ref==0)
	    throw new IOException("Unable to connect to camera.");
    }

    /** Open a camera.
     * @param devname The path to the device to open.
     **/
    public Camera(String devname) throws IOException
    {
	this.ref=camera_open(devname);

	if (ref==0)
	    throw new IOException("Unable to connect to camera " + devname);
    }

    /** Close the camera. Methods should not be called on this object
     * after close.
     **/
    public synchronized void close()
    {
	camera_close(ref);
	ref=0;
    }

    /** Create a BufferedImage with the correct format/bitdepth for
     * future capture operations. A new BufferedImage must be created
     * whenever camera resolution/bit-depth changes.
     **/
    public synchronized BufferedImage createImage()
    {
	int width = camera_get_width(ref);
 	int height = camera_get_height(ref);
	BufferedImage bi = new BufferedImage(width,height,BufferedImage.TYPE_INT_RGB);
	return bi;
    }

    /** Capture a single frame, allocating a new buffer. **/
    public BufferedImage capture()
    {
	return capture(null);
    }

    /** Capture a single frame.
     * @param image The image to be overwritten with the current
     * camera frame.
     **/
    public BufferedImage capture(BufferedImage image)
    {
	// it's very common for people to do a while(1) capture loop,
	// but this hogs our semaphore.  So we'll yield before
	// acquiring the semaphore.
	Thread.yield();

	synchronized(this)
	    {
		if (image==null)
		    image=createImage();
		
		if (image.getWidth(null)!=getWidth() || image.getHeight(null)!=getHeight())
		    {
			log.output("Resizing input buffer (slow)");
			image=createImage();
		    }
		
		DataBufferInt dbuf = (DataBufferInt)image.getRaster().getDataBuffer();
		int buf[] = dbuf.getData();
		camera_capture(ref, buf);
	    }
	
	return image;
    }

    /** Request a new size from the camera. 
     * @param width Width, in pixels
     * @param height Height, in pixels.
     **/
    public synchronized void setCaptureSettings(int width, int height, int fps)
    {
	camera_set_capture_settings(ref, width, height, fps);
    }

    /** The current width setting of the camera. **/
    public synchronized int getWidth()
    {
	return camera_get_width(ref);
    }

    /** The current height setting of the camera. **/
    public synchronized int getHeight()
    {
	return camera_get_height(ref);
    }

    /** The current target fps setting of the camera. **/
    public synchronized int getFps()
    {
	return camera_get_fps(ref);
    }

    /** Set the current white balance mode, e.g., WB_INDOOR. For
     * computer vision applications, using automatic white balance can
     * be very undesirable.
     **/
    public synchronized void setWhiteBalanceMode(int mode)
    {
	camera_set_white_balance(ref, mode,0,0);
    }

    /** Determine the current white balance mode (e.g., WB_INDOOR). **/
    public synchronized int getWhiteBalanceMode()
    {
	return camera_get_white_balance(ref, 0);
    }

    /** If white balance mode is manual, returns the red gain. **/
    public synchronized int getWhiteBalanceRed()
    {
	return camera_get_white_balance(ref, 1);
    }

    /** If white balance mode is manual, returns the blue gain. **/
    public synchronized int getWhiteBalanceBlue()
    {
	return camera_get_white_balance(ref, 2);
    }

    /** Sets the camera to manual white balance, using the provided
     * red and blue gains.
     * @param red [0,65535] Red channel gain
     * @param blue [0,65535] Blue channel gain. 
     **/
    public synchronized void setWhiteBalanceManual(int red, int blue)
    {
	camera_set_white_balance(ref, WB_MANUAL,red,blue);
    }

    /** Set the LED's blinking rate. The resolution is internally only
     * about 100ms. Set (100,0) for on, (0,100) for off.
     * @param onTime [0-25000] (ms)
     * @param offTime [range unknown] (ms)
     **/
    public synchronized void setLed(int onTime, int offTime)
    {
	camera_set_led(ref, onTime, offTime);
    }

    public synchronized int getLedOn()
    {
	return camera_get_led_on(ref);
    }

    public synchronized int getLedOff()
    {
	return camera_get_led_off(ref);
    }

    /** Set quality (compression) level.
     * @param value [0-3]. 0=no compression, 3=high compression. 
     **/
    public synchronized void setQuality(int value)
    {
	camera_set_quality(ref, value);
    }

    /** Get the current quality level.
     **/
    public synchronized int getQuality()
    {
	return camera_get_quality(ref);
    }

    /** Set Contour filter.
     * @param value [0-65535]. 0=blur, 65535=sharpen. -1 means auto.
     **/
    public synchronized void setContour(int value)
    {
	camera_set_contour(ref, value);
    }

    /** Get contour filter value. **/
    public synchronized int getContour()
    {
	return camera_get_quality(ref);
    }

    /** Set Backlight Compensation mode. Only operates if auto gain is on. 
     * @param value Enabled if true.
     **/
    public synchronized void setBacklightCompensation(boolean value)
    {
	camera_set_backlight(ref, value ? 1 : 0);
    }

    /** Get BacklightCompensation mode. **/
    public synchronized boolean getBacklightCompensation()
    {
	return camera_get_backlight(ref)!=0;
    }
 
     /** Set Flicker Compensation mode.
     * @param value Enabled if true.
     **/
    public synchronized void setFlicker(boolean value)
    {
	camera_set_flicker(ref, value ? 1 : 0);
    }

    /** Get Flicker Compensation mode. **/
    public synchronized boolean getFlicker()
    {
	return camera_get_flicker(ref)!=0;
    }
 
    /** Set Noise Reduction mode.
     * @param value [0,3]. 0=off, 3=lots of filtering.
     **/
    public synchronized void setNoiseReduction(int value)
    {
	camera_set_noisereduction(ref, value);
    }

    /** Get Noise Reduction mode.**/
    public synchronized int getNoiseReduction()
    {
	return camera_get_noisereduction(ref);
    }

    /** Set gain control. 
     * @param value [0,65535]. 0=low gain, 65535=max gain. gain<0 means auto. 
     **/
    public synchronized void setGain(int value)
    {
	camera_set_gain(ref, value);
    }

    /** Get gain value. **/
    public synchronized int getGain()
    {
	return camera_get_gain(ref);
    }

    /** Set shutter speed. 0=short shutter, 65535=maximum shutter, but
     * scale is nonlinear. speed<0 means auto.**/
    public synchronized void setShutter(int value)
    {
	camera_set_shutter(ref, value);
    }

    /** Converts the image into RGB. The modification is done
     * in-place, if the image buffer type is INT_RGB, otherwise, a new
     * image is allocated. The HSV image is returned.
     **/
    public static BufferedImage rgbToHsv(BufferedImage image)
    {
	image = ImageUtil.convertImage(image, BufferedImage.TYPE_INT_RGB);

	DataBufferInt dbuf = (DataBufferInt)image.getRaster().getDataBuffer();
	int buf[] = dbuf.getData();

	rgb_to_hsv(buf, image.getWidth(null), image.getHeight(null));

	return image;
    }

    /** Converts the image into RGV. The modification is done
     * in-place, if the image buffer type is INT_RGB, otherwise, a new
     * image is allocated. The HSV image is returned.
     **/
    public static BufferedImage rgbToRgv(BufferedImage image)
    {
	image = ImageUtil.convertImage(image, BufferedImage.TYPE_INT_RGB);

	DataBufferInt dbuf = (DataBufferInt)image.getRaster().getDataBuffer();
	int buf[] = dbuf.getData();

	rgb_to_rgv(buf, image.getWidth(null), image.getHeight(null));

	return image;
    }

    public static BufferedImage channelSelect(BufferedImage image, int channel)
    {
	image = ImageUtil.convertImage(image, BufferedImage.TYPE_INT_RGB);

	DataBufferInt dbuf = (DataBufferInt)image.getRaster().getDataBuffer();
	int buf[] = dbuf.getData();

	channel_select(buf, image.getWidth(null), image.getHeight(null), channel);

	return image;
    }
}
