package maslab.orc;

import maslab.util.*;

import java.io.*;

/** Orc wrapper for a motor. **/
public class Motor
{
    Orc orc;
    int motorport;
    int quadphaseport;

    boolean motorflip;
    boolean quadflip;

    int lastencoder=0;
    int encoderCounts=0;
    long absoluteCounts=0;

    /** Create a new Motor wrapper.
     * @param orc The orc object.
     * @param motorport The port that the motor is connectoed to [0,3]
     * @param motorflip Set to true if the motor runs "backwards'
     **/
    public Motor(Orc orc, int motorport, boolean motorflip)
    {
	this.orc=orc;
	this.motorport=motorport;
	this.motorflip=motorflip;

	this.quadphaseport=-1;
	this.quadflip=false;
    }

    /** Create a new Motor wrapper, assuming normal quadrature phase decoding.
     * @param orc The orc object.
     * @param motorport The port that the motor is connectoed to [0,3]
     * @param motorflip Set to true to reverse the motor direction.
     * @param quadphaseport The port that the quadphase is connected to (16 or 18)
     * @param quadflip Set to true to reverse the quadphase direction.
     **/
    public Motor(Orc orc, int motorport, boolean motorflip, int quadphaseport, boolean quadflip)
    {
	this(orc, motorport, motorflip, quadphaseport, quadflip, Orc.PinMode.QUADPHASE);
    }

    /** Create a new Motor wrapper, using the specified wheel position decoding mode..
     * @param orc The orc object.
     * @param motorport The port that the motor is connectoed to [0,3]
     * @param motorflip Set to true to reverse the motor direction.
     * @param quadphaseport The port that the quadphase is connected to (16 or 18)
     * @param quadflip Set to true to reverse the quadphase direction.
     * @param m Set the quadphase to the particular type (fast, normal, mono)
     **/
    public Motor(Orc orc, int motorport, boolean motorflip, int quadphaseport, boolean quadflip, Orc.PinMode m)
    {
	this.orc=orc;
	this.motorport=motorport;
	this.motorflip=motorflip;
	this.quadphaseport=quadphaseport;
	this.quadflip=quadflip;

	orc.pinModeWrite(quadphaseport, m);
	if (m!=Orc.PinMode.MONOPHASE)
	    orc.pinModeWrite(quadphaseport+1, m);

	UpdateThread ut=new UpdateThread();
	ut.start();
    }


    /** Set the motor's PWM.
     * @param pwm The direction/strength to drive, [-255, 255].
     **/
    public void set(int pwm)
    {
	if (motorflip)
	    pwm=-pwm;

	if (pwm>255)
	    pwm=255;
	if (pwm<-255)
	    pwm=-255;

	orc.motorWrite(motorport, pwm);
    }

    /** Get the motor's (actual) PWM. Depending on the slew rate, it
     * takes a while for the read pwm value to reach the requested
     * pwm.
     **/
    public int get()
    {
	int pwm=orc.motorActualRead(motorport);
	
	if (motorflip)
	    pwm=-pwm;

	return pwm;
    }

    /** Get the requested PWM, as last set. **/
    public int getGoal()
    {
	int pwm=orc.motorGoalRead(motorport);
	
	if (motorflip)
	    pwm=-pwm;

	return pwm;
    }

    /** Get the motor current in amps. (Very approximate.) **/
    public double current()
    {
	return orc.motorCurrentRead(motorport);
    }

    /** Returns the number of counts since the last call to this
     * function. 
     **/
    synchronized public int encoder()
    {
	if (quadphaseport==-1)
	    return 0;

	updateEncoder();
	int value=encoderCounts;
	encoderCounts=0;

	return value;
    }

    /** Return the total number of encoder counts since the app was
	started.  Note that even a long will eventually overflow!
    **/
    synchronized public long encoderAbsolute()
    {
	updateEncoder();
	return absoluteCounts;
    }

    synchronized void updateEncoder()
    {
	int thisencoder=orc.quadphaseRead(quadphaseport);
	int delta=orc.diff16(thisencoder, lastencoder);

	lastencoder=thisencoder;

	if (quadflip)
	    delta=-delta;

	encoderCounts+=delta;
	absoluteCounts+=delta;
    }

    class UpdateThread extends Thread
    {
	public UpdateThread()
	{
	    setDaemon(true);
	}

	public void run()
	{
	    while(true)
		{
		    try {
			Thread.sleep(250);
		    } catch (InterruptedException ex) {
		    }

		    
		}
	}
    }

    /** A very simple little motor test program. **/
    public static void main(String[] args)
    {
	Orc orc;
	
	try {
	    orc=new Orc();
	} catch (Exception ex) {
	    System.out.println("Couldn't connect to orc.");
	    return;
	}

	System.out.println("Assuming there's a motor on port 0...");

	BufferedReader ins=new BufferedReader(new InputStreamReader(System.in));

	Motor mot=new Motor(orc, 0, false, 16, false);

	while(true)
	    {
		String line="";

		try {
		    System.out.print("Set pwm to: ");
		    line=ins.readLine();

		    int pwm=Integer.parseInt(line);
		    mot.set(pwm);
		} catch (Exception ex) {
		    System.out.println("Huh?");
		}


		System.out.println("PWM:     "+mot.get());
		System.out.println("Current: "+mot.current());
		System.out.println("Encoder: "+mot.encoder());
	    }
    }

}
