package maslab.orc;

import java.net.*;
import java.io.*;
import java.awt.image.*;
import java.awt.*;
import java.awt.color.ColorSpace;
import java.util.*;
 
import maslab.util.*;

class OrcPacket
{
    public byte[] buf;

    public final static int 
	MASTER_TIMESTAMP=4,
	MASTER_DIGITAL_IN=6,
	MASTER_SERVO=8,
	MASTER_ANALOG=16,
	MASTER_LPF=36,
	MASTER_PINMODE=41,
	MASTER_US_RANGE=49,
	MASTER_US_PINGCOUNT=53,
	MASTER_CLOCK=54,
	MASTER_DAC=57,
	MASTER_GYRO=58;

    public final static int
	SLAVE_PINMODE=4,
	SLAVE_DIGITAL_IN=6,
	SLAVE_QUADPHASE=7,
	SLAVE_MOTOR=13,
	SLAVE_MOTORDIR=25, // (goal0=0-1, actual0=2-3)
	SLAVE_ANALOG_IN=27,
	SLAVE_MOTOR_CURRENT=27,
	SLAVE_SERVO_CURRENT=35,
	SLAVE_LPF=43;

    public final static int
	LCD_JOYX=4,
	LCD_JOYY=5,
	LCD_BUTTONS=6,
	LCD_MOVE=7,
	LCD_BATTERY_VOLTAGE=9;

    public OrcPacket()
    {
    }

    // Just like the other constructor, except note that the byte array is
    // added to the END of the packet.
    public OrcPacket(byte[] b, int blen, Object... args)
    {
	buf=new byte[args.length+blen+3]; // plus header, length, checksum
	buf[0]=(byte) 237;
	buf[1]=(byte) (buf.length);

	for (int i=0;i<args.length;i++)
	    {
		Object o=args[i];

		if (o instanceof Byte)
		    buf[i+2]=(Byte) args[i];
		else if (o instanceof Character)
		    buf[i+2]=(byte) ((Character) args[i]).charValue();
		else if (o instanceof Integer)
		    buf[i+2]=((Integer) args[i]).byteValue();
		else
		    assert(false);
	    }

	for (int i=0;i<blen;i++)
	    {
		buf[args.length+2+i]=b[i];
	    }

	buf[buf.length-1]=computeChecksum(buf);
    }

    public OrcPacket(Object... args)
    {
	buf=new byte[args.length+3]; // plus header, length, checksum
	buf[0]=(byte) 237;
	buf[1]=(byte) (args.length+3);

	for (int i=0;i<args.length;i++)
	    {
		Object o=args[i];

		if (o instanceof Byte)
		    buf[i+2]=(Byte) args[i];
		else if (o instanceof Character)
		    buf[i+2]=(byte) ((Character) args[i]).charValue();
		else if (o instanceof Integer)
		    buf[i+2]=((Integer) args[i]).byteValue();
		else
		    assert(false);
	    }

	buf[buf.length-1]=computeChecksum(buf);
    }

    public static OrcPacket fromInputStream(InputStream ins) throws IOException
    {
	OrcPacket p=new OrcPacket();
	int len=0;
	int c;

	// wait for the signature byte
	do {
	    c=ins.read();
	    if (c<0)
		return null;
	} while (c!=237);

	// read the length
	len=ins.read()&0x00ff;

	if (len==0)
	    return null;

	p.buf=new byte[len];
	p.buf[0]=(byte) 237;
	p.buf[1]=(byte) (len&0x00ff);

	ins.read(p.buf, 2, len-2);

	byte chk=computeChecksum(p.buf);

	if (chk!=p.buf[len-1])
	    return null;

	return p;
    }

    public void write(OutputStream outs) throws IOException
    {
	outs.write(buf);
    }

    /** Computes checksum (ignoring the last byte) **/
    protected static byte computeChecksum(byte[] buf)
    {
	int chk=0;
	for (int i=0;i<buf.length-1;i++)
	    {
		chk = (chk<<1) + (buf[i]&0x00ff) + ((chk&0x80)>0 ? 1 : 0);
	    }
	return (byte) (chk&0x00ff);
    }

    public boolean isSuccess()
    {
	return (buf!=null && buf.length>4 && buf[3]==0);
    }

    public String toString()
    {
	StringBuffer sb=new StringBuffer();

	//	sb.append("OrcPacket: \n");

	if (buf==null)
	    {
		sb.append("null");
		return sb.toString();
	    }

	//	sb.append("Length  : "+(buf[1]&0x00ff)+"\n");
	//	sb.append("Flags   : "+(buf[2]&0x00ff)+"\n");
	//	sb.append("Payload : 0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F\n");

	//	sb.append("          ");
	sb.append("Payload:  ");
	for (int i=3;i<buf.length-1;i++)
	    {
		sb.append(""+(StringUtil.formatHex(buf[i]&0x00ff,2))+" ");
		if ((i-2)%16==0 && i>2)
		    sb.append("\n          ");
	    }
	//	sb.append("\nChecksum: "+StringUtil.formatHex((buf[buf.length-1]&0x00ff),2));
	//	sb.append("\n");

	return sb.toString();
    }

    public void print()
    {
	System.out.println(toString());
    }
}
