package maslab.orcspy;

import maslab.util.*;
import maslab.orc.*;

import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import java.io.*;
import java.util.*;

public class PinPanel extends JPanel implements OrcUpdateListener
{
    public static final long serialVersionUID=1001;

    public enum ServoProfile { RAW, HITEC300, FUTABA100 };

    EnumSet<Orc.PinMode> modes;
    int port;

    JLabel     namelabel;

    JComboBox  modebox;
    JLabel     modelabel;
    JPopupMenu  modemenu;

    JComboBox  digvalue;
    JLabel     diglabel;

    GoalSlider servoslider;
    JComboBox  servoprofile;

    GoalSlider pwmperiod;
    GoalSlider pwmdutycycle;

    GoalSlider dacslider;

    SmallButton    usping;
    JPanel     cardpanel;
    CardLayout cardlayout;

    JLabel     analogfilterlabel;
    JPopupMenu analogfiltermenu;
    
    Color      lightBlue=new Color(200, 200, 255);
    Color      darkBlue=new Color(100, 100, 255);
    Color      yellow=new Color(255, 255, 0);

    Formatter  fmt=new Formatter();

    SignalPlot  plot=new SignalPlot();

    SmallButton digouttoggle;

    static Logger     log=new Logger("Pin");

    static final int INDENT_PX=20;

    Orc         orc;

    int        uspingcount=0;
    
    boolean    minimized=false;

    class PinModeItem extends JMenuItem
    {
	public static final long serialVersionUID=1001;
	public Orc.PinMode mode;

	public PinModeItem(Orc.PinMode m)
	{
	    super(m.toString());

	    this.mode=m;
	}
    }
    
    class AnalogFilterItem extends JMenuItem
    {
	public static final long serialVersionUID=1001;
	
	public int filterMode;

	public AnalogFilterItem(int m)
	{
	    super(""+m);
	    this.filterMode=m;
	}
    }

    public PinPanel(Orc orc, int port, EnumSet<Orc.PinMode> modes) 
    {
	super(new GridBagLayout());

	this.orc=orc;
	this.port=port;
	this.modes=modes;

	GridBagConstraints gbc=new GridBagConstraints();
	GridBagConstraints pgbc=new GridBagConstraints();
	pgbc.weightx=0;	pgbc.weighty=0;
	pgbc.anchor=GridBagConstraints.CENTER; pgbc.fill=GridBagConstraints.BOTH;
	
	Font smallfont=new Font("Dialog",Font.PLAIN, 9);

	/* name row */
	JPanel namerow=new JPanel(new GridBagLayout());
	namerow.setBackground(darkBlue);
	namelabel=new JLabel(" "+port);
	namelabel.setForeground(yellow);
	namelabel.setFont(smallfont);

	gbc.weightx=1;	gbc.weighty=1;
	gbc.anchor=GridBagConstraints.WEST; gbc.fill=GridBagConstraints.NONE;
	gbc.gridx=0; gbc.gridy=0; 
	namerow.add(namelabel, gbc);

	modelabel=new JLabel("");
	modelabel.setFont(smallfont);
	modemenu=new JPopupMenu();

	PinModeItemActionListener pmial=new PinModeItemActionListener();
	for (Orc.PinMode cap: modes)
	    {
		PinModeItem pmi=new PinModeItem(cap);
		pmi.addActionListener(pmial);
		modemenu.add(pmi);
	    }

	
	gbc.anchor=GridBagConstraints.EAST; gbc.fill=GridBagConstraints.NONE;
	gbc.gridx=1; gbc.gridy=0; 
	namerow.add(modelabel, gbc);
	gbc.gridx=2; gbc.gridy=0; 
	namerow.add(new PopupMenuButton(modemenu, lightBlue), gbc);

	pgbc.gridx=0; pgbc.gridy=0; 
	add(namerow,pgbc);

	/* digout toggle row */
	JPanel digoutrow=new JPanel(new FlowLayout(FlowLayout.RIGHT,0,0));
	digouttoggle=new SmallButton("toggle");
	digouttoggle.addActionListener(new DigOutToggleListener());
	digoutrow.add(digouttoggle);
	digoutrow.setBackground(lightBlue);
	
	/* ultrasound ping row */
	JPanel uspingrow=new JPanel(new FlowLayout(FlowLayout.RIGHT,0,0));
	usping=new SmallButton("ping");
	usping.addActionListener(new USPingActionListener());
	uspingrow.add(usping);
	uspingrow.setBackground(lightBlue);

	/* analog row */
	JPanel analogrow=new JPanel(new FlowLayout(FlowLayout.RIGHT,0,0));
	analogfilterlabel=new JLabel("LPF: 0");
	analogfilterlabel.setFont(smallfont);

	analogfiltermenu=new JPopupMenu();
	AnalogFilterActionListener afal=new AnalogFilterActionListener();
	for (int i=0;i<16;i++)
	    {
		AnalogFilterItem afi=new AnalogFilterItem(i);
		analogfiltermenu.add(afi);
		afi.addActionListener(afal);
	    }
	analogrow.add(analogfilterlabel);
	analogrow.add(new PopupMenuButton(analogfiltermenu, Color.white));

	analogrow.setBackground(lightBlue);

	/* DAC row */
	JPanel dacrow=new JPanel(new GridBagLayout());
	gbc.gridx=0; gbc.gridy=0; gbc.anchor=GridBagConstraints.WEST;
	dacrow.add(Box.createHorizontalStrut(INDENT_PX),gbc);

	dacslider=new GoalSlider(0,255,true);
	dacslider.addGoalSliderListener(new DACSliderListener());
	gbc.gridx=1; gbc.gridwidth=2; gbc.gridy=0; gbc.fill=GridBagConstraints.HORIZONTAL;
	gbc.weightx=9999;
	gbc.anchor=GridBagConstraints.WEST;
	dacrow.add(dacslider, gbc);
	dacrow.setBackground(lightBlue);

	/* PWM row */
	JPanel pwmrow=new JPanel(new GridBagLayout());
	gbc.gridx=0; gbc.gridy=0; gbc.anchor=GridBagConstraints.WEST;
	gbc.gridwidth=1; gbc.gridheight=1; gbc.weightx=1;
	pwmrow.add(Box.createHorizontalStrut(INDENT_PX),gbc);
	
	pwmperiod=new GoalSlider(0,255,true);
	pwmperiod.addGoalSliderListener(new PWMSliderListener());
	gbc.gridx=1; gbc.gridwidth=1; gbc.gridy=0; gbc.fill=GridBagConstraints.HORIZONTAL;
	gbc.weightx=9999;
	gbc.anchor=GridBagConstraints.WEST;
	pwmrow.add(pwmperiod, gbc);

	pwmdutycycle=new GoalSlider(0,255,true);
	pwmdutycycle.addGoalSliderListener(new PWMSliderListener());
	gbc.gridx=2; gbc.gridwidth=1; gbc.gridy=-1; gbc.fill=GridBagConstraints.HORIZONTAL;
	gbc.weightx=9999;
	gbc.anchor=GridBagConstraints.WEST;
	pwmrow.add(pwmdutycycle, gbc);

	pwmrow.setBackground(lightBlue);

	/* servo row */
	JPanel servorow=new JPanel(new GridBagLayout());
	gbc.gridx=0; gbc.gridy=0; gbc.anchor=GridBagConstraints.WEST;
	servorow.add(Box.createHorizontalStrut(INDENT_PX),gbc);
	servoslider=new GoalSlider(0,65535,true);
	servoslider.addGoalSliderListener(new ServoGoalSliderListener());

	gbc.gridx=1; gbc.gridwidth=2; gbc.gridy=0; gbc.fill=GridBagConstraints.HORIZONTAL;
	gbc.weightx=9999;
	gbc.anchor=GridBagConstraints.WEST;
	servorow.add(servoslider, gbc);

	servorow.setBackground(lightBlue);
	
	/*********** setup the cards ***********/
	cardlayout=new CardLayout();
	cardpanel=new JPanel(cardlayout);
	cardpanel.add(uspingrow,Orc.PinMode.SONAR_PING.toString());
	cardpanel.add(servorow,Orc.PinMode.SERVO.toString());
	cardpanel.add(digoutrow, Orc.PinMode.DIGITAL_OUT.toString());
	cardpanel.add(analogrow, Orc.PinMode.ANALOG_IN.toString());
	cardpanel.add(dacrow, Orc.PinMode.ANALOG_OUT.toString());
	cardpanel.add(pwmrow, Orc.PinMode.PWM.toString());

	pgbc.gridx=0; pgbc.gridy=1; 
	add(cardpanel,pgbc);

	/* Add the plot */
	plot.setSize(new Dimension(200,15));
	plot.setMinimumSize(new Dimension(100,15));
	plot.setPreferredSize(new Dimension(200,15));

	JPanel pcrow=new JPanel(new GridBagLayout());
	gbc.gridx=0; gbc.gridy=0; gbc.anchor=GridBagConstraints.WEST;
	gbc.weighty=99999;
	gbc.weightx=99999;
	pcrow.add(Box.createHorizontalStrut(INDENT_PX));
	gbc.gridx=1; gbc.gridy=0; gbc.anchor=GridBagConstraints.EAST; 
	gbc.fill=GridBagConstraints.BOTH;
	gbc.insets=new Insets(2, 2, 2, 2);
	pcrow.add(plot,gbc);
	pcrow.setBackground(lightBlue);

	gbc.insets=new Insets(0,0,0,0);
	pgbc.gridx=0; pgbc.gridy=2;
	pgbc.weightx=9999;pgbc.weighty=999; 
	add(pcrow,pgbc);

	cardpanel.setVisible(false);
	updateCards();

	addMouseListener(new MyMouseListener());

	orc.addUpdateListener(this);
    }

    public void addMode(Orc.PinMode m)
    {
	modes.add(m);
    }

    public void orcUpdated(Orc orc)
    {
	Orc.PinMode c=orc.pinModeRead(port);
	modelabel.setText(c.toString());
	
	updateCards();
	
	//Orc.PinMode c=orc.getPinMode(port);
	float f, f2;
	int   i, i2;
	boolean b;

	switch (c)
	    {
	    case ANALOG_IN:
		f = (float) orc.analogRead(port);
		plot.addData(Color.yellow, f/5.0f, StringUtil.formatDouble(f,4)+" V");

		analogfilterlabel.setText("LPF: "+orc.analogLPFRead(port));
		break;

	    case SONAR_PING:
		i = orc.ultrasoundPingCount(port);
		plot.addData(Color.white, (i!=uspingcount) ? 1f : 0f, "");
		uspingcount=i;
		break;

	    case DIGITAL_OUT:
	    case DIGITAL_IN:
	    case DIGITAL_IN_PULLUP:
	    case DIGITAL_IN_PULLDOWN:
		b = orc.digitalRead(port);
		plot.addData(Color.white, b ? 1f : 0f, b ? "1" : "0");
		break;
	       
	    case SONAR_ECHO:
		double d = orc.ultrasoundRange(port);
		plot.addData(lightBlue, (float) (d/500.0), StringUtil.formatDouble(d,1)+" cm");
		break;

	    case SERVO:
		i = orc.servoRead(port);
		f = i/65535.0f;
		servoslider.setActualValue(i);
		if (port<2)
		    {
			f2 = (float) orc.servoCurrentRead(port);
			plot.addData(Color.yellow, f, ""+i, 
				     Color.red, f2,""+((int) (f2*1000.0))+" mA");
		    }
		else
		    {
			plot.addData(Color.yellow, f, ""+i);
		    }
		break;

	    case QUADPHASEFAST:
	    case QUADPHASE:
	    case MONOPHASE:
		i = orc.quadphaseRead(port);
		f = i/65535.0f;
		i2 = orc.quadphaseErrorsRead(port);
		f2 = (i2%16)/16.0f;

		plot.addData(Color.yellow, f,""+i, lightBlue, f2, ""+i2);
		break;

	    case ANALOG_OUT:
		i = orc.dacRead(port);
		dacslider.setActualValue(i);
		f = i/255.0f;
		plot.addData(Color.yellow, f, ""+i);
		break;

	    case PWM:
		pwmperiod.setActualValue(orc.pwmPeriodRead(port));
		pwmdutycycle.setActualValue(orc.pwmDutyCycleRead(port));

		i=orc.pwmPeriodRead(port);
		f=i/255.0f;
		i2=orc.pwmDutyCycleRead(port);
		f2=i2/255.0f;

		plot.addData(Color.yellow, f, ""+i, lightBlue, f2, ""+i2);
		break;
					 
	    }
    }

    void updateCards()
    {
	Orc.PinMode c=orc.pinModeRead(port);
	
	if (c==Orc.PinMode.SERVO || c==Orc.PinMode.DIGITAL_OUT || c==Orc.PinMode.SONAR_PING ||
	    c==Orc.PinMode.ANALOG_IN || c==Orc.PinMode.ANALOG_OUT || c==Orc.PinMode.PWM)
	    {
		cardlayout.show(cardpanel, c.toString());
		cardpanel.setVisible(true);
	    }
	else
	    {
		cardpanel.setVisible(false);
	    }

	return;
    }

    public class AnalogFilterActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    AnalogFilterItem afi=(AnalogFilterItem) e.getSource();

	    orc.analogLPFWrite(port, afi.filterMode);
	}
    }

    public class PinModeItemActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    PinModeItem pmi=(PinModeItem) e.getSource();
	    orc.pinModeWrite(port, pmi.mode);
	    updateCards();
	}
    }

    public class USPingActionListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    orc.ultrasoundPing(port);
	}
    }

    public class DigitalItemListener implements ItemListener
    {
	public void itemStateChanged(ItemEvent e)
	{
	    updateCards();
	}
    }

    public class DigOutToggleListener implements ActionListener
    {
	public void actionPerformed(ActionEvent e)
	{
	    orc.digitalWrite(port, !orc.digitalRead(port));
	}
    }

    public class PWMSliderListener implements GoalSliderListener
    {
	public void goalValueChanged(GoalSlider gs, int goalvalue)
	{
	    orc.pwmWrite(port, 0, pwmperiod.getGoalValue(), pwmdutycycle.getGoalValue());
	}
    }


    public class DACSliderListener implements GoalSliderListener
    {
	public void goalValueChanged(GoalSlider gs, int goalvalue)
	{
	    orc.dacWrite(port, goalvalue);
	}
    }

    public class ServoGoalSliderListener implements GoalSliderListener
    {
	public void goalValueChanged(GoalSlider gs, int goalvalue)
	{
	    orc.servoWrite(port, goalvalue);
	}
    }

    public class ServoProfileItemListener implements ItemListener
    {
	public void itemStateChanged(ItemEvent e)
	{
	    ServoProfile sp=(ServoProfile) servoprofile.getSelectedItem();

	    if (sp==ServoProfile.RAW)
		{
		    servoslider.setMinimum(0);
		    servoslider.setMaximum(65535);
		}
	    else if (sp==ServoProfile.HITEC300)
		{
		    servoslider.setMinimum(20000);
		    servoslider.setMaximum(30000);
		}
	    else if (sp==ServoProfile.FUTABA100)
		{
		    servoslider.setMinimum(15000);
		    servoslider.setMaximum(20000);
		}
	}
    }

    public class MyMouseListener implements MouseListener
    {
	public void mousePressed(MouseEvent e) {}
	
	public void mouseReleased(MouseEvent e) {}
	
	public void mouseEntered(MouseEvent e) {}
	
	public void mouseExited(MouseEvent e) {}
	
	public void mouseClicked(MouseEvent e) {

	    if (e.getClickCount() == 2)
		{
		    minimized=!minimized;

		    cardpanel.setVisible(!minimized);
		    plot.setVisible(!minimized);
		}
	}
    }
    
}
