package maslab.telemetry;

import java.io.*;
import java.util.*;
import java.net.*;

import maslab.telemetry.*;
import maslab.util.*;

/* locking rules:

   You can lock channels, followed by a particular channel

   You cannot acquire any locks nor block if you acquire a channel lock.
*/

/** Connects all clients together, distributing messages according to 
 * subscription and advertisement requests. 
 **/
public class JugHub
{
    protected Logger log=new Logger("JugHub");

    protected class Channel
    {
	Channel(String name)
	{
	    subscribers=new LinkedList<Connection>();
	    publishers=new LinkedList<Connection>();
	    this.name=name;
	}

	// linked lists of Connections
	LinkedList<Connection> subscribers;
	LinkedList<Connection> publishers;
	String name;
    }

    // String -> Channel.
    protected HashMap<String,Channel> channels;

    /** The default port to listen for incoming TCP connections **/
    public static final int DEFAULTPORT=7780;

    protected static int numInstances=0;

    public static JugHub localJugHub=null;

    /** Create a new hub, listening on the default port. **/
    public JugHub()
    {
	channels=new HashMap<String,Channel>();
	channels.put("*",new Channel("*"));

	try {
	    ListenerThread lt=new ListenerThread(DEFAULTPORT);
	    lt.start();
	} catch (Exception ex) {
	    log.error("Unable to listen",ex);
	}

	log.verbose("Hub started.");

	numInstances++;
	if (numInstances>1)
	    log.warn("More than one JugHub instance created!");

	localJugHub=this;
    }

    /** Return the Channel object for a named channel, creating a new
     * one if necessary.
     * @param cname The channel name.
     **/
    protected Channel getChannel(String cname)
    {
	synchronized(channels)
	    {
		Channel c=(Channel) channels.get(cname);
		if (c==null)
		    {
			c=new Channel(cname);
			channels.put(cname,c);
		    }
		return c;
	    }
    }

    /** Send a packet to all the channels subscribed to cname.
     * @param cname The channel to broadcast on, or "*" for broadcast to
     * all clients.
     * @param p The packet to send
     **/
    protected void sendPackets(String cname, JugPacket p)
    {
	Channel c=getChannel(cname);

	synchronized(c)
	    {
		Iterator i=c.subscribers.iterator();
		while (i.hasNext())
		    {
			Connection conn=(Connection) i.next();
			conn.out(p);
		    }
	    }
    }

    protected void sendPacketToPublishers(String cname, JugPacket p)
    {
	Channel c=getChannel(cname);

	synchronized(c)
	    {
		Iterator i=c.publishers.iterator();
		while (i.hasNext())
		    {
			Connection conn=(Connection) i.next();
			conn.out(p);
		    }
	    }
    }

    /** Receives a packet of type COMMAND_MSG from a client in the same JVM.
     **/
    public void directReceive(JugPacket p)
    {
	if (p.command!=JugPacket.COMMAND_MSG)
	    {
		log.error("directReceive called with non message packet!");
		return;
	    }

	p.command=JugPacket.NOTICE_MSG;
	if (p.data==null)
	    log.warn("null message received on "+p.channelName);
	else
	    {
		sendPackets(p.channelName,p);
		log.vverbose("received message "+p.data.length);
	    }
    }

    /** Listen for incoming TCP connections **/
    protected class ListenerThread extends Thread
    {
	ServerSocket serversock;

	public ListenerThread(int port) throws IOException
	{
	    serversock=new ServerSocket(port);
	    setDaemon(true);
	}

	public void run()
	{
	    while(true)
		{
		    Socket sock;

		    try {
			sock=serversock.accept();
			new TCPConnection(sock);
			log.verbose("New incoming connection.");
		    } catch (Exception ex) {
			log.error("Unable to accept",ex);
		    }
		}
	}
    }

    public class TCPConnection extends Connection
    {
	Socket sock;
	MTQueue<JugPacket> outqueue;
	WriterThread writer;
	ReaderThread reader;
	JugPacket EXIT=new JugPacket(0);
	long clockSkew=0;

	public TCPConnection(Socket sock)
	{
	    this.sock=sock;

	    outqueue=new MTQueue<JugPacket>();
	    try {
		writer=new WriterThread();
		writer.start();
		reader=new ReaderThread();
		reader.start();

		init();
	    } catch (IOException ex) {
		log.error("Unable to create writer/reader",ex);
	    }
	}

	/** JugHub calls this method when it wishes to send a message **/
	public void out(JugPacket p)
	{
	    /* hack for MASLAB: drop frames */
	    if (outqueue.size()>10 && p.size()>2048)
		{
		    log.warn("dropping packet of size "+p.size());
		    return;
		}

	    if (outqueue.size()>1000)
		{
		    log.error("queue overflow. dropping packet.");
		    return;
		}
	    outqueue.put(p);
	}

	protected class WriterThread extends Thread
	{
	    DataOutputStream outs;

	    public WriterThread() throws IOException
	    {
		setDaemon(true);

		outs=new DataOutputStream(new BufferedOutputStream(sock.getOutputStream()));
	    }

	    public void run()
	    {
		while(true)
		    {
			Object o=outqueue.getBlock();

			if (o==EXIT)
			    break;

			try {
			    if (o instanceof byte[])
				outs.write((byte[]) o);
			    else if (o instanceof String)
				outs.writeBytes((String) o);
			    else if (o instanceof JugPacket)
				((JugPacket) o).write(outs);
			    else
				log.error("Unknown type "+o.getClass().getName());

			    log.vverbose("sent message");
			} catch (Exception ex) {
			    // assume connection closed.
			    break;
			}
		    }

		try {
		    sock.close();
		} catch (Exception ex) {
		    log.error("Unable to close socket",ex);
		}

		// clean up all of our outqueues: remove them from
		// all channel subscription lists.
		synchronized (channels)
		    {
			Iterator i=channels.values().iterator();

			while (i.hasNext())
			    {
				Channel c=(Channel) i.next();
				synchronized(c)
				    {
					c.subscribers.remove(TCPConnection.this);
					c.publishers.remove(TCPConnection.this);
					
					if (c.publishers.size()==0)
					    {
						JugPacket p=new JugPacket(JugPacket.NOTICE_UNADV, c.name);
						sendPackets("*",p);
					    }
					
					// send the appropriate unsub events.
					try {
					    sendAllSubCountMessages(c);
					} catch (IOException ex) {
					    log.warn("IOexception generating unsub");
					}
				    }
			    }
		    }
		log.verbose("disconnected");
	    }
	}

	protected class ReaderThread extends Thread
	{
	    DataInputStream ins;

	    public ReaderThread() throws IOException
	    {
		setDaemon(true);
		ins=new DataInputStream(new BufferedInputStream(sock.getInputStream()));
	    }

	    public void run()
	    {
		while (true)
		    {
			try {
			    JugPacket p=new JugPacket(ins);
			    in(p);
			} catch (IOException ex) {
			    outqueue.put(EXIT);
			    return;
			} catch (Exception ex) {
			    log.warn("Unexpected exception",ex);
			}
		    }
	    }
	}
    }
    
    /** Represents one TCP connection to a client, including reader and writer threads. 
     **/
    public abstract class Connection
    {
	public Connection()
	{
	}

	/** should be called by sub classes after initialization is finished
	 * and connection is established. 
	 **/
	public void init()
	{
	    Channel c=getChannel("*");
	    synchronized(c)
		{
		    c.subscribers.add(this);
		}
	    doList();
	}

	/** JugHub calls this method when it wishes to send a message **/
	public abstract void out(JugPacket p);

	/** Called by a connection when a packet is received from a JugClient **/
	public final void in(JugPacket p) throws IOException
	{
	    DataInputStream dins;
	    DataOutputStream douts;
	    Channel c;
	    
	    switch (p.command)
		{
		case JugPacket.COMMAND_PING:
		    dins=p.getDataInputStream();
		    
		    int nonce=dins.readInt();
		    
		    p=new JugPacket(JugPacket.NOTICE_PONG);
		    douts=p.getDataOutputStream();
		    douts.writeInt(nonce);
		    
		    out(p);
		    break;
		    
		case JugPacket.COMMAND_ADV:
		    c=getChannel(p.channelName);
		    
		    if (c.publishers.size()==0)
			{
			    p.command=JugPacket.NOTICE_ADV;
			    sendPackets("*",p);
			}

		    synchronized (c)
			{
			    if (!c.publishers.contains(Connection.this))
				c.publishers.add(Connection.this);
			}
		    
		    sendSubCountMessage(c, this);
		    break;
		    
		case JugPacket.COMMAND_UNADV:
		    c=getChannel(p.channelName);
		    
		    synchronized(c)
			{
			    if (c.publishers.contains(Connection.this))
				c.publishers.remove(Connection.this);

			    if (c.publishers.size()==0)
				{
				    p.command=JugPacket.NOTICE_UNADV;
				    sendPackets("*",p);
				}
			}

		    
		    break;
		    
		case JugPacket.COMMAND_LIST:
		    doList();
		    break;
		    
		case JugPacket.COMMAND_MSG:
		    p.command=JugPacket.NOTICE_MSG;
		    if (p.data==null)
			log.warn("null message received on "+p.channelName);
		    else
			{
			    sendPackets(p.channelName,p);
			    log.vverbose("received message "+p.data.length);
			}
		    break;
		    
		case JugPacket.COMMAND_SUB:
		    c=getChannel(p.channelName);

		    synchronized(c)
			{
			    if (!c.subscribers.contains(Connection.this))
				c.subscribers.add(Connection.this);
			}
		    sendAllSubCountMessages(c);
		    break;
		    
		case JugPacket.COMMAND_UNSUB:
		    c=getChannel(p.channelName);
		    synchronized(c)
			{
			    c.subscribers.remove(Connection.this);
			}

		    sendAllSubCountMessages(c);
		    break;
		    
		case JugPacket.COMMAND_SUBCOUNT:
		    sendSubCountMessage(getChannel(p.channelName),this);
		    break;
		    
		default:
		    break;
		}
	}
	
	protected void sendSubCountMessage(Channel c, Connection conn) throws IOException
	{
	    JugPacket p=new JugPacket(JugPacket.NOTICE_SUBCOUNT, c.name);
	    DataOutputStream douts=p.getDataOutputStream();
	    douts.writeInt(c.subscribers.size());
	    douts.flush();
	    conn.out(p);
	}
	
	protected void sendAllSubCountMessages(Channel c) throws IOException
	{
	    synchronized(c)
		{
		    Iterator i=c.publishers.iterator();
		    while (i.hasNext())
			{
			    Connection conn=(Connection) i.next();
			    sendSubCountMessage(c, conn);
			}
		}
	}
	
	protected void doList()
	{
	    log.debug("doList");

	    synchronized (channels)
		{
		    Iterator i=channels.keySet().iterator();
		    while (i.hasNext())
			{
			    String cname=(String) i.next();
			    log.debug("doList: "+cname);
			    
			    // don't expose our internal broadcast channel
			    if (cname.equals("*"))
				continue;
			    
			    Channel c=getChannel(cname);
			    synchronized(c)
				{
				    if (c.publishers.size()>0)
					{
					    JugPacket p=new JugPacket(JugPacket.NOTICE_ADV, cname);
					    out(p);
					}
				}
			}
		}
	}
    }

    public class LocalConnection extends JugHub.Connection
    {
	public LocalConnection()
	{
	    init();
	}
	
	public void out(JugPacket p)
	{
	    
	}
	
    }

    /** Create a Hub on the default port and begin listening. **/
    public static void main(String[] args)
    {
	ConsoleLoggerPublisher clp=new ConsoleLoggerPublisher();
	clp.setGlobalLevel(Logger.VDEBUG);

	Logger.addPublisher(clp);

	new JugHub();

	while(true)
	    {
		try {
		    Thread.sleep(1000);
		} catch (Exception ex) {
		}
	    }
    }
}
