package maslab.util;

import java.util.*;
import java.io.*;

/** A simple logging facility designed for all of a program's textual
 * output. Output sources are categorized by "channel names",
 * typically on a per-class basis. Output can be redirected
 * arbitrarily by providing publishers. **/
public class Logger
{
    /** Used only by LoggerPublishers: indicates that no messages
     * should be published. You should not publish messages at this
     * priority.
     **/
    public static final int SILENT=0;

    /** failure has occurred, typically fatal **/
    public static final int ERROR=1;

    /** a recoverable failure has occurred, or something
	suspicious occured. **/
    public static final int WARN=2;

    /** the expected output of the program. **/
    public static final int OUTPUT=3;

    /** occasional status messages **/
    public static final int VERBOSE=4;

    /** more frequent status messages **/
    public static final int VVERBOSE=5;

    /** extensive debugging information **/
    public static final int DEBUG=6;

    /** ludicrous quantities of debug information **/
    public static final int VDEBUG=7;

    protected String name;
    protected static ArrayList<LoggerPublisher> publishers=new ArrayList<LoggerPublisher>();

    public static ConsoleLoggerPublisher consolePublisher=new ConsoleLoggerPublisher();

    public static long startTime=System.currentTimeMillis();

    /** Convert an integer value corresponding to a debug level
     * into a human-readable string, e.g. "VERBOSE".
     * @param level One of the static levels provided by this class.
     * @return a human-readable string, e.g. "VERBOSE".
     **/
    public static String levelToString(int level)
    {
	switch (level)
	    {
	    case 1:
		return "ERROR";
	    case 2:
		return "WARN";
	    case 3:
		return "OUTPUT";
	    case 4:
		return "VERBOSE";
	    case 5:
		return "VVERBOSE";
	    case 6:
		return "DEBUG";
	    case 7:
		return "VDEBUG";
	    default:
		return "?????";
	    }
    }

    static {
	addPublisher(Logger.consolePublisher);
    }

    /** Create a new logger with the specified name. **/
    public Logger(String name)
    {
	this.name=name;
    }

    /** Create a logger which will take its name as the class name of
     * the argument.
     **/
    public Logger(Object o)
    {
	String fullName=o.getClass().getName();
	int idx=fullName.lastIndexOf('.');

	if (idx>=0)
	    name=fullName.substring(idx+1);
	else
	    name=fullName;
    }

    /** Log a message for this channel **/
    public void log(int level, String message)
    {
	log(name, level, message);
    }

    /** Log a message for this channel**/
    public void log(int level, String message, Exception ex)
    {
	log(name, level, message, ex);
    }

    /** A convenience method **/
    public void error(String message, Exception ex)
    {
	log(name, Logger.ERROR, message, ex);
    }

    /** A convenience method **/
    public void error(String message)
    {
	log(name, Logger.ERROR, message);
    }

    /** A convenience method **/
    public void warn(String message, Exception ex)
    {
	log(name, Logger.WARN, message, ex);
    }

    /** A convenience method **/
    public void warn(String message)
    {
	log(name, Logger.WARN, message);
    }

    /** A convenience method **/
    public void output(String message)
    {
	log(name, Logger.OUTPUT, message);
    }

    /** A convenience method **/
    public void output(String message, Exception ex)
    {
	log(name, Logger.OUTPUT, message, ex);
    }

    /** A convenience method **/
    public void verbose(String message)
    {
	log(name, Logger.VERBOSE, message);
    }

    /** A convenience method **/
    public void vverbose(String message)
    {
	log(name, Logger.VVERBOSE, message);
    }

    /** A convenience method **/
    public void debug(String message)
    {
	log(name, Logger.DEBUG, message);
    }

    /** A convenience method **/
    public void vdebug(String message)
    {
	log(name, Logger.VDEBUG, message);
    }

    /** Add a new publisher
     * @param publisher a new LoggerPublisher which will be told about
     * all events.
     **/
    public static void addPublisher(LoggerPublisher publisher)
    {
	synchronized (publishers)
	    {
		if (!publishers.contains(publisher))
		    publishers.add(publisher);
	    }
    }

    /** Remove a publisher that has previously been added.
     * @param publisher The publisher to be removed.
     **/
    public static void removePublisher(LoggerPublisher publisher)
    {
	synchronized (publishers)
	    {
		if (publishers.contains(publisher))
		    publishers.remove(publisher);
	    }
    }

    /** Produce a log message containing information about an exception that
     * occured. Many of the convenience methods ultimately call this function.
     * The convenience methods should typically be used for readability.
     **/
    public static void log(String name, int level, String message, Exception ex)
    {
	StringBuffer sb=new StringBuffer();

	ByteArrayOutputStream bouts=new ByteArrayOutputStream();
	PrintWriter pw=new PrintWriter(bouts);
	ex.printStackTrace(pw);
	pw.flush();

	sb.append(message);
	sb.append("\n");
	sb.append("  Exception: ");

	String[] lines=StringUtil.split(new String(bouts.toByteArray()),"\n");
	for (int i=0;i<lines.length;i++)
	    {
		if (i>0)
		    sb.append("        ");

		sb.append(lines[i]+"\n");
	    }

	log(name, level, sb.toString());
    }

    /** The actual log method. All other log calls ultimately call this method
     * which actually invokes the publishers.
     * @param name The channel name
     * @param level The log level
     * @param message The message
     **/
    public static void log(String name, int level, String message)
    {
	synchronized (publishers)
	    {
		for (LoggerPublisher publisher : publishers)
		    {
			publisher.publish(name, level, message);
		    }
	    }
    }

    public static void main(String args[])
    {
	Logger log=new Logger("foobar");

	try {
	    throwSomething();
	} catch (Exception ex) {
	    log.error("this is the message",ex);
	}

    }

    protected static void throwSomething()
    {
	Integer i=new Integer(5);
	Object o=i;
	String s=(String) o;
    }
}
