package maslab.vis;

import maslab.util.*;
import maslab.data.*;
import maslab.laser.*;
import maslab.geom.*;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.util.*;

/** Displays data for a laser scan. **/
public class VisLaserScan extends VisObject
{
    public double size=6; // in pixels

    public Color color=Color.blue;
    
    public ArrayList<GPoint2D> points;
    public double x=0,y=0,theta=0;
    public float alpha=1.0f;

    public boolean useSlowPaintMode=false;
    public double maxDistance=Double.MAX_VALUE;

    /** Create a new laser scan object.
     */
    public VisLaserScan(Color color, ArrayList<GPoint2D> points, 
			double x, double y, double theta)
    {
	this.color=color;
	this.size=size;
	this.alpha=(float) alpha;

	if (alpha<1.0f)
	    enableSlowPaintMode(3);

	this.x=x;
	this.y=y;
	this.theta=theta;
	this.points=points;
    }

    /** Select whether to use the higher-quality but much slower drawing routines.
     * @param size Size, in pixels, of each dot.
     **/
    public void enableSlowPaintMode(double size)
    {
	useSlowPaintMode=true;
	this.size=(float) size;
    }

    public void paint(VisCanvas vc, Graphics2D g, BufferedImage bi)
    {
	if (useSlowPaintMode) 
	    {
		slowpaint(vc, g, bi);
		return;
	    }

	int nump=points.size();

	AffineTransform t=g.getTransform();
	t.translate(x, y);
	t.rotate(theta);

	int width=bi.getWidth();
	int height=bi.getHeight();

	double d[]=new double[2];
	double dout[]=new double[2];

	int xi,yi;
	float drgb[]=new float[3];
	float srgb[]=new float[3];

	color.getRGBColorComponents(srgb);

	for (int i=0;i<nump;i++)
	    {
		GPoint2D p=(GPoint2D) points.get(i);
		if (p.distanceTo(GPoint2D.ORIGIN)>maxDistance)
		    continue;

		d[0]=p.getX();
		d[1]=p.getY();
		
		t.transform(d,0,dout,0,1);
		xi=(int) dout[0];
		yi=(int) dout[1];

		if (xi>=0 && yi>=0 && xi<width && yi<height)
		    {
			int argb=bi.getRGB(xi,yi);
			Color destcolor=new Color(argb&0x00ffffff);
			destcolor.getRGBColorComponents(drgb);
			int clr=0;

			for (int c=0;c<3;c++)
			    {
				float f=alpha*srgb[c]+(1-alpha)*drgb[c];
				clr=(clr<<8) + ((int) (f*255));
			    }
			clr|=0xff000000;
			bi.setRGB(xi,yi,clr);
		    }
		//		System.out.println(""+xm+" "+ym+" "+xint+" "+yint);
	    }
    }

    public void slowpaint(VisCanvas vc, Graphics2D g, BufferedImage bi)
    {
	g.translate(x, y);
	g.rotate(theta);

	double scale=vc.getScale();
	double sz=size/scale;

	g.setColor(color);
	int nump=points.size();

	Ellipse2D.Double e=new Ellipse2D.Double(0,0,0,0);

	for (int i=0;i<nump;i++)
	    {
		GPoint2D p=(GPoint2D) points.get(i);
		if (p.distanceTo(GPoint2D.ORIGIN)>maxDistance)
		    continue;

		e.x=p.getX()-sz/2;
		e.y=p.getY()-sz/2;
		e.width=sz;
		e.height=sz;
		g.fill(e);
	    }
    }

}
