/*
 * (c) Copyright Meraki Networks 2006
 *
 */
#include <linux/config.h>
#include <linux/interrupt.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/fs.h>
#include <linux/ioport.h>
#include <linux/notifier.h>
#include <linux/reboot.h>
#include <linux/init.h>


extern void sb_watchdog(void *sbh, uint ticks);
extern void *sbh;

static unsigned long wdt_is_open;
#define WATCHDOG_CLOCK	48000000		/* Hz */
// I have no idea how long these numbers actually correspond
// to... this seems to give about 2 seconds..
static unsigned int heartbeat = 0xffffffff;

extern struct notifier_block *reboot_notifier_list;

static void 
bcmwdt_ping(void)
{
	sb_watchdog(sbh, heartbeat);
}

static int 
bcmwdt_open(struct inode *inode, struct file *file)
{
	if (test_and_set_bit(0, &wdt_is_open))
		return -EBUSY;

	printk("bcmwdt: starting watchdog w/timeout %u cycles\n", heartbeat);
	bcmwdt_ping();

	return nonseekable_open(inode, file);
}

static ssize_t 
bcmwdt_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	if (count) {
		bcmwdt_ping();
	}
	return count;
}

static int 
bcmwdt_release(struct inode *inode, struct file *file)
{
	clear_bit(0, &wdt_is_open);
	return 0;
}

static int 
bcmwdt_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		 unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	int __user *p = argp;
	unsigned long new_heartbeat;
	int status = 0;

	static struct watchdog_info ident = {
		.options =		WDIOF_SETTIMEOUT |
					WDIOF_MAGICCLOSE |
					WDIOF_KEEPALIVEPING,
		.firmware_version =	1,
		.identity =		"ar2315",
	};

	switch(cmd)
	{
		default:
			return -ENOIOCTLCMD;
		case WDIOC_GETSUPPORT:
			return copy_to_user(argp, &ident, sizeof(ident)) ? -EFAULT : 0;
		case WDIOC_GETSTATUS:
			//wdt_get_status(&status);
			return put_user(status, p);
		case WDIOC_GETBOOTSTATUS:
			return put_user(0, p);
		case WDIOC_KEEPALIVE:
			bcmwdt_ping();
			return 0;
		case WDIOC_SETTIMEOUT:
			if (get_user(new_heartbeat, p))
				return -EFAULT;

			heartbeat = WATCHDOG_CLOCK * new_heartbeat;

			bcmwdt_ping();
			/* fallthrough */
		case WDIOC_GETTIMEOUT:
			return put_user(heartbeat, p);
	}
}

static struct file_operations bcmwdt_fops = {
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
	.write		= bcmwdt_write,
	.ioctl		= bcmwdt_ioctl,
	.open		= bcmwdt_open,
	.release	= bcmwdt_release,
};

static struct miscdevice bcmwdt_miscdev = {
	.minor	= WATCHDOG_MINOR,
	.name	= "watchdog",
	.fops	= &bcmwdt_fops,
};

static int __init 
bcmwdt_init(void)
{
	int ret = 0;

	printk("%s using heartbeat %u cycles\n", __func__, heartbeat);
	ret = misc_register(&bcmwdt_miscdev);
	if (ret) {
		printk(KERN_ERR "%s: cannot register miscdev on minor=%d (err=%d)\n",
		       __func__, WATCHDOG_MINOR, ret);
		misc_deregister(&bcmwdt_miscdev);
		return ret;
	}
	return ret;
}

static void __exit 
bcmwdt_exit(void)
{
	printk("%s\n", __func__);
	misc_deregister(&bcmwdt_miscdev);
}

module_init(bcmwdt_init);
module_exit(bcmwdt_exit);
