/* Copyright (C) 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gdevlips.c */
/* Canon Laser Writer printer(LIPS II+/III) driver for Ghostscript */
/* ver 1.2   28-Nov-1992 by kura@nacsis.ac.jp */
/* ver 1.3   30-Apr-1993 by kura@nacsis.ac.jp */
/* ver 1.4   05-Sep-1993 by kura@mmp.cl.nec.co.jp (was kura@nacsis.ac.jp) */
/* ver 1.4.1 10-Sep-1993 by kura@mmp.cl.nec.co.jp */

#include "gdevprn.h"
#include "gdevxprn.h"

/* optimize output code */
#define lips_optimize

/* If you want to use LIPS by 7bit encoding, #define lips_7bit. */
/* #define lips_7bit */

/* If you want to use 7bit control set, #define lips_7bit_control. */
/* #define lips_7bit_control */

/* If you want to use 7bit job control command, #define lips_7bit_job. */
#define lips_7bit_job


#ifdef lips_7bit
#define lips_7bit_control
#define lips_7bit_data
#endif


/* If you want to compile with ghostscript-2.4.1, #define gs241 */
#ifdef gs241
#define gdev_prn_raster(pdev)	gdev_mem_bytes_per_scan_line(pdev)
#endif /* gs241 */


#define lips_b_putc(c)	gdev_ext_prn_putc((c), lips_prn)
#define lips_b_puts(s)	gdev_ext_prn_puts((s), lips_prn)
#define lips_setprn(fp)	(lips_prn = (fp))

#ifndef lips_optimize

#define lips_putc(c)	lips_b_putc(c)
#define lips_puts(s)	lips_b_puts(s)

#else

#define lips_setline(i)	(lips_outline = (i))
#define lips_select(i)	(lips_outmode = (i))

#endif

#ifndef lips_7bit_data
#define lips_putx(c)	lips_putc(c)
#endif

#define lips_c_esc(s)	lips_t_putc('\033'), lips_puts(s)
#define lips_c_dcs(s)	lips_t_putc('\220'), lips_puts(s)
#define lips_c_st()	lips_t_putc('\234')
#define lips_c_csi(s)	lips_t_putc('\233'), lips_puts(s)
#define lips_c_hts()	lips_t_putc('\210')
#define lips_c_nel()	lips_t_putc('\205')
#define lips_c_si()	lips_t_putc('\017')
#define lips_c_so()	lips_t_putc('\016')
#define lips_c_cr()	lips_t_putc('\r')
#define lips_c_ff()	lips_t_putc('\f')
#define lips_c_ht()	lips_t_putc('\t')
#define lips_c_is1()	lips_t_putc('\037')
#define lips_c_is2()	lips_t_putc('\036')

#define LEFT_MARGIN	16		/* 16inch/100 */
#define RIGHT_MARGIN	27		/* 27inch/100 */
#define TOP_MARGIN	23		/* 23inch/100 */
#define BOTTOM_MARGIN	27		/* 27inch/100 */
#define TOP_OFFSET	23		/* 23inch/100 */

#define LENGTH_UNIT	100		/* 1/100 inch */

#ifdef lips_7bit_data	/* data format `/':hex, `.':binary */
#define FORMAT_CHAR		'/'
#else
#define FORMAT_CHAR		'.'
#endif

#ifdef lips_optimize
#define OUT_NORMAL	0
#define OUT_LINE1	1
#define OUT_LINE2	2
#endif

#define lips_x_dots(x,mag)	(int)(((long)(x)*(mag))/LENGTH_UNIT)
#define lips_y_dots(y,mag)	(int)(((long)(y)*(mag))/LENGTH_UNIT)

#define lips_setlocate(x,y)	(lips_xlocate = (x),lips_ylocate = (y))

typedef struct {
	char	*name;			/* page format name */
	char	*cmd;			/* page format select command */
	int	 width;			/* page width */
	int	 height;		/* page height */
} page_format;

private enum { lips2, lips2plus, lips3 } lips_mode;

private FILE* lips_prn;
private int lips_dpi;
private int lips_7bit_code = 0;
private int lips_xlocate;
private int lips_ylocate;
private int lips_ymargin = 0;
#ifdef lips_optimize
private int lips_outline = OUT_NORMAL;
private int lips_outmode = OUT_NORMAL;
private long lips_counter1;
private long lips_counter2;
#endif
private page_format* lips_page_format;

/* The device descriptor */
private dev_proc_print_page(lips_print_page);
#define lips_device(name, dpi)	prn_device(prn_ext_resizable_procs, name, \
	82,				/* width_10ths, 8.2" */ \
	117,				/* height_10ths, 11.7" */ \
	dpi,				/* x_dpi */ \
	dpi,				/* y_dpi */ \
	0,0,0,0,			/* margins */ \
	1, lips_print_page)

gx_device_printer gs_lips_device  = lips_device("lips", 240);
gx_device_printer gs_lips2_device = lips_device("lips2", 240);
gx_device_printer gs_lips3_device = lips_device("lips3", 300);
gx_device_printer gs_ht2313_31_device = lips_device("ht2313-31", 400);


/* page format list */
/* unit of width and height = 1/100inch */
private page_format page_formats[] = {
	{ "a4", "14p",  828, 1170 },
#ifndef NOPAPERSIZE
	{ "a5", "16p",  585,  828 },
	{ "a6", "18p",  414,  585 },
	{ "b4", "24p", 1014, 1434 },
	{ "b5", "26p",  717, 1014 },
#endif
};

#define default_page_format	page_formats[0]

#define page_format_num	(sizeof(page_formats)/sizeof(*page_formats))


/* ------ internal routines ------ */
#ifdef lips_optimize
private void
lips_putc(int c)
{
	if (lips_outmode == lips_outline) {
		lips_b_putc(c);
		return;
	}
	switch (lips_outline) {
	case OUT_NORMAL:
		lips_b_putc(c);
		break;
	case OUT_LINE1:
		lips_counter1 ++;
		break;
	case OUT_LINE2:
		lips_counter2 ++;
		break;
	}
}

private void
lips_puts(char* s)
{
	if (lips_outmode == lips_outline) {
		lips_b_puts(s);
		return;
	}
	switch (lips_outline) {
	case OUT_NORMAL:
		lips_b_puts(s);
		break;
	case OUT_LINE1:
		lips_counter1 += strlen(s);
		break;
	case OUT_LINE2:
		lips_counter2 += strlen(s);
		break;
	}
}
#endif

private void
lips_t_putc(int i)
{
	i &= 0xFF;
	if (lips_7bit_code == 0)	/* if 8bit */
		lips_putc(i);
	else if (i >= 0x80 && i <= 0x9E) {
		lips_putc(033);	/* ESC */
		lips_putc(i - 0x40);
	}
	else
		lips_putc(i);
}

#ifdef lips_7bit_data
private void
lips_putx(int i)
{
	static char hex[] = "0123456789abcdef";

	lips_putc(hex[(i >> 4) & 15]);
	lips_putc(hex[i & 15]);
}
#endif

private void
lips_ints(int i)
{
	if (i >= (1 << 6))
		lips_ints(i >> 6);
	i &= (1 << 6) - 1;
	lips_putc(0x40 + i);
}

private void
lips_int(int i)
{
	int plus;

	if (i < 0) {
		i = -i;
		plus = 0;
	}
	else
		plus = 0x10;
	if (i >= (1 << 4))
		lips_ints(i >> 4);
	i &= (1 << 4) - 1;
	lips_putc(0x20 + plus + i);
}

/* ------ command format select ------ */
private void
lips_check_mode(const char* modename)
{
	if (!strcmp(modename, "lips2"))		/* LIPS-II+ */
		lips_mode = lips2plus;
	else if (!strcmp(modename, "lips3"))	/* LIPS-III */
		lips_mode = lips3;
	else					/* LIPS-II, HT-2313-31 */
		lips_mode = lips2;
}

/* ------ page format select ------ */
private void
lips_page_select(gx_device_printer* pdev)
{
#ifndef NOPAPERSIZE
	extern char *gdev_ext_paper_size();
	extern char *gdev_ext_default_paper_size();
	extern int gdev_ext_paper_width;
	extern int gdev_ext_paper_height;
	char *psize;
	int i;

	lips_ymargin = 0;
	psize = gdev_ext_paper_size();
	if (psize != NULL) {
		for (i = 0; i < page_format_num; i ++) {
			if (!strcmp(page_formats[i].name, psize)) {
				int width, height;
				width = (int)((long)gdev_ext_paper_width *
					pdev->x_pixels_per_inch / 10);
				height = (int)((long)gdev_ext_paper_height *
					pdev->y_pixels_per_inch / 10);
				if (width > pdev->width
						 || height > pdev->height)
					break;
				lips_page_format = &page_formats[i];
				lips_ymargin = pdev->height - height;
				return;
			}
		}
	}
	psize = gdev_ext_default_paper_size();
	if (psize != NULL) {
		for (i = 0; i < page_format_num; i ++) {
			if (!strcmp(page_formats[i].name, psize)) {
				lips_page_format = &page_formats[i];
				return;
			}
		}
	}
#endif
	lips_page_format = &default_page_format;
}

private void
lips_move_x(int i)
{
	char buf[30];
	char direction;
	int distance;

	if (i == lips_xlocate)
		return;
	if (i < lips_xlocate) {
		direction = 'j';	/* left */
		distance = lips_xlocate - i;
	}
	else {
		direction = 'a';	/* right */
		distance = i - lips_xlocate;
	}
	if (distance == 1)
		sprintf(buf, "%c", direction);
	else
		sprintf(buf, "%d%c", distance, direction);
	lips_c_csi(buf);
	lips_xlocate = i;
}

private void
lips_move_y(int i)
{
	char buf[30];
	char direction;
	int distance;

	if (i == lips_ylocate)
		return;
	if (i < lips_ylocate) {
		direction = 'k';	/* upper */
		distance = lips_ylocate - i;
	}
	else {
		direction = 'e';	/* lower */
		distance = i - lips_ylocate;
	}
	if (distance == 1)
		sprintf(buf, "%c", direction);
	else
		sprintf(buf, "%d%c", distance, direction);
	lips_c_csi(buf);
	lips_ylocate = i;
}

private void
lips_start()
{
#ifdef lips_optimize
	lips_setline(OUT_NORMAL);
	lips_select(OUT_NORMAL);
	lips_counter1 = lips_counter2 = 0;
#endif

/* Initialize printer (Laser Shot LIPS II/II+/III) */
	lips_c_esc("<");		/* soft reset */

	if (lips_mode != lips2) {	/* if not LIPS II */
		/* JOB END */
		lips_c_dcs("0J"); lips_c_st();

		/* TEXT MODE */
		lips_c_esc("%@");

		/* JOB START */
		if (lips_mode == lips3) {	/* if LIPS III */
			/* JOB START LIPS III, 300dpi, EUC */
			lips_c_dcs("31;300;2J"); lips_c_st();
			lips_dpi = 300;
		}
		else {				/* otherwise LIPS II+ */
			/* JOB START LIPS II+, 240dpi, EUC */
			lips_c_dcs("21;240;2J"); lips_c_st();
			lips_dpi = 240;
		}
		lips_c_esc("<");		/* soft reset */
		/* display message */
		lips_c_dcs("2yPrinting (gs)"); lips_c_st();
	}
	else {
		lips_dpi = 240;
	}

#ifndef lips_7bit_control
	lips_7bit_code = 0;	/* 8bit mode */
#endif

	lips_c_csi(lips_page_format->cmd); /* current page size, portlate */
	lips_c_csi("?1l");		/* auto cr-lf disable */
	lips_c_csi("?2h");		/* auto ff disable */
	lips_c_csi("11h");		/* set size unit mode */
	lips_c_csi("7 I");		/* set size unit = 1dot */
	lips_c_csi("f");		/* move to home position */
	lips_setlocate(0, 64 + lips_y_dots(TOP_OFFSET, lips_dpi));
}

private void
lips_end()
{
#ifdef lips_optimize
	lips_setline(OUT_NORMAL);
#endif

#ifdef lips_7bit_job
	lips_7bit_code = 1;	/* 7bit mode */
#endif
	if (lips_mode == lips2) {
		lips_c_esc("=");
	}
	else {
		lips_c_dcs("0J"); lips_c_st();	/* job end */
	}
}

/* Send the page to the printer. */
private int
lips_print_page(gx_device_printer *pdev, FILE *prn_stream)
{	int line_size = gdev_prn_raster(pdev);
	byte *in = (byte *)gs_malloc(1, line_size, "lips_print_page(in)");
	byte *out = (byte *)gs_malloc(1, line_size+4, "lips_print_page(out)");
	int lnum, xdots, ydots, xbytes, ystart, yend, yoffset, xstart, xoffset;
	byte xmask;
	char outbuf[30];
#ifdef lips_optimize
	int xend, loop;
	byte prev2, cnt2;
#endif

	if ( in == 0 || out == 0 ) {
		if ( in != 0 )
		    gs_free((char *)in, 1, line_size, "lips_print_page(in)");
		return -1;
	}

#if defined(lips_7bit_control) || defined(lips_7bit_job)
	lips_7bit_code = 1;	/* 7bit mode */
#endif

	/* command format selection */
	lips_check_mode(pdev->dname);

	/* page format selection */
	lips_page_select(pdev);		/* lips_page_format was set */

	/* Set printer stream */
	lips_setprn(prn_stream);

	/* Initialize the printer */
	lips_start();

	/* Calculate parameters */
	xdots = lips_x_dots(lips_page_format->width - RIGHT_MARGIN,
						pdev->x_pixels_per_inch);
	xbytes = (xdots + 7) >> 3;
	xmask = 0xff << (-xdots & 7);
	if (xbytes > line_size) {
		xbytes = line_size;
		xdots = xbytes << 3;
		xmask = 0xff;
	}
	ydots = lips_y_dots(lips_page_format->height - TOP_MARGIN,
						pdev->y_pixels_per_inch);
	if (pdev->height >= ydots) {
		yoffset = -lips_ymargin;
		if ((ystart = pdev->height - ydots) < lips_ymargin)
			ystart = lips_ymargin;
	}
	else {
		ystart = 0;
		yoffset = (int)((long)(ydots - pdev->height) * lips_dpi
						/ pdev->y_pixels_per_inch);
		ydots = pdev->height;
	}
	yend = pdev->height
			 - lips_y_dots(BOTTOM_MARGIN, pdev->y_pixels_per_inch);
	if (yend <= ystart)
		goto skip;

	/* Bitmap print set up */
	xoffset = -lips_x_dots(LEFT_MARGIN, lips_dpi); /* x */

	for ( lnum = yend; -- lnum >= ystart; yend -- ) {
		int bytes;
		gdev_prn_copy_scan_lines(pdev, lnum, in, line_size);
		in[xbytes - 1] &= xmask;
		for ( bytes = xbytes; bytes > 0; bytes -- )
			if ( in[bytes - 1] )
				break;
		if ( bytes )
			break;
	}
	for ( ; ystart < yend; ystart ++ ) {
		int bytes;
		gdev_prn_copy_scan_lines(pdev, ystart, in, line_size);
		in[xbytes - 1] &= xmask;
		for ( bytes = xbytes; bytes > 0; bytes -- )
			if ( in[bytes - 1] )
				break;
		if ( bytes )
			break;
	}

	/* check start position of column */
#ifdef lips_optimize
	xstart = xbytes - 1; xend = 0;
	for ( lnum = ystart; lnum < yend; lnum ++ ) {
		int	xpos;
		if ( lnum != ystart ) {
			gdev_prn_copy_scan_lines(pdev, lnum, in, line_size);
			in[xbytes - 1] &= xmask;
		}
		for ( xpos = 0; xpos < xstart; xpos ++ )
			if ( in[xpos] ) {
				xstart = xpos;
				break;
			}
		for ( xpos = xbytes - 1; xpos > xend; xpos -- )
			if ( in[xpos] ) {
				xend = xpos;
				break;
			}
		if ( xstart == 0 && xend == xbytes - 1 )
			break;
	}

	xoffset += xstart * 8;
#else
	xstart = 0; /* xend = xbytes - 1; */
#endif

	lips_move_x(xoffset);
	lips_move_y(yoffset + ystart);

#ifdef lips_optimize
	for (loop = 0; loop < 2; loop ++) {
		if ( loop ) {
			if (lips_mode == lips2)
				break;
			lips_setline(OUT_LINE2);
			sprintf(outbuf, "%ld;%d;;9;%d%cr", /* always compress */
				lips_counter2, xend - xstart + 1,
				yend - ystart, FORMAT_CHAR);
			lips_c_csi(outbuf);
			lips_setlocate(xoffset, yoffset + ystart);

			if (lips_counter1 < lips_counter2)
				lips_select(OUT_LINE1);
			else {
				lips_select(OUT_LINE2);
				lips_c_csi(outbuf);
			}
		}
		else if (lips_mode == lips2)
			lips_select(OUT_LINE1);
		prev2 = 0; cnt2 = 0;
#endif
		/* Print lines with 1 pixel height of graphics */
		for ( lnum = ystart; lnum < yend; lnum++ ) {
			byte *inp, *inp_end;
			byte *outp, *outp_end;
			byte prev = 0;
			char* p;
			int bytes;
			int cnt = 0;

			/* scan one line */
			gdev_prn_copy_scan_lines(pdev, lnum, in, line_size);
			in[xbytes - 1] &= xmask;

#ifdef lips_optimize
			lips_setline(OUT_LINE2);
			inp = in + xstart; inp_end = in + xend;
			while ( inp <= inp_end ) {
				if ( prev2 == *inp )
					cnt2 ++;
				else {
					if ( cnt2 > 1 )
						lips_putx(cnt2 - 2);
					prev2 = *inp;
					cnt2 = 1;
				}
				if ( cnt2 <= 2 )
					lips_putx(*inp);
				else if ( cnt2 == 255 ) {
					lips_putx(cnt2 - 2);
					cnt2 = 0;
				}
				inp ++;
			}

			lips_setline(OUT_LINE1);
#endif

			for ( bytes = xbytes; bytes > xstart; bytes -- )
				if ( in[bytes - 1] )
					break;
			if ( bytes == xstart )
				continue;

			bytes -= xstart;
			inp = in + xstart; inp_end = inp + bytes;
			outp = out + xstart; outp_end = outp + bytes;

			/* check compress or not */
			while ( inp < inp_end && outp < outp_end ) {
				if ( prev == *inp )
					cnt ++;
				else {
					if ( cnt > 1 )
						*outp ++ = cnt - 2;
					prev = *inp;
					cnt = 1;
				}
				if ( cnt <= 2 )
					*outp ++ = *inp;
				else if ( cnt == 255 ) {
					*outp ++ = cnt - 2;
					cnt = 0;
				}
				inp ++;
			}
			if ( cnt > 1 )
				*outp ++ = cnt - 2;

			/* print one line */
			lips_move_y(yoffset + lnum); /* y */
			if ( lips_mode != lips2 && outp + 4 < outp_end ) {
				/* use compress mode 3 */
				outp_end = outp;
				outp = out + xstart;
				p = ";;9;1"; /* data compress mode 3 */
			}
			else {
				/* no compress */
				outp_end = inp_end;
				outp = in + xstart;
				p = ""; /* data not-compressed */
			}
			sprintf(outbuf, "%d;%d%s", outp_end - outp, bytes, p);
			lips_c_csi(outbuf);
			lips_putc(FORMAT_CHAR);
			lips_putc('r');

			while ( outp < outp_end ) {
				lips_putx(*outp++);
			}
		}
#ifdef lips_optimize
		lips_setline(OUT_LINE2);
		if ( cnt2 > 1 )
			lips_putx(cnt2 - 2);
	}
#endif

skip:
	/* Reinitialize the printer */
	lips_end();

	gs_free((char *)in, 1, line_size, "lips_print_page(in)");
	gs_free((char *)out, 1, line_size+4, "lips_print_page(out)");
	return 0;
}
