/* Copyright (C) 1990, 1992 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gdevp201.c */
/* NEC PC-PR201 printer driver for Ghostscript */
#ifdef PC9801
#include <pc.h>
#endif
#include "gdevprn.h"

private dev_proc_print_page(pr201_print_page);

/*
 * The only available resolutions are 160 x 160.
 */

/* The device descriptor */
gx_device_printer gs_pr201_device =
  prn_device(prn_std_procs, "pr201",
	80,			/* width_10ths, 8" */
	110,			/* height_10ths, 11" */
	160,			/* x_dpi */
	160,			/* y_dpi */
	0,0,0,0,		/* margins */
	1, pr201_print_page);

/* ---- Printer output routines ---- */

#ifdef PC9801
private int
is_printer(gx_device_printer *pdev)
{
	return (strlen(pdev->fname) == 0 || strcmp(pdev->fname, "PRN") == 0);
}

private void
pc98_prn_out(int c)
{
	while(!(inportb(0x42) & 0x04));
	outportb(0x40, c);
	outportb(0x46, 0x0e);
	outportb(0x46, 0x0f);
}
#endif

private int
prn_putc(gx_device_printer *pdev, int c)
{
#ifdef PC9801
	if(is_printer(pdev)) {
		pc98_prn_out(c);
		return 0;
	}
#endif
	return fputc(c, pdev->file);
}

private int
prn_puts(gx_device_printer *pdev, char *ptr)
{
#ifdef PC9801
	if(is_printer(pdev)) {
		while(*ptr)
			pc98_prn_out(*ptr ++);
		return 0;
	}
#endif
	return fputs(ptr, pdev->file);
}

private int
prn_write(gx_device_printer *pdev, char *ptr, int size)
{
#ifdef PC9801
	if(is_printer(pdev)) {
		while(size --)
			pc98_prn_out(*ptr ++);
		return size;
	}
#endif
	return fwrite(ptr, 1, size, pdev->file);
}

private int
prn_flush(gx_device_printer *pdev)
{
#ifdef PC9801
	if(is_printer(pdev))
		return 0;
#endif		
	return fflush(pdev->file);
}

/* ------ internal routines ------ */

/* Transpose a block of 8x8 bits */
private int
pr201_transpose_8x8(byte *src, int src_step, byte *dst, int dst_step)
{
	byte mask, s, d0, d1, d2, d3, d4, d5, d6, d7;
	int i;

	d0 = d1 = d2 = d3 = d4 = d5 = d6 = d7 = 0;

	for(i=0, mask=0x01; i<8; i++, mask <<= 1) {
		s = *src;
		if(s & 0x80) d0 |= mask;
		if(s & 0x40) d1 |= mask;
		if(s & 0x20) d2 |= mask;
		if(s & 0x10) d3 |= mask;
		if(s & 0x08) d4 |= mask;
		if(s & 0x04) d5 |= mask;
		if(s & 0x02) d6 |= mask;
		if(s & 0x01) d7 |= mask;
		src += src_step;
	}

	*dst = d0;
	*(dst += dst_step) = d1;
	*(dst += dst_step) = d2;
	*(dst += dst_step) = d3;
	*(dst += dst_step) = d4;
	*(dst += dst_step) = d5;
	*(dst += dst_step) = d6;
	*(dst += dst_step) = d7;

	return 0;
}

/* Send the page to the printer. */
private int
pr201_print_page(gx_device_printer *pdev, FILE *prn_stream)
{	int line_size = gdev_prn_raster(pdev);
	int height = pdev->height;
	int bits_per_column = 24;
	int bytes_per_column = bits_per_column / 8;
	int chunk_size = bits_per_column * line_size;
	byte *in, *out;
	int lnum, skip;
	char prn_buf[16];

	in = (byte *)
		gs_malloc(bits_per_column, line_size, "pr201_print_page(in)");
	out = (byte *)
		gs_malloc(bits_per_column, line_size, "pr201_print_page(out)");
	if(in == 0 || out == 0)
		return -1;

	/* Initialize printer */
	prn_puts(pdev, "\033cl");	/* Software Reset */
	prn_puts(pdev, "\033P");	/* Proportional Mode */
	prn_puts(pdev, "\033T18");	/* 18/120 inch per line */

	/* Send Data to printer */
	lnum = 0;
	skip = 0;
	while(lnum < height) {
		byte *inp, *outp, *out_beg, *out_end;
		int x, y, num_lines, size, mod;

		/* Copy scan lines */
		if(gdev_prn_copy_scan_lines(pdev, lnum, in, chunk_size) < 0)
			break;

		/* The number of lines to process */
		if((num_lines = height - lnum) > bits_per_column)
			num_lines = bits_per_column;

		/* Test for all zero */
		size = line_size * num_lines;
		if(in[0] == 0 && 
		   !memcmp((char *)in, (char *)in + 1, size - 1)) {
			lnum += bits_per_column;
			skip ++;
			continue;
		}

		/* Fill zero */
		if(num_lines < bits_per_column) {
			size = line_size * (bits_per_column - num_lines);
			memset(in + line_size * num_lines, 0, size);
		}
		lnum += bits_per_column;

		/* Vertical tab to the appropriate position. */
		while(skip > 72) {
			sprintf(prn_buf, "\037%c", 16 + 72);
			prn_puts(pdev, prn_buf);
			skip -= 72;
		}
		if(skip > 0) {
			sprintf(prn_buf, "\037%c", 16 + skip);
			prn_puts(pdev, prn_buf);
		}

		/* Transpose in blocks of 8 scan lines. */
		for(y = 0; y < bytes_per_column; y ++) {
			inp = in + line_size * 8 * y;
			outp = out + y;
			for(x = 0; x < line_size; x ++) {
				pr201_transpose_8x8(inp, line_size,
						    outp, bytes_per_column);
				inp ++;
				outp += bits_per_column;
			}
		}

		/* Remove trailing 0s. */
		out_end = out + chunk_size - 1;
		while(out_end >= out) {
			if(*out_end)
				break;
			out_end --;
		}
		size = (out_end - out) + 1;
		if((mod = size % bytes_per_column) != 0)
			out_end += bytes_per_column - mod;

		/* Remove leading 0s. */
		out_beg = out;
		while(out_beg <= out_end) {
			if(*out_beg)
				break;
			out_beg ++;
		}
		out_beg -= (out_beg - out) % bytes_per_column;

		/* Dot addressing */
		sprintf(prn_buf, "\033F%04d", 
			(out_beg - out) / bytes_per_column);
		prn_puts(pdev, prn_buf);

		/* Dot graphics */
		size = out_end - out_beg + 1;
		sprintf(prn_buf, "\033J%04d", size / bytes_per_column);
		prn_puts(pdev, prn_buf);
		prn_write(pdev, out_beg, size);

		/* Carriage Return */
		prn_putc(pdev, '\r');
		skip = 1;
	}

	/* Form Feed */
	prn_putc(pdev, '\f');
	prn_flush(pdev);

	gs_free((char *)out, 
		bits_per_column, line_size, "pr201_print_page(out)");
	gs_free((char *)in,
		bits_per_column, line_size, "pr201_print_page(in)");

	return 0;
}
