/*
 * zkfzeit.c --- Draw Fonts with Zeit(TM) Vector Font Set
 *
 * Aug.5, 1991 Programmed by N.Katayama (katayama.nacsis.ac.jp)
 * Mar.4, 1992 modifyed   by K-ras
 */

#include <stdio.h>
#include <ctype.h>
#include "math_.h"
#include "memory_.h"
#include "ghost.h"
#include "oper.h"
#include "errors.h"
#include "gspath.h"
#include "gsmatrix.h"
#include "gscoord.h"
#include "gsstate.h"
#include "state.h"
#ifdef X68000
#include <iocslib.h>
#endif

#define D_X_CL	500	/* horizontal center of data */
#define D_Y_CL	472	/* vertical center of data */
#define	D_Y_BL	872	/* base line of data */

#define B_X_CL	500	/* horizontal center in BuildChar */
#define B_Y_CL	400	/* vertical center in BuildChar */
#define B_Y_BL	0	/* base line in BuildChar */

/* Imported procedure */

extern int kf_is_vchar(P1(int));
extern int kf_vmatrix(P5(int, floatp, floatp, gs_rect *, gs_matrix *));

/* Forward declaration */

private floatp map_x(P1(double));
private floatp map_y(P1(double));
private int draw_font(P2(char *, int));

/*
 * zkfzeit
 */

int
zkfzeit(register os_ptr op)
{
	char *buffer;
	int code, len, jis_code, wmode;

	check_type(op[-2], t_integer);		/* JIS Code */
	check_type(op[-1], t_integer);		/* WMode */
	check_type(op[0], t_string);		/* File Name */

	len = r_size(op);
	if((buffer = gs_malloc(len + 1, 1, "zkfzeit")) == 0)
		return e_VMerror;
	memcpy(buffer, (char *)op->value.bytes, len);
	buffer[len] = 0;

	jis_code = op[-2].value.intval;
	wmode = op[-1].value.intval;

	if(wmode && kf_is_vchar(jis_code)) {
		/* vertical kanji character */
		gs_matrix smat, vmat;
		gs_rect bbox;

		if((code = gs_gsave(igs)) < 0)
			return code;
		if((code = draw_font(buffer, jis_code)) < 0)
			return code;
		if((code = gs_flattenpath(igs)) < 0)
			return code;
		if((code = gs_pathbbox(igs, &bbox)) < 0)
			return code;
		if((code = gs_grestore(igs)) < 0)
			return code;

		gs_currentmatrix(igs, &smat);
		kf_vmatrix(jis_code, 
			   (floatp)B_X_CL, (floatp)B_Y_CL, &bbox, &vmat);
		if((code = gs_concat(igs, &vmat)) < 0)
			return code;
		if((code = draw_font(buffer, jis_code)) < 0)
			return code;
		gs_setmatrix(igs, &smat);
	}
	else {
		if((code = draw_font(buffer, jis_code)) < 0)
			return code;
	}

	pop(3);
	gs_free(buffer, len + 1, 1, "zkfzeit");
	return 0;
}

/* -------- Initialization procedure -------- */

op_def zkfzeit_op_defs[] = {
	{"3kfzeit", zkfzeit},
	op_def_end(0)
};

/* -------- Internal routines -------- */

/*
 * Mapping from coordinates of data to coordinates in BuildChar
 */

private floatp
map_x(double x)
{
	static double factor = 
		(double)(B_Y_CL - B_Y_BL) / (double)(D_Y_CL - D_Y_BL);

	return B_X_CL + (x - D_X_CL) * (factor > 0 ? factor : -factor);
}

private floatp
map_y(double y)
{
	static double factor = 
		(double)(B_Y_CL - B_Y_BL) / (double)(D_Y_CL - D_Y_BL);

	return B_Y_CL + (y - D_Y_CL) * factor;
}

/*------------------------------------------------------------------------*/
/*------------ The part below is zeit.h written by T.Kawamoto ------------*/
/*------------------------------------------------------------------------*/

/*
 *	zeit font file library Ver 1.00
 *		for Human68k
 *	Copyright 1991 by Ext(T.Kawamoto)
 *
 *	zeit.h
 */

#include <stdio.h>
#ifdef X68000
#include <stdlib.h>
#include <doslib.h>
#endif

#define Zerror_no_memory    (-100)
#define Zerror_cannot_read  (-101)
#define Zerror_cannot_write (-102)
#define Zerror_end_of_file  (-103)
#define Zerror_no_font      (-104)

typedef struct
{
  int x, y;
} Point;

typedef struct CYCLE
{
  struct CYCLE *next;
  Point point;
} Cycle;

typedef struct FONT
{
  struct FONT *next;
  Cycle *cycle;
} Font;

typedef union FREE
{
  union FREE *next;
  Cycle cycle;
  Font font;
} Free;

#define CODE_SIZE 4418

typedef struct
{
  unsigned char dummy[2];
  unsigned long offsets[CODE_SIZE];
} Zs_Header;

#define Zs_Header_SIZE (2 + (CODE_SIZE) * 4)

/*-----------------------------------------------------------------------*/
/*---------------------------- end of zeit.h ----------------------------*/
/*-----------------------------------------------------------------------*/

/***************************************************
 * Zeit Font Accsess Routines
 *     Original by Ext(T.Kawamoto)
 *     modifyed by K-ras
 **************************************************/
private void  Zdraw_cycle(P1(Cycle *));
private long  Zread_long(P1(FILE *));
private int   Zread_10_bits(P1(FILE *));
private int   Zread_header(P2(FILE *, Zs_Header *));
private Cycle *Zread_cycle(P1(FILE *));
private Font  *Zread_font();
private int   jis2c(P1(int));
private Free  *Zalloc();
private Cycle *Zalloc_cycle();
private Font  *Zalloc_font();
private void  Zfree(P1(Free *));
private void  Zfree_font(P1(Font *));

/*
 * Hash Index
 */

typedef struct index_item_s {
	struct index_item_s *next;
	char      *str;
        FILE      *fd1;
        FILE      *fd2;
        Zs_Header read_header1;
        Zs_Header read_header2;
} index_item;
private index_item *hash_index[256];

/* Hash function */

private int 
hash(unsigned char *str)
{
	unsigned char hash;

	for(hash = 0; *str != 0; str++)
		hash ^= *str;

	return hash;
}
		
/* store */

private int
store(char *str, FILE *fd1, FILE *fd2, Zs_Header *hptr1, Zs_Header *hptr2)
{
	int key = hash((unsigned char *)str);
	index_item *item;

	if((item = (index_item *)malloc(sizeof(index_item))) == 0)
		return e_VMerror;
	if((item->str = (char *)malloc(strlen(str) + 1)) == 0)
		return e_VMerror;
	strcpy(item->str, str);
	item->next = hash_index[key];
	item->fd1 = fd1;
	item->fd2 = fd2;
        memcpy(&(item->read_header1),hptr1,Zs_Header_SIZE);
        memcpy(&(item->read_header2),hptr2,Zs_Header_SIZE);
	hash_index[key] = item;
	return 0;
}

/* search */

private int
search(char *str, index_item **iptr)
{
	int key = hash((unsigned char *)str);
	index_item *item;

	item = hash_index[key];
	while(item != 0) {
		if(strcmp(item->str, str) == 0) {
			*iptr = item;
			return 1;
		}
		item = item->next;
	}
	return 0;
}

/*
 * Open font 
 */

private int 
open_font(char *file, index_item **iptr)
{
	int		code;
        FILE		*wfd1,*wfd2;
        index_item	*wptr;
        Zs_Header	header1,header2;
	char	 	fname[128];

        if(search(file, &wptr)){
                *iptr = wptr;
		return 0;
	} else {
	        strcpy(fname, file);
	        strcat(fname, ".vf1");
		if((wfd1 = fopen(fname,"rb")) == NULL) {
			perror(fname);
			return -1;
		}
	        Zread_header(wfd1,&header1);
	        strcpy(fname, file);
	        strcat(fname, ".vf2");
		if((wfd2 = fopen(fname,"rb")) == NULL) {
			perror(fname);
			return -1;
		}
	        Zread_header(wfd2,&header2);
		if((code = store(file, wfd1, wfd2, &header1, &header2)) < 0)
			return code;
                search(file, &wptr);
                *iptr = wptr;
		return 0;
	}
}

/*
 * moveto
 */
#define moveto(x, y)	gs_moveto(igs, map_x(x), map_y(y))

/*
 * lineto
 */
#define lineto(x, y)	gs_lineto(igs, map_x(x), map_y(y))

/*
 * Draw Font
 */
private int 
draw_font(char *font_name, int jis)
{
        Font       *ptr, *fontptr;
        index_item *iptr;
        float      x,y;

	if(open_font(font_name, &iptr) != 0) {
	        return e_rangecheck;
	}
        if((fontptr = Zread_font(jis, iptr)) == NULL) {
		return 0;
        }

        for(ptr = fontptr; ptr; ptr = ptr->next){
            x = (float)(ptr->cycle->point.x);
            y = (float)(ptr->cycle->point.y);
            moveto(x, y);
            Zdraw_cycle(ptr->cycle);
        }

        Zfree_font(fontptr);

        return 0;
}

/***************************************************
 * Zeit Font Accsess Routines
 *     Original by Ext(T.Kawamoto)
 *     modifyed by K-ras
 **************************************************/
/*
 *	zeit font file library Ver 1.00
 *		for Human68k
 *	Copyright 1991 by Ext(T.Kawamoto)
 */
private void 
Zdraw_cycle(Cycle *ptr)
{
    const Cycle *ptr0;
    float x, y;

        ptr0 = ptr;
        do {
	    x   = (float)(ptr->next->point.x);
	    y   = (float)(ptr->next->point.y);
	    lineto(x , y);
	    ptr = ptr->next;
	} while (ptr != ptr0);
}

private int 
Zread_byte(FILE *fp)
{
        return fgetc(fp);
}

private long
Zread_long(FILE *fp)
{
	unsigned long dummy;

        dummy = Zread_byte(fp);
        dummy |= (unsigned long)Zread_byte(fp) << 8;
        dummy |= (unsigned long)Zread_byte(fp) << 16;
        dummy |= (unsigned long)Zread_byte(fp) << 24;

        return dummy;
}

/****************************************/
/*					*/
/*	seek, read 10 bits		*/
/*					*/
/****************************************/
static short left;
static unsigned long divide;

private void 
Zseek(FILE *fp, long offset)
{
        fseek(fp, offset, 0);
        left       = 0;
        divide = 0;
}

private int 
Zread_10_bits(FILE *fp)
{
        if (left < 10) {
		divide <<= 16;
	        divide |= (unsigned long)Zread_byte(fp);
	        divide |= (unsigned long)Zread_byte(fp) << 8;
		left   += 16;
	}

        left -= 10;
        return (divide >> left) & 0x3ff;
}

/****************************************/
/*					*/
/* Zeit font header index read		*/
/*					*/
/****************************************/
private int 
Zread_header(FILE *fp, Zs_Header *hptr)
{
    int i;

        Zseek(fp, 0);
        hptr->dummy[0] = Zread_byte(fp);
        hptr->dummy[1] = Zread_byte(fp);
        for (i = 0; i < CODE_SIZE; ++i)
	    hptr->offsets[i] = Zread_long(fp);

        return 0;
}

/****************************************/
/*					*/
/*	cycle, font			*/
/*					*/
/****************************************/
private Cycle * 
Zread_cycle(FILE *fp)
{
    int   x0, y0, x, y;
    Cycle *cptr0, *cptr;

        x0 = Zread_10_bits(fp);
        y0 = Zread_10_bits(fp);
        if (x0 == 1023 && y0 == 1023)
	    return NULL;
        if ((cptr0 = cptr = Zalloc_cycle()) == NULL)
	    return cptr;
        cptr->point.x = x0;
        cptr->point.y = y0;
        while (1) {
	    x = Zread_10_bits(fp);
	    y = Zread_10_bits(fp);
	    if (x == 1023 && y == 1023)
		break;
	    if ((cptr->next = Zalloc_cycle()) == NULL)
		return cptr->next;
	    cptr          = cptr->next;
	    cptr->point.x = x;
	    cptr->point.y = y;
	}
        if (cptr->point.x == cptr0->point.x && cptr->point.y == cptr0->point.y) {
	    cptr->next = cptr0->next;
	    Zfree((Free *)cptr0);
	    return cptr;
	} else {
	    cptr->next = cptr0;
	    return cptr0;
	}
}

private Font * 
Zread_font(int code, index_item *iptr)
{
    Font  font, *fptr;
    Cycle *cptr;
    int   code_offset;
    long  seek_offset, no_font_flag;
    FILE  *fp;

        code_offset = jis2c(code);
        if (code < 0x5000) {
	    fp = iptr->fd1;
	    seek_offset  = no_font_flag = iptr->read_header1.offsets[code_offset];
	    seek_offset += Zs_Header_SIZE;
	} else {
	    fp = iptr->fd2;
	    seek_offset  = no_font_flag = iptr->read_header2.offsets[code_offset];
	    seek_offset += Zs_Header_SIZE;
	}
        if (code_offset == 0 || no_font_flag == 0xffffffff)
	    return NULL;

        Zseek(fp, seek_offset);
        fptr = &font;
        while (1) {
	    cptr = Zread_cycle(fp);
	    if (cptr == NULL)
		break;
	    if ((fptr->next = Zalloc_font()) == NULL)
		return fptr->next;
	    fptr        = fptr->next;
	    fptr->cycle = cptr;
	}
        fptr->next = NULL;
        return font.next;
}

private int
jis2c(int code)
{
    int  zscode;

        if ( code < 0x5000 ){
	    zscode = (((code >> 8) & 0xff) - 0x21) * 0x5e + (code & 0xff) - 0x21;
	} else {
	    zscode = (((code >> 8) & 0xff) - 0x50) * 0x5e + (code & 0xff) - 0x21;
	}

        return zscode;
}

/****************************************/
/*					*/
/*	alloc, free			*/
/*					*/
/****************************************/
static Free *free_ptr = NULL;

private Free *
Zalloc(void)
{
    Free *ptr;

        if (free_ptr == NULL) {
	    int i;

	    if ((ptr = (Free *)malloc(sizeof(Free) * 1024)) == NULL) {
		fprintf(stderr, "\ngs: Zalloc malloc error\n");
                exit(0);
            }
            free_ptr = ptr;
            for (i = 0; i < 1023; ++i, ++ptr)
                ptr->next = ptr + 1;
            ptr->next = NULL;
        }
        ptr      = free_ptr;
        free_ptr = free_ptr->next;

        return ptr;
}

private Cycle * 
Zalloc_cycle(void)
{
    Free *ptr;

        ptr = Zalloc();
        if (ptr == NULL)
	    return (Cycle *)ptr;
        return &(ptr->cycle);
}

private Font * 
Zalloc_font(void)
{
    Free *ptr;

        ptr = Zalloc();
        if (ptr == NULL)
	    return (Font *)ptr;
        return &(ptr->font);
}

private void 
Zfree(Free *ptr)
{
        if (ptr == NULL)
	    return;
        ptr->next = free_ptr;
        free_ptr  = ptr;
}

private void 
Zfree_cycle(Cycle *ptr)
{
    Cycle *ptr0, *ptr2;

        if (ptr == NULL)
	    return;
        ptr0 = ptr;
        do {
	    ptr2 = ptr->next;
	    Zfree((Free *)ptr);
	    ptr = ptr2;
	} while (ptr != ptr0);
}

private void 
Zfree_font(Font *ptr)
{
    Font *ptr2;

        while (ptr != NULL) {
	    Zfree_cycle(ptr->cycle);
	    ptr2 = ptr->next;
	    Zfree((Free *)ptr);
	    ptr = ptr2;
	}
}

