/* Copyright (C) 1989, 1990, 1991 Aladdin Enterprises.  All rights reserved.
   Distributed by Free Software Foundation, Inc.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gdevnwp.c */
/* Sony NWP-533/537 driver for GhostScript */

#include "gdevprn.h"
#include <sys/file.h>
#include <sys/ioctl.h>
#include <newsiop/lbp.h>

/* The offset of the drawable area */
#define	LEFT_OFFSET	0.2	/* inch */
#define BOTTOM_OFFSET	0.2	/* inch */

/* The device procedures */
private dev_proc_open_device(nwp533_open);
private dev_proc_open_device(nwp537_open);
private dev_proc_open_device(nwp_open);
private dev_proc_output_page(nwp_output_page);
private dev_proc_close_device(nwp_close);
private dev_proc_get_initial_matrix(nwp_get_initial_matrix);

#define nwp_procs(proc_open) {\
	proc_open, \
	nwp_get_initial_matrix, \
	gx_default_sync_output, \
	nwp_output_page, \
	nwp_close, \
	gdev_prn_map_rgb_color, \
	gdev_prn_map_color_rgb, \
	NULL,	/* fill_rectangle */ \
	NULL,	/* tile_rectangle */ \
	NULL,	/* copy_mono */ \
	NULL,	/* copy_color */ \
	NULL,	/* draw_line */ \
	gx_default_get_bits, \
	gdev_prn_get_props, \
	gdev_prn_put_props \
}

private gx_device_procs nwp533_procs = nwp_procs(nwp533_open);
private gx_device_procs nwp537_procs = nwp_procs(nwp537_open);

/* The device descriptor */
gx_device_printer gs_nwp533_device =
  prn_device(nwp533_procs, "nwp533",
	0, 0,		/* width and height are set in nwp_open */
	0, 0,		/* density is set in nwp_open */
	0,0,0,0,	/* margins */
	1, 0);

gx_device_printer gs_nwp537_device =
  prn_device(nwp537_procs, "nwp537",
	0, 0,		/* width and height are set in nwp_open */
	0, 0,		/* density is set in nwp_open */
	0,0,0,0,	/* margins */
	1, 0);

/* The printer file descriptor */
static int printer_fd = -1;

#define DIAG_NORMAL	"Normal status."
#define DIAG_WAIT	"The fixer is not heated up enough."
#define DIAG_PAUSE	"The fixer and the main motor stopped."
#define DIAG_NO_CR	"Cartridge is not equipped or not set properly."
#define DIAG_NO_PAPER	"Out of paper."
#define DIAG_JAM	"A paper is jamming."
#define DIAG_OPEN	"The top door of the printer is open."
#define DIAG_TEST	"Now, test printing."
#define DIAG_FIXER	"Fixer got some trouble."
#define DIAG_SCANNER	"Printer scanner is out of order."
#define DIAG_MOTOR	"Motor of the printer scanner does not work."

#define STAT_ERROR	-1
#define STAT_WAIT	1
#define	STAT_RETRY	0

/* Analyze the printer status */
private int
analyze_status(struct lbp_stat *pst, char **pdiag)
{
	unsigned char *stat = pst->stat;
	static char diag_none[256];

	if(stat[0] & ST0_WAIT) {
		*pdiag = DIAG_WAIT;
		return STAT_WAIT;
	}
	else if(stat[0] & ST0_PAUSE) {
		*pdiag = DIAG_PAUSE;
		return STAT_WAIT;
	}
	else if(stat[0] & ST0_CALL) {
		if(stat[1] & ST1_NO_CARTRIGE) {
			*pdiag = DIAG_NO_CR;
			return STAT_WAIT;
		}
		else if(stat[1] & ST1_NO_PAPER) {
			*pdiag = DIAG_NO_PAPER;
			return STAT_WAIT;
		}
		else if(stat[1] & ST1_JAM) {
			*pdiag = DIAG_JAM;
			return STAT_WAIT;
		}
		else if(stat[1] & ST1_OPEN) {
			*pdiag = DIAG_OPEN;
			return STAT_WAIT;
		}
		else if(stat[1] & ST1_TEST) {
			*pdiag = DIAG_TEST;
			return STAT_WAIT;
		}
		else if(stat[2] & ST2_FIXER) {
			*pdiag = DIAG_FIXER;
			return STAT_ERROR;
		}
		else if(stat[2] & ST2_SCANNER) {
			*pdiag = DIAG_SCANNER;
			return STAT_ERROR;
		}
		else if(stat[2] & ST2_MOTOR) {
			*pdiag = DIAG_MOTOR;
			return STAT_ERROR;
		}
	}
	else {
		*pdiag = DIAG_NORMAL;
		return STAT_RETRY;
	}

	sprintf(diag_none, 
		"Unexpected status: 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x 0x%02x",
		stat[0], stat[1], stat[2], stat[3], stat[4], stat[5]);
	*pdiag = diag_none;
	return STAT_ERROR;
}

#define True	(1 == 1)
#define False	(0 == 1)

/* return True if should retry - False if should quit */
private int
analyze_error()
{
	struct lbp_stat st;
	char *diag, *pre_diag;
	int status;

	diag = pre_diag = NULL;

	for(;;) {
		if(ioctl(printer_fd, LBIOCRESET, 0) < 0) {
			perror("NWP: ioctl(LBIOCRESET)");
			return False;
		}
		if(ioctl(printer_fd, LBIOCSTATUS, &st) < 0) {
			perror("NWP: ioctl(LBIOCSTATUS)");
			return False;
		}

		status = analyze_status(&st, &diag);

		switch(status) {
		case STAT_RETRY:
			return True;
		case STAT_WAIT:
			if(diag != NULL && diag != pre_diag) {
				fprintf(stderr, "NWP: %s", diag);
				fprintf(stderr, " Waiting ...\n");
				fflush(stderr);
			}
			pre_diag = diag;
			sleep(5);
			break;
		case STAT_ERROR:
			fprintf(stderr, "Error in NWP: %s\n", diag);
			return False;
		}
	}
}

#define ppdev ((gx_device_printer *)pdev)

private int
nwp533_open(gx_device *pdev)
{
	pdev->width = A4_XDOTS;
	pdev->height = A4_YDOTS;
	pdev->x_pixels_per_inch = pdev->y_pixels_per_inch = DPI;

	return nwp_open(pdev);
}

private int
nwp537_open(gx_device *pdev)
{
	pdev->width = B4_XDOTS;
	pdev->height = B4_YDOTS;
	pdev->x_pixels_per_inch = pdev->y_pixels_per_inch = DPI;

	return nwp_open(pdev);
}

/* The original stdout file descriptor */
static int stdout_fd = -1;

private int
nwp_open(gx_device *pdev)
{
	int code;

	if((code = gdev_prn_open(pdev)) < 0)
		return code;

	if(printer_fd >= 0)
		return 0;

	if(strcmp(ppdev->fname, "-") == 0) {

		fprintf(stderr, "NWP: Opening the standard output.\n");

		if(stdout_fd < 0) {
			/* save the original standard output */
			if((stdout_fd = dup(1)) < 0) {
				perror("NWP: dup()");
				return -1;
			}
			/* bind the file descriptor 1 to 2 */
			dup2(2, 1);
		}

		if((printer_fd = dup(stdout_fd)) < 0) {
			perror("NWP: dup()");
			return -1;
		}
	}
	else {
		char *fname = ((strlen(ppdev->fname) == 0) ? 
			       "/dev/lbp" : ppdev->fname);
		
		fprintf(stderr, "NWP: Opening the device (%s).\n", fname);
		
		if((printer_fd = open(fname, O_WRONLY)) < 0) {
			perror("NWP: open()");
			return -1;
		}
	}

	return 0;
}

private int
nwp_close(gx_device *pdev)
{
	fprintf(stderr, "NWP: Closing the device.\n");

	if(printer_fd >= 0) {
		/* Wait for the current printing to finish before close */
		while(ioctl(printer_fd, LBIOCSTOP, 0) < 0) {
			if(!analyze_error())
				break;
		}
		close(printer_fd);
		printer_fd = -1;
	}

	return gdev_prn_close(pdev);
}

/* Send the page to the printer. */
private int
nwp_output_page(gx_device *pdev, int num_copies, int flush)
{
	byte *in;
	int i, lbp_width8, lbp_height;
	uint line_size;
	struct lbp_size lbp_size;

	while(ioctl(printer_fd, LBIOCSTOP, 0) < 0) {
		if(!analyze_error())
			return -1;
	}

	if(ioctl(printer_fd, LBIOCGETSIZE, &lbp_size) < 0) {
		perror("NWP: ioctl(LBIOCGETSIZE)");
		return -1;
	}
	lbp_width8 = lbp_size.lbp_x / 8;
	lbp_height = lbp_size.lbp_y;

	fprintf(stderr, "NWP: Sending data to the device.\n");

	if(lseek(printer_fd, 0, 0) < 0) {
		perror("NWP: lseek()");
		return -1;
	}

	line_size = gdev_prn_raster(ppdev);
	in = (byte *)gs_malloc(line_size, 1, "nwp_output_page(in)");

	if(line_size < lbp_width8 || pdev->height < lbp_height) {
		fprintf(stderr, "NWP: paper size (%d x %d) is too large.\n",
			lbp_width8 * 8, lbp_height);
		return -1;
	}

	for(i=0; i<lbp_height; i++) {
		int lnum = pdev->height - lbp_height + i;
		gdev_prn_copy_scan_lines(ppdev, lnum, in, line_size);
		if(write(printer_fd, (char *)in, lbp_width8) != lbp_width8) {
			perror("NWP: write");
			gs_free((char *)in, line_size, 1, 
				"nwp_output_page(in)");
			return -1;
		}
	}
	gs_free((char *)in, line_size, 1, "nwp_output_page(in)");

	while(ioctl(printer_fd, LBIOCSTART, 0) < 0) {
		if(!analyze_error())
			return -1;
	}

	fprintf(stderr, "NWP: Started printing.\n");

	return 0;
}

/* Get the initial matrix */
private void
nwp_get_initial_matrix(gx_device *dev, gs_matrix *pmat)
{
	pmat->xx = dev->x_pixels_per_inch / 72.0;
	pmat->xy = 0;
	pmat->yx = 0;
	pmat->yy = dev->y_pixels_per_inch / -72.0;
	pmat->tx = - LEFT_OFFSET * dev->x_pixels_per_inch;
	pmat->ty = dev->height + BOTTOM_OFFSET * dev->y_pixels_per_inch;
}
