/*
 * Copyright (c) 1994  Kazushi (Jam) Marukawa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice in the documentation and/or other materials provided with 
 *    the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT 
 * OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
 * BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN 
 * IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * Routines to manipulate the "line buffer".
 * The line buffer holds a line of output as it is being built
 * in preparation for output to the screen.
 */

#include "less.h"

#if ISO

#include <stdio.h>
#include <assert.h>

/*
 * Special variables to specify what codes set is treated.
 */
static int read_iso7;	/* Understand the ISO-2022 7bit code */
static int read_iso8;	/* Understand the ISO-2022 8bit code */
static int output_iso;	/* Display the ISO-2022 7bit code */


#if JAPANESE
/*
 * Special variables to specify what codes set is treated.
 */
static int read_all_kanji;	/* Understand the JIS, UJIS and SJIS codes */
				/* without KANA code */
static int read_ujis;		/* Understand the UJIS code */
static int read_sjis;		/* Understand the SJIS code */
static int output_ujis;		/* Display the UJIS code */
static int output_sjis;		/* Display the SJIS code */

static int first_ujis;	/* If treat all kanji, first we check ujis code */

/*
 * Kanji convetion
 */
#define ISJIS(c)		(0x21 <= (c) && (c) <= 0x7e)
#define ISUJIS(c)		(0xa1 <= (c) && (c) <= 0xfe)
#define ISUJISSS(c)		((c) == 0x8e || (c) == 0x8f)
#define ISUJISKANJI(c1,c2)	(ISUJIS(c1) && ISUJIS(c2))
#define ISUJISKANA(c1,c2)	((c1) == 0x8e && ISUJIS(c2))
#define ISUJISKANA1(c)		((c) == 0x8e)
#define ISUJISKANJISUP(c1,c2,c3) ((c1) == 0x8f && ISUJIS(c2) && ISUJIS(c3))
#define ISSJISKANJI(c1,c2)	(((0x81 <= (c1) && (c1) <= 0x9f) || \
				  (0xe0 <= (c1) && (c1) <= 0xfc)) && \
				 (0x40 <= (c2) && (c2) <= 0xfc && (c2) != 0x7f))
#define ISSJISKANA(c)		(0xa1 <= (c) && (c) <= 0xdf)
#else
static int read_sjis;		/* Dummy variable */
#endif


/*
 * Definitions for understanding the escape sequence.
 * Following escape sequences which be understood by less:
 *  ESC 2/4 2/8,2/9,2/10,2/11,2/13,2/14,2/15 F
 *  ESC 2/4 4/0,4/1,4/2
 *  ESC 2/6 F
 *  ESC 2/8,2/9,2/10,2/11,2/13,2/14,2/15 F
 *  ESC 2/12 F		This is used in MULE.  Less support this as input.
 *  0/14,0/15
 *  ESC 4/14,4/15,6/14,6/15,7/12,7/13,7/14
 *  8/14,8/15
 */
static enum {
	NOESC,		/* No */	ESC_,		/* ^[ */
	ESC_2_4,	/* ^[$ */	ESC_2_4_8,	/* ^[$( */
	ESC_2_4_9,	/* ^[$) */	ESC_2_4_10,	/* ^[$* */
	ESC_2_4_11,	/* ^[$+ */	ESC_2_4_13,	/* ^[$- */
	ESC_2_4_14,	/* ^[$. */	ESC_2_4_15,	/* ^[$/ */
	ESC_2_6,	/* ^[& */	ESC_2_8,	/* ^[( */
	ESC_2_9,	/* ^[) */	ESC_2_10,	/* ^[* */
	ESC_2_11,	/* ^[+ */	ESC_2_12,	/* ^[, */
	ESC_2_13,	/* ^[- */	ESC_2_14,	/* ^[. */
	ESC_2_15	/* ^[/ */
} escape_sequence = NOESC;

/*
 * How to associate each character set with a status in ISO2022.
 *  Defined only well known character set and a types of set.
 */
#define TYPE_94_CHARSET		0x000
#define TYPE_96_CHARSET		0x100
#define TYPE_94N_CHARSET	0x200
#define TYPE_96N_CHARSET	0x300
#define TYPE_MASK		0x300
#define CHARSET_FT_MASK		0x0ff
#define CHARSET_FT_COL_MASK	0x0f0
#define DEF_94_CHARSET		TYPE_94_CHARSET
#define ASCII			(TYPE_94_CHARSET | 'B')
#define JISX0201KANA		(TYPE_94_CHARSET | 'I')
#define JISX0201ROMAN		(TYPE_94_CHARSET | 'J')
#define DEF_96_CHARSET		TYPE_96_CHARSET
#define LATIN1			(TYPE_96_CHARSET | 'A')
#define LATIN2			(TYPE_96_CHARSET | 'B')
#define LATIN3			(TYPE_96_CHARSET | 'C')
#define LATIN4			(TYPE_96_CHARSET | 'D')
#define GREEK			(TYPE_96_CHARSET | 'F')
#define ARABIC			(TYPE_96_CHARSET | 'G')
#define HEBREW			(TYPE_96_CHARSET | 'H')
#define CYRILLIC		(TYPE_96_CHARSET | 'L')
#define LATIN5			(TYPE_96_CHARSET | 'M')
#define DEF_94N_CHARSET		TYPE_94N_CHARSET
#define JISX0208_78KANJI	(TYPE_94N_CHARSET | '@')
#define GB2312			(TYPE_94N_CHARSET | 'A')
#define JISX0208KANJI		(TYPE_94N_CHARSET | 'B')
#define KSC5601			(TYPE_94N_CHARSET | 'C')
#define JISX0212KANJISUP	(TYPE_94N_CHARSET | 'D')
#define DEF_96N_CHARSET		TYPE_96N_CHARSET

#if JAPANESE
/*
 * Other definitions of codes set.  Only input_set use following with
 * above definitions.
 */
#define SJIS			0x400
#define UJIS			0x500
#endif

/*
 * Variables to control of escape sequences as output.
 */
static int disp = ASCII;	/* Current character set */
static int old_disp = ASCII;	/* Last displayed character set */

static int def_g0 = ASCII;	/* Default g0 plane status */
static int def_g1 = ASCII;	/* Default g1 plane status */
static int def_g2 = ASCII;	/* Default g2 plane status */
static int def_g3 = ASCII;	/* Default g3 plane status */
static int g0 = ASCII;		/* Current g0 plane status */
static int g1 = ASCII;		/* Current g1 plane status */
static int g2 = ASCII;		/* Current g2 plane status */
static int g3 = ASCII;		/* Current g3 plane status */
static int *gl = &g0;		/* Current gl plane status */
static int *gr = &g1;		/* Current gr plane status */
static int *sg = NULL;		/* Current status of single-shifted plane */

static int input_set = ASCII;	/* Last non ASCII character set of input */


/*
 * Buffers to keep all bytes of a multi-bytes character until it is
 * proved to be right sequence.
 */
static unsigned char multibuf[10];
static unsigned char multiattr[10];
static int multiindex;		/* Index for multi bytes character. */
static int multilen;		/* The length of multi bytes character. */
static int origlen;		/* The length of multi bytes character */
				/* before converting, and use this to adjust */
				/* a current position if a character doesn't */
				/* fit a screen. */


/*
 * Last position of inputs.  This is used of continues of reading.
 */
static POSITION last_position = -1;	/* last position */


/*
 * Return value for several functions.
 */
typedef enum {
	LAST,		/* Get a last byte of sequence. */
	CONT,		/* Get a right byte for sequence. */
			/* Continue to buffering. */
	WRONG,		/* Get a wrong byte for sequence. */
	WRONGONE	/* Get a wrong byte for sequence. */
			/* However it may be a right byte for other sequence. */
} WHATIS;


	static int
code_length(disp)
	int disp;
{
	switch (disp & TYPE_MASK)
	{
	case TYPE_94_CHARSET:
	case TYPE_96_CHARSET:
		return (1);
	case TYPE_94N_CHARSET:
	case TYPE_96N_CHARSET:
		switch (disp & CHARSET_FT_COL_MASK)
		{
		case 0x40:
		case 0x50:
			return (2);
		case 0x60:
			return (3);
		case 0x70:
			return (4);	/* or more bytes */
		}
	}
	assert(0);
}

	static WHATIS
fix_iso_status()
{
	register int i;

	/*
	 * Check wrong character in ISO 2022 coding.
	 */
	if ((disp & TYPE_MASK) == TYPE_94_CHARSET ||
	    (disp & TYPE_MASK) == TYPE_94N_CHARSET)
	{
		if ((multibuf[0] & 0x7f) == 0x7f)
			return (multiindex == 1 ? WRONGONE : WRONG);
		if ((multibuf[0] & 0x7f) == 0x20)
		{
			if (multiindex != 1)
				return (WRONG);
			multilen = multiindex;
			multiattr[0] = HEAD_W1_CHAR;
			return (LAST);
		}
	}

	/*
	 * Buffering all ISO 2022 coding.  If multi bytes code
	 * finished, flush it.
	 */
	if (multiindex < code_length(disp))
	{
		return (CONT);
	} else if (output_iso)
	{
		multilen = multiindex;
		multibuf[0] &= ~0x80;
		if ((disp & TYPE_MASK) == TYPE_94N_CHARSET ||
		    (disp & TYPE_MASK) == TYPE_96N_CHARSET)
			multiattr[0] = HEAD_W2_CHAR;
		else
			multiattr[0] = HEAD_W1_CHAR;
		for (i = 1; i < multilen; i++)
		{
			multibuf[i] &= ~0x80;
			multiattr[i] = REST_CHAR;
		}
		return (LAST);
	}
#if JAPANESE
	/*
	 * Convert among several Japanese code sets
	 */
	if (multiindex == 1 && (disp == ASCII || disp == JISX0201ROMAN))
	{
		multilen = multiindex;
		multibuf[0] &= ~0x80;
		multiattr[0] = HEAD_W1_CHAR;
		return (LAST);
	} else if (multiindex == 1 && disp == JISX0201KANA)
	{
		if (output_ujis)
		{
			multilen = multiindex = 2;
			multibuf[1] = multibuf[0] | 0x80;
			multiattr[1] = REST_CHAR;
			multibuf[0] = 0x8e;
			multiattr[0] = HEAD_W1_CHAR;
			return (LAST);
		} else if (output_sjis)
		{
			multilen = multiindex;
			multibuf[0] |= 0x80;
			multiattr[0] = HEAD_W1_CHAR;
			return (LAST);
		}
	} else if (multiindex == 2 &&
		   (disp == JISX0208_78KANJI || disp == JISX0208KANJI))
	{
		if (output_ujis)
		{
			multilen = multiindex;
			multibuf[0] |= 0x80;
			multiattr[0] = HEAD_W2_CHAR;
			multibuf[1] |= 0x80;
			multiattr[1] = REST_CHAR;
			return (LAST);
		} else if (output_sjis)
		{
			register int c1, c2;
			multilen = multiindex;
			c1 = ((multibuf[0] & ~0x80) - 0x21) / 2 + 0x81;
			c2 = (multibuf[1] & ~0x80) +
			     ((multibuf[0] & 1) ? 0x40 - 0x21 : 0x9e - 0x21);
			multibuf[0] = c1 + (c1 >= 0xa0 ? 0x40 : 0);
			multiattr[0] = HEAD_W2_CHAR;
			multibuf[1] = c2 + (c2 >= 0x7f ? 1 : 0);
			multiattr[1] = REST_CHAR;
			return (LAST);
		}
	} else if (multiindex == 2 && disp == JISX0212KANJISUP)
	{
		if (output_sjis)
			return (WRONG);
		if (output_ujis)
		{
			multilen = multiindex = 3;
			multibuf[2] = multibuf[1] | 0x80;
			multibuf[1] = multibuf[0] | 0x80;
			multibuf[0] = 0x8f;
			multiattr[0] = HEAD_W2_CHAR;
			multiattr[1] = REST_CHAR;
			multiattr[2] = REST_CHAR;
			return (LAST);
		}
	}
#endif
	if (multiindex != 1)
		return (WRONG);
	if (control_char(multibuf[0]))
		return (WRONGONE);
	multilen = multiindex;
	multiattr[0] = HEAD_W1_CHAR;
	return (LAST);
}

#if JAPANESE
	static WHATIS
fix_ujis_status()
{
	if (multiindex == 1)
		return (CONT);
	else if (multiindex == 2 && ISUJISKANA(multibuf[0], multibuf[1]))
	{
		multiattr[0] = HEAD_W1_CHAR;
		multiattr[1] = REST_CHAR;
		if (output_sjis)
		{
			multilen = multiindex = 1;
			multibuf[0] = multibuf[1];
			return (LAST);
		} else if (output_iso)
		{
			multilen = multiindex = 1;
			multibuf[0] = multibuf[1] & ~0x80;
			return (LAST);
		} else
		{
			multilen = multiindex;
			return (LAST);
		}
	} else if (multiindex == 2 && ISUJISKANJI(multibuf[0], multibuf[1]))
	{
		multiattr[0] = HEAD_W2_CHAR;
		multiattr[1] = REST_CHAR;
		if (output_sjis)
		{
			register int c1, c2;
			multilen = multiindex;
			c1 = ((multibuf[0] & ~0x80) - 0x21) / 2 + 0x81;
			c2 = (multibuf[1] & ~0x80) +
			     ((multibuf[0] & 1) ? 0x40 - 0x21 : 0x9e - 0x21);
			multibuf[0] = c1 + (c1 >= 0xa0 ? 0x40 : 0);
			multibuf[1] = c2 + (c2 >= 0x7f ? 1 : 0);
			return (LAST);
		} else if (output_iso)
		{
			multilen = multiindex;
			multibuf[0] &= ~0x80;
			multibuf[1] &= ~0x80;
			return (LAST);
		} else
		{
			multilen = multiindex;
			return (LAST);
		}
	} else if (multiindex == 2)
	{
		return (CONT);
	} else if (multiindex == 3 &&
		   ISUJISKANJISUP(multibuf[0], multibuf[1], multibuf[2]))
	{
		multiattr[0] = HEAD_W2_CHAR;
		multiattr[1] = REST_CHAR;
		multiattr[2] = REST_CHAR;
		if (output_sjis)
			return (WRONG);
		if (output_iso)
		{
			multilen = multiindex = 2;
			multibuf[0] = multibuf[1] & ~0x80;
			multibuf[1] = multibuf[2] & ~0x80;
			return (LAST);
		} else
		{
			multilen = multiindex;
			return (LAST);
		}
	}
}

	static WHATIS
fix_sjis_status()
{
	if (multiindex == 1)
	{
		if (!ISSJISKANA(multibuf[0]))
			return (LAST);
		multiattr[0] = HEAD_W1_CHAR;
		if (output_ujis)
		{
			multilen = multiindex = 2;
			multibuf[1] = multibuf[0];
			multiattr[1] = REST_CHAR;
			multibuf[0] = 0x8e;
			return (LAST);
		} else if (output_iso)
		{
			multilen = multiindex;
			multibuf[0] &= ~0x80;
			return (LAST);
		} else
		{
			multilen = multiindex;
			return (LAST);
		}
	} else if (multiindex == 2)
	{
		if (!ISSJISKANJI(multibuf[0], multibuf[1]))
			return (WRONG);
		multiattr[0] = HEAD_W2_CHAR;
		multiattr[1] = REST_CHAR;
		if (output_ujis)
		{
			register int c1, c2;
			multilen = multiindex;
			c1 = multibuf[0] - (multibuf[0] >= 0xe0 ? 0x40 : 0);
			c2 = multibuf[1] - (multibuf[1] >= 0x80 ? 1 : 0);
			multibuf[0] = ((c1 - 0x81) * 2 +
				       (c2 >= 0x9e ? 1 + 0x21 : 0x21)) | 0x80;
			multibuf[1] = (c2 - (c2 >= 0x9e ? 0x9e - 0x21 :
							  0x40 - 0x21)) | 0x80;
			return (LAST);
		} else if (output_iso)
		{
			register int c1, c2;
			multilen = multiindex;
			c1 = multibuf[0] - (multibuf[0] >= 0xe0 ? 0x40 : 0);
			c2 = multibuf[1] - (multibuf[1] >= 0x80 ? 1 : 0);
			multibuf[0] = ((c1 - 0x81) * 2 +
				       (c2 >= 0x9e ? 1 + 0x21 : 0x21));
			multibuf[1] = (c2 - (c2 >= 0x9e ? 0x9e - 0x21 :
							  0x40 - 0x21));
			return (LAST);
		} else
		{
			multilen = multiindex;
			return (LAST);
		}
	}
}
#endif

	static WHATIS
what_multi()
{
	int c = multibuf[multiindex - 1];

	if (multiindex == 1)
	{
		disp = ASCII;
		if (c < 0x20)
		{
			return (WRONGONE);
		} else if (c <= 0x7f || read_iso8 && (0xa0 <= c && c <= 0xff))
		{
			/*
			 * Decide current character set.
			 */
			disp = (sg ? *sg : (c & 0x80) ? *gr : *gl);
			if (disp != ASCII)
				input_set = disp;
			return (fix_iso_status());
		} else if (control_char(c))
		{
			return (WRONGONE);
		}
#if JAPANESE
#if SJIS_PRE
		if (!first_ujis && read_sjis && ISSJISKANA(c))
#else
		if (!read_all_kanji && read_sjis && ISSJISKANA(c))
#endif
		{
			disp = JISX0201KANA;
			first_ujis = 0;
			input_set = SJIS;
			return (fix_sjis_status());
		} else if (read_ujis || read_sjis)
		{
			return (CONT);
		}
#endif
		if (read_iso7 || read_iso8)
			return (WRONGONE);
		disp = DEF_96_CHARSET;
		return (fix_iso_status());
	}

	if (c < 0x20)
	{
		return (WRONG);
	} else if (disp != ASCII &&
		   (c <= 0x7f ||
		    read_iso8 && 0xa0 <= c && c <= 0xff))
	{
		if (disp != (sg ? *sg : (c & 0x80) ? *gr : *gl))
			return (WRONG);
		return (fix_iso_status());
	} else if (control_char(c))
	{
		return (WRONG);
	}
#if JAPANESE
#if SJIS_PRE
	if (read_sjis && multiindex == 2 &&
	    ISSJISKANJI(multibuf[0], c) &&
	    (!first_ujis || !ISUJIS(c) || !ISUJIS(multibuf[0])))
#else
	if (read_sjis && multiindex == 2 &&
	    ISSJISKANJI(multibuf[0], c) &&
	    (!first_ujis || !ISUJIS(c) ||
	     (output_sjis ? !ISUJIS(multibuf[0]) && !ISUJISKANA1(multibuf[0]) :
			    !ISUJIS(multibuf[0]) && !ISUJISSS(multibuf[0]))))
#endif
	{
		disp = JISX0208KANJI;
		first_ujis = 0;
		input_set = SJIS;
		return (fix_sjis_status());
	} else if (read_ujis && multiindex == 2)
	{
		if (ISUJISKANA(multibuf[0], c))
		{
			disp = JISX0201KANA;
			first_ujis = 1;
			input_set = UJIS;
			return (fix_ujis_status());
		} else if (ISUJISKANJI(multibuf[0], c))
		{
			disp = JISX0208KANJI;
			first_ujis = 1;
			input_set = UJIS;
			return (fix_ujis_status());
		} else if (ISUJISKANJISUP(multibuf[0], c, 0xa1))
		{
			return (CONT);
		}
	} else if (read_ujis && multiindex == 3 &&
		   ISUJISKANJISUP(multibuf[0], multibuf[1], c))
	{
		disp = JISX0212KANJISUP;
		first_ujis = 1;
		input_set = UJIS;
		return (fix_ujis_status());
	}
#endif
	return (WRONG);
}

/*
 * Check routines 
 */
	static int
check_ft(c, type, plane)
	register int c;
	int type;
	int *plane;
{
	if (0x40 <= c && c <= 0x7e)
	{
		*plane = type | c;
		escape_sequence = NOESC;
		return (0);
	} else
		return (-1);
}

	static void
fix_status_for_escape_sequence()
{
	if (escape_sequence == NOESC)
	{
		switch ((sg ? *sg : *gl) & TYPE_MASK)
		{
		case TYPE_96_CHARSET:
		case TYPE_96N_CHARSET:
			change_control_char(0177, 0);
			break;
		case TYPE_94_CHARSET:
		case TYPE_94N_CHARSET:
			change_control_char(0177, 1);
			break;
		}
		switch ((sg ? *sg : *gr) & TYPE_MASK)
		{
		case TYPE_96_CHARSET:
		case TYPE_96N_CHARSET:
			change_control_char(0377, 0);
			break;
		case TYPE_94_CHARSET:
		case TYPE_94N_CHARSET:
			change_control_char(0377, 1);
			break;
		}
	}
}

	static WHATIS
check_escape_sequence(c)
	register int c;
{
	switch (escape_sequence)
	{
	case ESC_:
		switch (c)
		{
		case '$': escape_sequence = ESC_2_4; break;
		case '&': escape_sequence = ESC_2_6; break;
		case '(': escape_sequence = ESC_2_8; break;
		case ')': escape_sequence = ESC_2_9; break;
		case '*': escape_sequence = ESC_2_10; break;
		case '+': escape_sequence = ESC_2_11; break;
		case ',': escape_sequence = ESC_2_12; break;
		case '-': escape_sequence = ESC_2_13; break;
		case '.': escape_sequence = ESC_2_14; break;
		case '/': escape_sequence = ESC_2_15; break;
		case 'N': sg = &g2; escape_sequence = NOESC; /*SS2*/break;
		case 'O': sg = &g3; escape_sequence = NOESC; /*SS3*/break;
		case 'n': gl = &g2; escape_sequence = NOESC; break;
		case 'o': gl = &g3; escape_sequence = NOESC; break;
		case '|': gr = &g3; escape_sequence = NOESC; break;
		case '}': gr = &g2; escape_sequence = NOESC; break;
		case '~': gr = &g1; escape_sequence = NOESC; break;
		default:  return (WRONG);
		}
		break;
	case ESC_2_4:
		switch (c)
		{
		case '(': escape_sequence = ESC_2_4_8; break;
		case ')': escape_sequence = ESC_2_4_9; break;
		case '*': escape_sequence = ESC_2_4_10; break;
		case '+': escape_sequence = ESC_2_4_11; break;
		case '-': escape_sequence = ESC_2_4_13; break;
		case '.': escape_sequence = ESC_2_4_14; break;
		case '/': escape_sequence = ESC_2_4_15; break;
		case '@':
		case 'A':
		case 'B': if (check_ft(c, TYPE_94N_CHARSET, &g0) == 0) break;
		default:  return (WRONG);
		}
		break;
	case ESC_2_6:
		break;
	case ESC_2_8:
		if (check_ft(c, TYPE_94_CHARSET, &g0) == 0)
			break;
		return (WRONG);
	case ESC_2_9:
		if (check_ft(c, TYPE_94_CHARSET, &g1) == 0)
			break;
		return (WRONG);
	case ESC_2_10:
		if (check_ft(c, TYPE_94_CHARSET, &g2) == 0)
			break;
		return (WRONG);
	case ESC_2_11:
		if (check_ft(c, TYPE_94_CHARSET, &g3) == 0)
			break;
		return (WRONG);
	case ESC_2_12:
		if (check_ft(c, TYPE_96_CHARSET, &g0) == 0)
			break;
		return (WRONG);
	case ESC_2_13:
		if (check_ft(c, TYPE_96_CHARSET, &g1) == 0)
			break;
		return (WRONG);
	case ESC_2_14:
		if (check_ft(c, TYPE_96_CHARSET, &g2) == 0)
			break;
		return (WRONG);
	case ESC_2_15:
		if (check_ft(c, TYPE_96_CHARSET, &g3) == 0)
			break;
		return (WRONG);
	case ESC_2_4_8:
		if (check_ft(c, TYPE_94N_CHARSET, &g0) == 0)
			break;
		return (WRONG);
	case ESC_2_4_9:
		if (check_ft(c, TYPE_94N_CHARSET, &g1) == 0)
			break;
		return (WRONG);
	case ESC_2_4_10:
		if (check_ft(c, TYPE_94N_CHARSET, &g2) == 0)
			break;
		return (WRONG);
	case ESC_2_4_11:
		if (check_ft(c, TYPE_94N_CHARSET, &g3) == 0)
			break;
		return (WRONG);
	case ESC_2_4_13:
		if (check_ft(c, TYPE_96N_CHARSET, &g1) == 0)
			break;
		return (WRONG);
	case ESC_2_4_14:
		if (check_ft(c, TYPE_96N_CHARSET, &g2) == 0)
			break;
		return (WRONG);
	case ESC_2_4_15:
		if (check_ft(c, TYPE_96N_CHARSET, &g3) == 0)
			break;
		return (WRONG);
	case NOESC:
		/*
		 * If this sequences are wrong if currently does buffering.
		 */
		if (multiindex != 1)
		{
			switch (c)
			{
			case 0033:
			case 0016:
			case 0017:
			case 0031: return (WRONG);
			case 0216:
			case 0217: if (!read_sjis) return (WRONG);
			default:   return (WRONGONE);
			}
		}
		switch (c)
		{
		case 0033: escape_sequence = ESC_; break;
		case 0016: gl = &g1; escape_sequence = NOESC; break;
		case 0017: gl = &g0; escape_sequence = NOESC; break;
		case 0031: sg = &g2; escape_sequence = NOESC; /*SS2*/ break;
		case 0216: if (read_sjis) return (WRONGONE);
			   sg = &g2; escape_sequence = NOESC; /*SS2*/ break;
		case 0217: if (read_sjis) return (WRONGONE);
			   sg = &g3; escape_sequence = NOESC; /*SS3*/ break;
		default:   return (WRONGONE);
		}
		break;
	}
	if (escape_sequence == NOESC)
	{
		fix_status_for_escape_sequence();
		return (LAST);
	}
	return (CONT);
}

	static unsigned char*
make_escape_sequence()
{
	static unsigned char p[6];
	int len;

	p[0] = '\033';
	switch (disp & TYPE_MASK)
	{
	case TYPE_94_CHARSET:
		p[1] = '(';
		p[2] = disp & CHARSET_FT_MASK;
		len = 3;
		break;
	case TYPE_94N_CHARSET:
		switch (disp & CHARSET_FT_MASK)
		{
		case '@':
		case 'A':
		case 'B':
			p[1] = '$';
			p[2] = disp & CHARSET_FT_MASK;
			len = 3;
			break;
		default:
			p[1] = '$';
			p[2] = '(';
			p[3] = disp & CHARSET_FT_MASK;
			len = 4;
			break;
		}
		break;
	case TYPE_96_CHARSET:
		p[1] = '-';
		p[2] = disp & CHARSET_FT_MASK;
		len = 3;
		break;
	case TYPE_96N_CHARSET:
		p[1] = '$';
		p[2] = '-';
		p[3] = disp & CHARSET_FT_MASK;
		len = 4;
		break;
	}
	if (read_iso7)
		switch (disp & TYPE_MASK)
		{
		case TYPE_94_CHARSET:
		case TYPE_94N_CHARSET:
			switch (old_disp & TYPE_MASK)
			{
			case TYPE_96_CHARSET:
			case TYPE_96N_CHARSET:
				p[len] = '\017';
				len++;
			}
			break;
		case TYPE_96_CHARSET:
		case TYPE_96N_CHARSET:
			switch (old_disp & TYPE_MASK)
			{
			case TYPE_94_CHARSET:
			case TYPE_94N_CHARSET:
				p[len] = '\016';
				len++;
			}
			break;
		}
	p[len] = '\0';
	return (p);
}

	static MULTI
adjust_charset(multi, strbuf, attrbuf, length)
	MULTI multi;
	register unsigned char **strbuf;
	register unsigned char **attrbuf;
	register int *length;
{
	register int i, len;
	register unsigned char *p;

	if (sg)
	{
		sg = NULL;
		fix_status_for_escape_sequence();
	}

	if (output_iso && read_iso8 &&
	    ((disp & TYPE_MASK) == TYPE_96_CHARSET ||
	     (disp & TYPE_MASK) == TYPE_96N_CHARSET))
		for (i = 0; i < multilen; i++)
			multibuf[i] |= 0x80;
	if (!output_iso || old_disp == disp)
	{
		*strbuf = multibuf;
		*attrbuf = multiattr;
		*length = multilen;
		multilen = 0;
		return (multi);
	}

	p = make_escape_sequence();
	len = strlen((char*)p);
	i = multilen;
	while (--i >= 0)
	{
		multibuf[i + len] = multibuf[i];
		multiattr[i + len] = multiattr[i];
	}
	for (i = 0; i < len; i++)
	{
		multibuf[i] = p[i];
		multiattr[i] = ESCAPE_SEQUENCE;
	}
	old_disp = disp;
	*strbuf = multibuf;
	*attrbuf = multiattr;
	*length = multilen + len;
	multilen = 0;
	if (multi == NO_OUTPUT)
		return (CHARS);
	return (multi);
}

	static MULTI
wrong_multi(multi, strbuf, attrbuf, length)
	MULTI multi;
	unsigned char **strbuf;
	unsigned char **attrbuf;
	int *length;
{
	register int i;

	escape_sequence = NOESC;
	assert(multiindex == origlen);
	multilen = multiindex;
	for (i = 0; i < multiindex; i++)
		if (multibuf[i] < 0x80 && control_char(multibuf[i]))
			multiattr[i] = HEAD_W1_CHAR;
		else
			multiattr[i] = WRONG_CHAR;
	multiindex = 0;
	disp = ASCII;
	return (adjust_charset(multi, strbuf, attrbuf, length));
}

	public void
init_multi(name)
	register char *name;
{
	register int len;

	len = strlen(name);
	/*
	 * Setup some special variables for coding of inputs.
	 */
	if (strncmp(name, "iso7", 4) == 0)
		read_iso7 = 1;
	else if (strncmp(name, "iso8", 4) == 0)
		read_iso8 = 1;
#if JAPANESE
	else if (strncmp(name, "jis", 3) == 0)
		read_iso7 = 1;
	else if (strncmp(name, "ujis", 4) == 0 ||
		 strncmp(name, "euc", 3) == 0)
		read_ujis = 1;
	else if (strncmp(name, "sjis", 4) == 0)
		read_sjis = 1;
	else if (strncmp(name, "japanese", 8) == 0)
	{
		read_all_kanji = 1;
		read_iso7 = 1;
		read_ujis = 1;
		read_sjis = 1;
	}
#endif
	/*
	 * Setup some special variables for coding of outputs.
	 */
	if (len == 4 && strcmp(name, "iso7") == 0 ||
	    strcmp(name + len - 5, "-iso7") == 0)
	{
		read_iso7 = 1;
		output_iso = 1;
	} else if (len == 4 && strcmp(name, "iso8") == 0 ||
		   strcmp(name + len - 5, "-iso8") == 0)
	{
		read_iso8 = 1;
		output_iso = 1;
#if JAPANESE
	} else if (len == 3 && strcmp(name, "jis") == 0 ||
		   strcmp(name + len - 4, "-jis") == 0)
	{
		read_iso7 = 1;
		output_iso = 1;
	} else if (len == 4 && strcmp(name, "ujis") == 0 ||
		   strcmp(name + len - 5, "-ujis") == 0 ||
		   len == 3 && strcmp(name, "euc") == 0 ||
		   strcmp(name + len - 4, "-euc") == 0)
	{
		read_ujis = 1;
		output_ujis = 1;
	} else if (len == 4 && strcmp(name, "sjis") == 0 ||
		   strcmp(name + len - 5, "-sjis") == 0)
	{
		read_sjis = 1;
		output_sjis = 1;
	} else if (len == 8 && strcmp(name, "japanese") == 0)
	{
		output_iso = 1;
#endif
	}

#if JAPANESE
	if (read_iso8 && (read_ujis || read_sjis))
	{
		error("invalid charset name: cannot use iso8 with other 8bit character set", NULL_PARG);
		quit(1);
		/*NOTREACHED*/
	}

	first_ujis = read_ujis ? 1 : 0;
#endif
}

struct planeset {
	char *name;
	char *planeset;
} planesets[] = {
	{ "ascii",		""	},
	{ "ctext",		"\\e-A"	},
	{ "latin1",		"\\e-A"	},
	{ "latin2",		"\\e-B"	},
	{ "latin3",		"\\e-C"	},
	{ "latin4",		"\\e-D"	},
	{ "greek",		"\\e-F"	},
	{ "alabic",		"\\e-G"	},
	{ "hebrew",		"\\e-H"	},
	{ "cyrillic",		"\\e-L"	},
	{ "latin5",		"\\e-M"	},
	{ "japanese",		"\\e$)B\\e*I\\e$+D"	},
	{ "ujis",		"\\e$)B\\e*I\\e$+D"	},
	{ "euc",		"\\e$)B\\e*I\\e$+D"	},
	{ NULL }
};

	public void
set_planeset(name)
	register char *name;
{
	register struct planeset *p;
	WHATIS ret;

	if (name == NULL)
		return;
	for (p = planesets; p->name != NULL; p++)
		if (strcmp(name, p->name) == 0)
		{
			name = p->planeset;
			break;
		}
	multiindex = 1;
	while (*name)
	{
		if (*name == '\\' &&
		    (*(name + 1) == 'e' || *(name + 1) == 'E'))
		{
			ret = check_escape_sequence(033);
			name += 2;
		} else
		{
			ret = check_escape_sequence(*name++);
		}
		if (ret == WRONG || ret == WRONGONE) {
			error("invalid plane set", NULL_PARG);
			quit(1);
			/*NOTREACHED*/
		}
	}
	def_g0 = g0;
	def_g1 = g1;
	def_g2 = g2;
	def_g3 = g3;
}

/*
 * Initialize some status for buffering.
 * If last position is not equal to current position,
 * initialize all plane status.
 */
	public void
prewind_multi(pos)
	POSITION pos;
{
	multiindex = 0;
	multilen = 0;
	if (last_position != pos)
	{
		g0 = def_g0;
		g1 = def_g1;
		g2 = def_g2;
		g3 = def_g3;
		gl = &g0;
		gr = &g1;
		sg = NULL;
		input_set = ASCII;
	}
}

/*
 * Buffering character untile get a guarantee that it is right sequence.
 */
	public MULTI
buffering_multi(c, strbuf, attrbuf, length)
	int c;
	unsigned char **strbuf;
	unsigned char **attrbuf;
	unsigned int *length;
{
	if (c < 0)
	{
		/*
		 * Flush out buffering characters.
		 */
		if (multiindex)
			return (wrong_multi(CHARS, strbuf, attrbuf, length));
		/*
		 * Change the mode to ASCII.
		 */
		disp = ASCII;
		return (adjust_charset(NO_OUTPUT, strbuf, attrbuf, length));
	}

	if (multiindex == 0)
		origlen = 0;
	multibuf[multiindex] = c;
	multiindex++;
	origlen++;

	if (read_iso7 || read_iso8)
	{
		switch (check_escape_sequence(c))
		{
		case LAST:		/* Right escape sequence, */
					/* store the status and dispose */
					/* the sequence. */
			origlen = multiindex = 0;
			return (NO_OUTPUT);
		case CONT:		/* Buffering. */
			return (NO_OUTPUT);
		case WRONGONE:		/* Not escape sequence */
			break;
		case WRONG:		/* Wrong escape sequence, */
					/* write them as binary characters */
			assert(multiindex != 1);
			multiindex--;
			origlen--;
			return (wrong_multi(BIN_CHARS,
					    strbuf, attrbuf, length));
		default:
			assert(0);
		}
	}

	switch (what_multi())
	{
	case LAST:			/* Right multi byte character, */
					/* output them */
		break;
	case CONT: 			/* Buffering */
		return (NO_OUTPUT);
	case WRONGONE:
		assert(multiindex == 1);
		return (wrong_multi(CHARS, strbuf, attrbuf, length));
	case WRONG:			/* Wrong multi byte character, */
					/* write them as binary characters */
		assert(multiindex != 1);
		multiindex--;
		origlen--;
		return (wrong_multi(BIN_CHARS, strbuf, attrbuf, length));
	default:
		assert(0);
	}
	multiindex = 0;
	return (adjust_charset(CHARS, strbuf, attrbuf, length));
}

/*
 * Return the number of buffering characters.
 */
	public POSITION
buffering_bytes()
{
	return (POSITION) origlen;
}

/*
 * Adjust the number of buffering characters.
 */
	public void
adjust_buffering_bytes(outputted_characters)
	int outputted_characters;
{
	origlen -= outputted_characters;
}

/*
 * Save current position to check it when next prewind_multi().
 */
	public void
pdone_multi(pos)
	POSITION pos;
{
	last_position = pos;
}

/*
 * Return string representation about multi bytes character
 * which was buffered.
 */
	public char*
current_set_string()
{
	static char buf[10];

	switch (input_set)
	{
#if JAPANESE
	/*
	 * Code set
	 */
	case SJIS:		return ("SJIS");
	case UJIS:		return ("UJIS");
#endif
	/*
	 * Character set
	 */
	case ASCII:		return ("ASCII");
	case JISX0201KANA:	return ("JIS-KANA");
	case JISX0201ROMAN:	return ("JIS-ROMAN");
	case LATIN1:		return ("LATIN1");
	case LATIN2:		return ("LATIN2");
	case LATIN3:		return ("LATIN3");
	case LATIN4:		return ("LATIN4");
	case GREEK:		return ("GREEK");
	case ARABIC:		return ("ARABIC");
	case HEBREW:		return ("HEBREW");
	case CYRILLIC:		return ("CYRILLIC");
	case LATIN5:		return ("LATIN5");
	case JISX0208_78KANJI:	return ("JIS-78KANJI");
	case GB2312:		return ("GB2312");
	case JISX0208KANJI:	return ("JIS-83KANJI");
	case KSC5601:		return ("KSC5601");
	case JISX0212KANJISUP:	return ("JIS-KANJISUP");
	}
	switch (input_set & TYPE_MASK)
	{
	case TYPE_94_CHARSET:
		strcpy(buf, "94( )");
		buf[3] = input_set & CHARSET_FT_MASK;
		return (buf);
	case TYPE_96_CHARSET:
		strcpy(buf, "96( )");
		buf[3] = input_set & CHARSET_FT_MASK;
		return (buf);
	case TYPE_94N_CHARSET:
		strcpy(buf, "94N( )");
		buf[4] = input_set & CHARSET_FT_MASK;
		return (buf);
	case TYPE_96N_CHARSET:
		strcpy(buf, "96N( )");
		buf[4] = input_set & CHARSET_FT_MASK;
		return (buf);
	}
}

#endif
