/*
 * This file is part of the OLC On-Line Consulting system.
 * It contains the lumberjack program, which gets rid of old logs.
 *
 * Copyright (C) 1990 by the Massachusetts Institute of Technology.
 * For copying and distribution information, see the file "mit-copyright.h".
 *
 *	$Source: /afs/athena.mit.edu/astaff/project/olcdev/highlander/src/server/lumberjack/RCS/lumberjack.c,v $
 *	$Id: lumberjack.c,v 1.23 1997/08/27 23:36:06 bert Exp $
 *	$Author: bert $
 */

#if !defined(lint) && !defined(SABER)
static char rcsid[] ="$Header: /afs/athena.mit.edu/astaff/project/olcdev/highlander/src/server/lumberjack/RCS/lumberjack.c,v 1.23 1997/08/27 23:36:06 bert Exp $";
#endif

#include <mit-copyright.h>

#include <sys/types.h>
#ifndef POSIX
#include <sys/dir.h>
#else
#include <dirent.h>
#include <fcntl.h>
#endif

#if defined(_IBMR2) && defined(ZEPHYR)
/* Conflict in definitions between AIX's dir.h and zephyr.h for STAT; keep */
/* the Zephyr one */
#undef STAT
#endif

#include <sys/types.h>
#include <sys/file.h>
#include <sys/wait.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <syslog.h>

#if !defined(WEXITSTATUS) || !defined(WTERMSIG)
/* BSD, need to define macro to get at exit status */
#define	WEXITSTATUS(st)	(st).w_retcode
#define	WTERMSIG(st)	(st).w_termsig
#endif

#include <olcd.h>		/* needed for SYSLOG_FACILITY */
#include <lumberjack.h>

/*** Support for shipping files to discuss ***/

/* Send off a file to Discuss.
 * Arguments:   fname:   filename for the data
 *              meeting: meeting path ("HOST.MIT.EDU:/usr/spool/discuss/name")
 *              subject: transaction subject
 * Returns: true if the relevant files can be removed.
 */
char send_to_discuss(char *fname, char *meeting, char *subject)
{
#ifdef POSIX
  int status;
#else
  union wait status;
#endif
  int fd;
  char err_buf[SYSLOG_LEN];
  size_t err_len;
  size_t size, i;
  pid_t child, pid;
  int rw[2];
  char *start, *end;

  /* create a pipe that will be used for the child's output. */
  if (pipe(rw) == -1)
    {
      syslog(LOG_CRIT, "pipe: %m (not running dspipe)");
      return 0;
    }

  /* f**k.  or vf**k, if available. */
#ifdef NO_VFORK
  child = fork();
#else
  child = vfork();
#endif
  if (child == -1)
    {
      syslog(LOG_CRIT, "fork: %m");
      return 0;
    }

  if (child == 0)
    {
      /* This is the child.  Set up stdout+err to the pipe, stdin from fname */

      close(rw[0]);	/* close the reading end of the pipe in child */
      if (rw[1] != 1)
	{
	  if (dup2(rw[1], 1) == -1)	/* send stdout into the pipe */
	    {
	      syslog(LOG_CRIT, "dup2(%d,1): %m", rw[1]);
	      return 0;
	    }
	}
      if (rw[1] != 2)
	{
	  if (dup2(rw[1], 2) == -1)	/* send stderr into the pipe */
	    {
	      syslog(LOG_CRIT, "dup2(%d,2): %m", rw[1]);
	      return 0;
	    }
	}
      if ((rw[1] != 1) && (rw[1] != 2))
	close(rw[1]);	/* close original pipe FD after we're done with it */

      fd = open(fname, O_RDONLY);
      if (fd == -1)
	{
	  syslog(LOG_ERR, "open(%s): %m", fname);
	  return 0;
	}
      if (fd != 0)
	{
	  if (dup2(fd, 0) == -1)	/* send logfile to stdin */
	    {
	      syslog(LOG_CRIT, "dup2(%d,0): %m", fd);
	      return 0;
	    }
	  close(fd);
	}

      execl(DSPIPE, DSPAV0, meeting, "-t", subject, NULL);
      /* if we're here, execl failed. */
      syslog(LOG_CRIT, "execl: %m");      
      return 0;
    }

  /* This is the parent.  Deal with the child's output and return status. */

  close(rw[1]);		/* close the writing end of the pipe in parent */

  /* We intentionally read and syslog the output from dspipe in
   * (relatively) small chunks, rather than allocating a dynamically sized
   * buffer, because we wish to enforce a limit on syslog message length
   * even if the line length is very large.  We simply read until EOF and
   * deal with the process later.  (Note: SIGCHLD will cause EINTR.)
   */

  err_len = 0;

  size = read(rw[0], err_buf, SYSLOG_LEN-1);
  while ((size == -1) && ((errno == EAGAIN) || (errno == EINTR)))
    size = read(rw[0], err_buf, SYSLOG_LEN-1);  /* redo on EAGAIN or EINTR */

  while (size > 0)
    {
      /* make all characters printable.  (removes NULs in the process) */
      for (i=0; i<size; i++)
	{
	  if (!isprint(err_buf[err_len+i]) && !isspace(err_buf[err_len+i]))
	    err_buf[err_len+i] = '?';
	}
      err_len += size;
      err_buf[err_len] = '\0';	/* NUL-terminate, so we can use string funcs */

      /* if we have a full line of output, ship it out. */
      for (start=err_buf, end=strchr(start, '\n') ;
	   end ;
	   start=end+1,     end=strchr(start, '\n'))
	{
	  *end = '\0';
	  if (start < end)
	    syslog(LOG_INFO, "dspipe: %s", start);
	}

      /* move the data to the beginning of the buffer. */
      if (start > err_buf)
	{
	  err_len = (err_buf + err_len) - start;
	  memmove(err_buf, start, err_len+1);  /* +1 is for terminating NUL */
	}

      /* if the buffer is full, ship all of it out anyway. */
      if (err_len == SYSLOG_LEN - 1)
	{
	  syslog(LOG_INFO, "dspipe [buffer full]: %s", err_buf);
	  err_len = 0;
	}

      /* read more data */
      size = read(rw[0], err_buf+err_len, (SYSLOG_LEN-1)-err_len);
      while ((size == -1) && ((errno == EAGAIN) || (errno == EINTR)))
	size = read(rw[0], err_buf, (SYSLOG_LEN-1)-err_len);
    }

  err_buf[err_len] = '\0';	/* NUL-terminate, so we can use string funcs */

  /* ship it out whatever's left in the buffer. */
  if (err_len != 0)
    syslog(LOG_INFO, "dspipe: %s", err_buf);

  close(rw[0]);	/* close pipe FD after we're done with it */

  /* get the child's exit status */
  pid = wait(&status);
  while (pid != child)
    {
      if (pid == -1)
	{
	  syslog(LOG_ERR, "wait: %m");
	  if (errno != EINTR)
	    return 0;
	}
      else
	syslog(LOG_ERR, "wait: got data for pid %d, child is pid %d",
	       pid, child);

      pid = wait(&status);
    }

  if (WIFEXITED(status))
    {
      if (WEXITSTATUS(status) == 0)
	return 1;	/* Yay!  We made it! */

      syslog(LOG_ERR, "dspipe exited with status %d", WEXITSTATUS(status));
    }
  else
    {
      /* received a signal */
      syslog(LOG_ERR, "dspipe exited on signal %d", WTERMSIG(status));
    }

  return 0;
}

/*** main ***/

/* Main body of the lumberjack program.
 * Note: This should probably be split up into several smaller, saner
 *    modules.  I did this with parts I made changes to, but it's not done.
 */
int main (int argc, char **argv)
{
  DIR *dirp;			/* pointer to directory */
#ifdef POSIX
  struct dirent *next;		/* directory entry (POSIX) */
#else
  struct direct *next;		/* directory entry (crufty systems) */
#endif
  int lock_fd;			/* file descriptor of lock file */
  FILE *file;			/* file stream used to read control file */
  char *logname;		/* name of log file */
  char *title;			/* title assigned to log */
  char *topic;			/* topic of question, also meeting name */
  char *username;		/* name of person that asked the question */
  char *prefix;			/* prefix for discuss meeting names */

  char *meeting;		/* full name of the Discuss meeting */
  char *subject;		/* subject of the Discuss transaction */

  char *delim;			/* temporary ptr to delimiters in strings */

/*
 *  Set up syslogging
 */

#ifdef ultrix
#ifdef LOG_CONS
  openlog ("lumberjack", LOG_CONS | LOG_PID);
#else
  openlog ("lumberjack", LOG_PID);
#endif /* LOG_CONS */
#else /* not ultrix */
#ifdef LOG_CONS
  openlog ("lumberjack", LOG_CONS | LOG_PID, SYSLOG_FACILITY);
#else
  openlog ("lumberjack", LOG_PID, SYSLOG_FACILITY);
#endif /* LOG_CONS */
#endif /* not ultrix */

/*
 *  Chdir to the directory containing the done'd logs, in case we dump
 *  core or something.
 */

  if (chdir(DONE_DIR))
    {
      syslog(LOG_CRIT, "chdir(%s): %m", DONE_DIR);
      exit(2);
    }

/*
 *  If we can't open/create the lock file and lock it, exit.
 */

  lock_fd = open(LOCKFILE, O_CREAT|O_RDWR, 0666);
  if (lock_fd < 0)
    {
      syslog(LOG_CRIT, "open (lock file): %m");
      exit(2);
    }

  if (do_lock(lock_fd) == -1)
    {
      syslog(LOG_ERR, "getting lock: %m");
      close(lock_fd);
      exit(2);
    }

/*
 * Find out where we're supposed to be putting these logs...
 */
  file = fopen(PREFIXFILE, "r");
  if (file == NULL)
    {
      syslog(LOG_CRIT, "open (prefix file): %m");
      exit(2);
    }
  prefix = get_line(file);
  if (prefix == NULL)
    {
      syslog(LOG_CRIT, "read (prefix file): %m");
      exit(2);
    }
  fclose(file);

  delim = strpbrk(prefix, " \t\n");
  if (delim != NULL)
    *delim = '\0';

/*
 *  Open the directory so we can get the entries out...
 */
  
  dirp = opendir(".");
  if (dirp == NULL)
    {
      syslog(LOG_CRIT, "opendir: %m");
      if (do_unlock(lock_fd) == -1)
	syslog(LOG_WARNING, "clearing lock: %m");
      close(lock_fd);
      exit(2);
    }

/*
 *  Read out the entries and process the ones that begin with "ctrl".
 */

  while ((next = readdir(dirp)) != NULL)
    {
      if (!strncmp(next->d_name, "ctrl", 4))
	{
	  /* Open the control file.  Failure probably means another
	   * lumberjack process got it first.
	   */
	  file = fopen(next->d_name, "r");
	  if (file == NULL)
	    {
	      syslog(LOG_WARNING, "open(%s): %m", next->d_name);
	      continue;
	    }

	  /* Read control data. */
	  logname = get_line(file);
	  if ((logname == NULL) || (*logname == '\0'))
	    {
	      syslog(LOG_ERR, "unable to get logfile name from %s: %m",
		     next->d_name);
	      fclose(file);
	      continue;
	    }
	  title = get_line(file);
	  if (title == NULL)	/* title may, in fact, be an empty string */
	    {
	      syslog(LOG_ERR, "unable to get title from %s: %m",
		     next->d_name);
	      fclose(file);
	      continue;
	    }
	  topic = get_line(file);
	  if ((topic == NULL) || (*topic == '\0'))
	    {
	      syslog(LOG_ERR, "unable to get topic from %s: %m",
		     next->d_name);
	      fclose(file);
	      continue;
	    }
	  username = get_line(file);
	  if ((username == NULL) || (*username == '\0'))
	    {
	      syslog(LOG_ERR, "unable to get username from %s: %m",
		     next->d_name);
	      fclose(file);
	      continue;
	    }

	  fclose(file);

/* If we've made it this far, we've got everything we need to ship to
 * discuss.
 */ 
	  meeting = malloc(strlen(prefix)+strlen(topic)+1);
	  if (meeting == NULL)
	    {
	      syslog(LOG_ERR, "%m in malloc()");
	      exit(1);
	    }
	  sprintf (meeting, "%s%s", prefix, topic);

	  subject = malloc(strlen(username)+strlen(title)+3);
	  if (subject == NULL)
	    {
	      syslog(LOG_ERR, "%m in malloc()");
	      exit(1);
	    }
	  sprintf (subject, "%s: %s", username, title);

	  if (send_to_discuss(logname, meeting, subject))
	    {
	      /* success: remove the log file and the control file. */
	      unlink(logname);
	      unlink(next->d_name);
	    }
	}
    }
  closedir(dirp);

  if (do_unlock(lock_fd) == -1)
    syslog(LOG_WARNING, "clearing lock: %m");
  close(lock_fd);

  exit(0);
}
