/*
 * This file is part of the OLC On-Line Consulting System.  It contains
 * functions for backing up the state of the OLC system.
 *
 *      Win Treese (MIT Project Athena)
 *      Ken Raeburn (MIT Information Systems)
 *      Tom Coppeto, Chris VanHaren, Lucien Van Elsen (MIT Project Athena)
 *
 * Copyright (C) 1988,1990 by the Massachusetts Institute of Technology.
 * For copying and distribution information, see the file "mit-copyright.h".
 *
 *	$Source: /mit/olcdev/highlander/src/server/olcd/RCS/backup.c,v $
 *	$Id: backup.c,v 1.28 1997/08/28 20:18:02 bert Exp bert $
 *	$Author: bert $
 */

#ifndef SABER
#ifndef lint
static char rcsid[] ="$Header: /mit/olcdev/highlander/src/server/olcd/RCS/backup.c,v 1.28 1997/08/28 20:18:02 bert Exp bert $";
#endif
#endif

#include <mit-copyright.h>
#include "olcd.h"

#include <ctype.h>              /* Character type macros. */
#include <signal.h>             /* System signal definitions. */
#include <sys/time.h>           /* System time definitions. */
#include <sys/file.h>           /* System file defs. */
#include <setjmp.h>             /* For string validation kludge */
#include <pwd.h>

#ifdef __STDC__
# define        P(s) s
#else
# define P(s) ()
#endif

#undef P

/*** top-level backup functions ***/

/* Back up the queue in both the binary and the ASCII format.
 * Note: the binary format is byte-order and word-size dependent.
 */

void backup_data(void)
{
  binary_backup_data();

  dump_data(ASCII_BACKUP_TEMP);
  rename(ASCII_BACKUP_TEMP, ASCII_BACKUP_FILE);
}

/* Restore the ASCII backup of the queue.  If it is missing, restore
 * the binary backup instead.
 * Note: the binary format is byte-order and word-size dependent, so
 *       any actual use of the binary backup is discouraged.  Unless
 *       you have a good excuse, at least.
 */

void load_data(void)
{
  if (access(ASCII_BACKUP_FILE, R_OK) == 0)
      {
        if (undump_data(ASCII_BACKUP_FILE) == SUCCESS)
	  return;  /* dump succeeded, we're done. */
      }

  /* ASCII dump doesn't exist or loading it failed. */
  log_error("reading ASCII backup didn't work, trying binary.");
  binary_load_data();
}


/*** things that cause olcd to not go boom in the night ***/

void
  ensure_consistent_state()
{
  register KNUCKLE **k_ptr, *k;
  KNUCKLE *foo;
  char msgbuf[BUFSIZ];
  int status;
  
  for (k_ptr = Knuckle_List; *k_ptr != (KNUCKLE *) NULL; k_ptr++) 
    {
      k = *k_ptr;
      if (k->connected != (KNUCKLE *) NULL) 
	{
	  /* check if connected person exists */
	  status = get_knuckle (k->connected->user->username,
				k->connected->instance, &foo, /* ??? */0);
	  if (status != SUCCESS || k != k->connected->connected) {
	    log_error("Inconsistency: user connected to %s (%d)\n",
		      k->user->username, k->instance);
	    /* problem: the question data is probably screwed due to earlier bugs.
	     * don't try to salvage because addresses to question->owner is in
	     * space. Take the loss on the question data, delete both knuckles,
	     * and continue. This should be looked into.
	     */
	    k->question = (QUESTION *) NULL;
	    delete_knuckle(k, /*???*/0);
	    k->connected->question = (QUESTION *) NULL;
	    k->connected->connected = (KNUCKLE *) NULL;
	    /* eat this too- chances are that if the connected person is */
	    /* invalid, it is total garbage. */
	    
	    /* delete_knuckle(k->connected); */
	    continue;
#if 0
	    if (k->question != (QUESTION *) NULL)
	      {
		if(k->status & QUESTION_STATUS)
		  k->question->owner = k;
		else
		  k->question->owner = (KNUCKLE *) NULL;
		
		if(k->question->owner != (KNUCKLE *) NULL)
		  {
		    if(k->question->owner->connected != (KNUCKLE *) NULL)
		      {
			k->question->owner->connected->connected = 
			  (KNUCKLE *) NULL;  
			write_message_to_user(k->question->owner->connected,
					      "Daemon error -- please reconnect.\n",
					      NO_RESPOND);
		      }
		    k->question->owner->connected = (KNUCKLE *) NULL;
		  }
	      }
#endif
	  }
	}
    }
}

void
  reconnect_knuckles()
{
  KNUCKLE **k_ptr;
  KNUCKLE *k;
  int status; 
  
  for (k_ptr = Knuckle_List; (*k_ptr) != (KNUCKLE *) NULL; k_ptr++)
    if((*k_ptr)->connected != (KNUCKLE *) NULL)
      {
	status = get_knuckle((*k_ptr)->cusername,(*k_ptr)->cinstance, &k,
			     /*???*/0);
	if(status == SUCCESS)
	  {
	    (*k_ptr)->connected = k;
	    if((*k_ptr)->question != (QUESTION *) NULL)
	      {
		(*k_ptr)->connected->question = (*k_ptr)->question;
		
		/* not good enough- should have some physical tag */
		if((*k_ptr)->status  & QUESTION_STATUS)
		  (*k_ptr)->question->owner = (*k_ptr);		    
	      }
	  }
	else
	  {
	    *((*k_ptr)->cusername) = '\0';
	    (*k_ptr)->connected = (KNUCKLE *) NULL;
	    if((*k_ptr)->question != (QUESTION *) NULL)
	      if((*k_ptr)->status  & QUESTION_STATUS)
		(*k_ptr)->question->owner = (*k_ptr);		    
	  }
      }
}

/*** I/O helpers ***/

#define LINE_CHUNK 1024

/* Read a line of arbitrary length from an I/O stream.
 * Arguments:   buf:  Address of a char* variable used as a buffer.  The
 *                    variable should be initialized to NULL or a malloc'd
 *                    buffer before the first call to this function, and
 *                    can simply be re-used later, but the buffer will be
 *                    overwritten and reallocated at will.  The value
 *                    should be free()'d after the last call to fget_line.
 *              size: Address of a size_t variable containing the size of
 *                    *buf.  The variable should be initialized to 0 if
 *                    *buf is NULL, or whatever the allocation size of *buf
 *                    is, otherwise.  It will be modified whenever buf is
 *                    grown.
 *              fp:   the stream to read.  (Must be open for reading!)
 * Returns: A pointer to the data read (same as *buf), or NULL if an error
 *          or EOF occurs and no data is available.
 * Note: trailing newlines are stripped from data.
 */
char *fget_line (char **buf, size_t *size, FILE *fp)
{
  size_t len;

  /* If we don't have a buffer yet, get one. */
  if (*size == 0)
    {
      *size = LINE_CHUNK;
      *buf = malloc(*size);
      if (*buf == NULL)
	{
	  log_error("fget_line: malloc failed: %m");
	  return NULL;
	}
    }

  /* Read some data, returning NULL if we fail. */
  if (fgets(*buf, *size, fp) == NULL)
    return NULL;

  while (1)
    {
      len = strlen(*buf);
      /* If the string is empty, return NULL.
       * (This shouldn't ever happen unless fp contains '\0' characters.)
       */
      if (len == 0)
	return NULL;
      /* If the string contains a full line, return that. */
      if ((*buf)[len-1] == '\n') {
	(*buf)[len-1] = '\0';
	return *buf;
      }
      /* If the line is shorter than *buf is, we ran into EOF; return data. */
      if (len+1 < *size)
	return *buf;

      /* OK, the only other option is that our line was too short... */
      *size += LINE_CHUNK;
      *buf = realloc(*buf, *size);
      if (*buf == NULL)
	{
	  log_error("fget_line: realloc failed: %m");
	  return NULL;
	}

      /* Try reading more data and see if we fare better. */
      if (fgets((*buf)+len, (*size)-len, fp) == NULL)
	return *buf;
    }
}

/* Read a line containing a specified fixed string followed by the
 * ASCII representation of a long value, and extract the value.  If
 * the line read does not start with the fixed string, report an error.
 *
 * Arguments:   val:  pointer to a long variable to store the result in.
 *              lead: String expected to appear at the start of the line.
 *              buf:  Address of a char* variable used as a buffer, as
 *                    for fget_line [ie, initialize to NULL, free afterwards].
 *              size: Address of a size_t variable containing the size of
 *                    *buf, as for fget_line [ie, initialize to 0].
 *              fp:   the stream to read.  (Must be open for reading!)
 * Returns: SUCCESS on success, ERROR on failure
 */
ERRCODE expect_long (long *val, char *lead, char **buf, size_t *size, FILE *fp)
{
  char *end;
  size_t llen = strlen(lead);

  if (! fget_line(buf, size, fp)) {
    log_error("can't read a line while expecting '%s'", lead);
    return ERROR;
  }
  if (strncmp(*buf, lead, llen)) {
    log_error("oops, I was expecting '%s' but I got '%s'", lead, *buf);
    return ERROR;
  }

  *val = strtol((*buf)+llen, &end, 10);
  if ((end == (*buf)+llen) || (*end != '\0')) {
    log_error("line doesn't end after expected long at '%s'", lead);
    return ERROR;
  }
  return SUCCESS;
}

/* Read a line containing a specified fixed string followed by a
 * string value with fixed manimum width.  If the line read does not
 * start with the fixed string, report an error.
 *
 * Arguments:   dst:    pointer to a char[] variable to store the result in.
 *		dstlen: size of the dst array.
 *              lead:   String expected to appear at the start of the line.
 *              buf:    Address of a char* variable used as a buffer, as
 *                      for fget_line [ie, initialize to NULL, free afterwards]
 *              size:   Address of a size_t variable containing the size of
 *                      *buf, as for fget_line [ie, initialize to 0].
 *              fp:     the stream to read.  (Must be open for reading!)
 * Returns: SUCCESS on success, ERROR on failure
 */
ERRCODE expect_str_fixwid (char *dst, size_t dstlen, char *lead,
			   char **buf, size_t *size, FILE *fp)
{
  size_t llen = strlen(lead);

  if (! fget_line(buf, size, fp)) {
    log_error("can't read a line while expecting '%s'", lead);
    return ERROR;
  }
  if (strncmp(*buf, lead, llen)) {
    log_error("oops, I was expecting '%s' but I got '%s'", lead, *buf);
    return ERROR;
  }

  strncpy(dst, (*buf)+llen, dstlen);
  dst[dstlen-1] = '\0';
  return SUCCESS;
}
