/****************************************************************************

		THIS SOFTWARE IS NOT COPYRIGHTED  
   
   HP offers the following for use in the public domain.  HP makes no
   warranty with regard to the software or it's performance and the 
   user accepts the software "AS IS" with all faults.

   HP DISCLAIMS ANY WARRANTIES, EXPRESS OR IMPLIED, WITH REGARD
   TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO THE WARRANTIES
   OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.

****************************************************************************/

/* Ported to PalmOS/m68k by Kresten Krab Thorup */

#define REMOTE_BAUD 38400

/****************************************************************************
 *  To enable debugger support, two things need to happen.  One, a
 *  call to set_debug_traps() is necessary in order to allow any breakpoints
 *  or error conditions to be properly intercepted and reported to gdb.
 *  Two, a breakpoint needs to be generated to begin communication.  This
 *  is most easily accomplished by a call to breakpoint().  Breakpoint()
 *  simulates a breakpoint by executing a trap #1.  The breakpoint instruction
 *  is hardwired to trap #1 because not to do so is a compatibility problem--
 *  there either should be a standard breakpoint instruction, or the protocol
 *  should be extended to provide some means to communicate which breakpoint
 *  instruction is in use (or have the stub insert the breakpoint).
 *  
 *  Some explanation is probably necessary to explain how exceptions are
 *  handled.  When an exception is encountered the 68000 pushes the current
 *  program counter and status register onto the supervisor stack and then
 *  transfers execution to a location specified in it's vector table.
 *  The handlers for the exception vectors are hardwired to jmp to an address
 *  given by the relation:  exceptionTable + 6 * exception.  By
 *  allowing 6 bytes for 
 *  each entry, a jsr, jmp, bsr, ... can be used to enter the exception 
 *  handler.  Using a jsr to handle an exception has an added benefit of
 *  allowing a single handler to service several exceptions and use the
 *  return address as the key differentiation.  The vector number can be
 *  computed from the return address by [ exception = (addr + 1530) / 6 ].
 *  The sole purpose of the routine _catchException is to compute the
 *  exception number and push it on the stack in place of the return address.
 *  The external function exceptionHandler() is
 *  used to attach a specific handler to a specific m68k exception.
 *  For 68020 machines, the ability to have a return address around just
 *  so the vector can be determined is not necessary because the '020 pushes an
 *  extra word onto the stack containing the vector offset
 * 
 *  Because gdb will sometimes write to the stack area to execute function
 *  calls, this program cannot rely on using the supervisor stack so it
 *  uses it's own stack area reserved in the int array remcomStack.  
 * 
 *************
 *
 *    The following gdb commands are supported:
 * 
 * command          function                               Return value
 * 
 *    g             return the value of the CPU registers  hex data or ENN
 *    G             set the value of the CPU registers     OK or ENN
 * 
 *    mAA..AA,LLLL  Read LLLL bytes at address AA..AA      hex data or ENN
 *    MAA..AA,LLLL: Write LLLL bytes at address AA.AA      OK or ENN
 * 
 *    c             Resume at current address              SNN   ( signal NN)
 *    cAA..AA       Continue at address AA..AA             SNN
 * 
 *    s             Step one instruction                   SNN
 *    sAA..AA       Step one instruction from AA..AA       SNN
 * 
 *    k             kill
 *
 *    ?             What was the last sigval ?             SNN   (signal NN)
 * 
 * All commands and responses are sent with a packet which includes a 
 * checksum.  A packet consists of 
 * 
 * $<packet info>#<checksum>.
 * 
 * where
 * <packet info> :: <characters representing the command or response>
 * <checksum>    :: < two hex digits computed as modulo 256 sum of <packetinfo>>
 * 
 * When a packet is received, it is first acknowledged with either '+' or '-'.
 * '+' indicates a successful transfer.  '-' indicates a failed transfer.
 * 
 * Example:
 * 
 * Host:                  Reply:
 * $m0,10#2a               +$00010203040506070809101112131415#42
 * 
 ****************************************************************************/

#include <stdio.h>
#include <string.h>
#include <setjmp.h>

#include <SerialMgr.h>
#include <ErrorMgr.h>
#include <SysEvtMgr.h>

#include <System/Preferences.h>
#include <UI/UIAll.h>
#define ALERT(M) FrmCustomAlert(4999, M, 0, 0)

/************************************************************************
 *
 * external low-level support routines 
 */
typedef void (*ExceptionHook)(long);   /* pointer to function with int parm */
typedef void (*Function)();           /* pointer to a function */

static int putDebugChar(int);   /* write a single character      */
static int putDebugString(char*,int);   /* write a single character      */
static int getDebugChar();   /* read and return a single char */

static void exceptionHandler(int,Function);  /* assign an exception handler */
static ExceptionHook exceptionHook;  /* hook variable for errors/exceptions */

/************************/
/* FORWARD DECLARATIONS */
/************************/
static void
initializeRemcomErrorFrame ();

static void
initPilotSerial();

static void debug_error (char *, char*);

/************************************************************************/
/* BUFMAX defines the maximum number of characters in inbound/outbound buffers*/
/* at least NUMREGBYTES*2 are needed for register packets */
#define BUFMAX (NUMREGBYTES*2+8)

static char initialized;  /* boolean flag. != 0 means we've been initialized */

long     remote_debug = 0;
/*  debug >  0 prints ill-formed commands in valid packets & checksum errors */ 

static const char hexchars[]="0123456789abcdef";

#define NUMREGBYTES (MAXREGS * 4)
enum regnames {D0,D1,D2,D3,D4,D5,D6,D7, 
               A0,A1,A2,A3,A4,A5,A6,A7, 
               PS,PC,
               MAXREGS };


/* We keep a whole frame cache here.  "Why?", I hear you cry, "doesn't
   GDB handle that sort of thing?"  Well, yes, I believe the only
   reason for this cache is to save and restore floating point state
   (fsave/frestore).  A cleaner way to do this would be to make the
 fsave data part of the registers which GDB deals with like any
   other registers.  This should not be a performance problem if the
   ability to read individual registers is added to the protocol.  */

typedef struct FrameStruct
{
    struct FrameStruct  *previous;
    long       exceptionPC;      /* pc value when this frame created */
    long       exceptionVector;  /* cpu vector causing exception     */
    short     frameSize;        /* size of cpu frame in words       */
    short     sr;               /* for 68000, this not always sr    */
    long       pc;
    short     format;
    long       fsaveHeader;
    long       morejunk[0];        /* exception frame, fp save... */
} Frame;

#define FRAMESIZE 100
long   gdbFrameStack[FRAMESIZE];
static Frame *lastFrame;

/*
 * these should not be static cuz they can be used outside this module
 */
long registers[NUMREGBYTES/4];
long superStack;


#define STACKSIZE 2000		/* 2k */
static long* remcomStack;
static long* stackPtr /* = &remcomStack[STACKSIZE/sizeof(long) - 1] */;

/*
 * In many cases, the system will want to continue exception processing
 * when a continue command is given.  
 * oldExceptionHook is a function to invoke in this case.
 */

static ExceptionHook oldExceptionHook;

/************* jump buffer used for setjmp/longjmp **************************/
jmpbuf remcomEnv[1];

/***************************  ASSEMBLY CODE MACROS *************************/
/* 									   */

#define SAVE_FP_REGS()
#define RESTORE_FP_REGS()

void return_to_super();
void return_to_user();

asm("
.text
.even
.globl return_to_super
return_to_super:
        movel   registers+60(%a4),%sp /* get new stack pointer */        
        movel   lastFrame(%a4),%a0   /* get last frame info  */              
        bra     return_to_any

.even
.globl return_to_user
return_to_user:
        movel   registers+60(%a4),%a0 /* get usp */                          
        movel   %a0,%usp           /* set usp */				
        movel   superStack(%a4),%sp  /* get original stack pointer */        

return_to_any:
        movel   lastFrame(%a4),%a0   /* get last frame info  */              
        movel   (%a0)+,lastFrame(%a4) /* link in previous frame     */        
        addql   #8,%a0           /* skip over pc, vector#*/              
        movew   (%a0)+,%d0         /* get # of words in cpu frame */       
        addw    %d0,%a0           /* point to end of data        */       
        addw    %d0,%a0           /* point to end of data        */       
        movel   %a0,%a1                                                   
#                                                                       
# copy the stack frame                                                  
        subql   #1,%d0                                                   
copyUserLoop:                                                               
        movew   -(%a1),-(%sp)                                               
        dbf     %d0,copyUserLoop
");                                                                     
        RESTORE_FP_REGS()                                              
   asm("   moveml  registers(%a4),%d0-%d7/%a0-%a6");			        
   asm("   rte");  /* pop and go! */                                    

#define DISABLE_INTERRUPTS()   asm("         oriw   #0x0700,%sr");
#define BREAKPOINT() asm("   trap #1");

extern void _debug_level7 ();

/* this function is called immediately when a level 7 interrupt occurs */
/* if the previous interrupt level was 7 then we're already servicing  */
/* this interrupt and an rte is in order to return to the debugger.    */
/* For the 68000, the offset for sr is 6 due to the jsr return address */
asm("
.text
.even
.globl _debug_level7
_debug_level7:
	movew   %d0,-(%sp)");
asm("	movew   6(%sp),%d0");
asm("	andiw   #0x700,%d0
	cmpiw   #0x700,%d0
	beq     _already7
        movew   (%sp)+,%d0	
        bra     _catchException
_already7:
	movew   (%sp)+,%d0");
asm("	lea     4(%sp),%sp");     /* pull off 68000 return address */
asm("	rte");

extern void _catchException ();

/* This function is called when an exception occurs.  It translates the
 * return address found on the stack into an exception vector # which
 * is then handled by either handle_exception or a system handler.
 * _catchException provides a front end for both.  
 *
 * stack on entry:                       stack on exit:
 *   Program counter MSWord              exception # MSWord 
 *   Program counter LSWord              exception # MSWord
 *   Status Register                     
 *   Return Address  MSWord              
 *   Return Address  LSWord             
 */
asm("
.text
.even
	/* stub for exception handler.  Since the PalmOS otherwise
           requires PIC code, we put this in the data 
           segment and patch it to the exact values.  */

.globl _catchException
_catchException:
        moveml  %d0-%d7/%a0-%a6,0xecececec  /* save registers */
	movel   #0xa5a5a5a5,%a5             /* restore %a5 */
	movel   %a5,%a4
	subl    #edata,%a4
	movel	lastFrame(%a4),%a0	/* last frame pointer */
");
SAVE_FP_REGS();        
asm("
        lea     registers(%a4),%a3   /* get address of registers     */
        movel   (%sp)+,%d2         /* pop return address           */
	movel   _exceptionTable(%a4),%d3
        subl    #6,%d2
	subl    %d3, %d2
	divs 	#6,%d2   	/*  exception number		*/
	extl    %d2   

        moveql  #3,%d3           /* assume a three word frame     */

        cmpiw   #3,%d2           /* bus error or address error ? */
        bgt     normal          /* if >3 then normal error      */
        movel   (%sp)+,-(%a0)       /* copy error info to frame buff*/
        movel   (%sp)+,-(%a0)       /* these are never used         */
        moveql  #7,%d3           /* this is a 7 word frame       */
     
normal:   
	movew   (%sp)+,%d1         /* pop status register          */
        movel   (%sp)+,%a2         /* pop program counter          */
        movew   %d1,66(%a3)      /* save sr		 	*/	
        movel   %a2,68(%a3)      /* save pc in _regisers[]      	*/
        movel   %a2,-(%a0)         /* copy pc to frame buffer      */
	movew   %d1,-(%a0)         /* copy sr to frame buffer      */

        movel   %sp,superStack(%a4)  /* save supervisor %sp          */

        andiw   #0x2000,%d1      /* were we in supervisor mode ? */
        beq     userMode       
        movel   %a7,60(%a3)      /* save %a7                  */
        bra     saveDone             
userMode:
        movel   %usp,%a1    	/* save user stack pointer 	*/
        movel   %a1,60(%a3)      /* save user stack pointer	*/
saveDone:

        movew   %d3,-(%a0)         /* push frame size in words     */
        movel   %d2,-(%a0)         /* push vector number           */
        movel   %a2,-(%a0)         /* push exception pc            */

#
# save old frame link and set the new value
	movel	lastFrame(%a4),%a1	/* last frame pointer */
	movel   %a1,-(%a0)		/* save pointer to prev frame	*/
        movel   %a0,lastFrame(%a4)

        movel   %d2,-(%sp)		/* push exception num           */
	movel   exceptionHook(%a4),%a0  /* get address of handler */
        jbsr     %a0@             /* and call it */
        clrl    %sp@             /* replace exception num parm with frame ptr */
        bsr    _returnFromException   /* jbsr, but never returns */
");


/*
 * remcomHandler is a front end for handle_exception.  It moves the
 * stack pointer into an area reserved for debugger use in case the
 * breakpoint happened in supervisor mode.
 */

/* a4 is already set up because this is called via exceptionHook */
asm("remcomHandler:");
asm("           addl    #4,%sp");        /* pop off return address     */
asm("           movel   (%sp)+,%d0");      /* get the exception number   */
asm("		movel   stackPtr(%a4),%sp"); /* move to remcom stack area  */
asm("		movel   %d0,-(%sp)");	/* push exception onto stack  */
asm("		bsr     handle_exception");    /* this never returns */
asm("           rts");                  /* return */

void _returnFromException( Frame *frame )
{
    /* if no passed in frame, use the last one */
    if (! frame)
    {
        frame = lastFrame;
	frame->frameSize = 4;
        frame->format = 0;
        frame->fsaveHeader = -1; /* restore regs, but we dont have fsave info*/
    }

    /* a 68000 cannot use the internal info pushed onto a bus error
     * or address error frame when doing an RTE so don't put this info
     * onto the stack or the stack will creep every time this happens.
     */
    frame->frameSize=3;

    /* throw away any frames in the list after this frame */
    lastFrame = frame;

    frame->sr = registers[(int) PS];
    frame->pc = registers[(int) PC];

    if (registers[(long) PS] & 0x2000)
    { 
        /* return to supervisor mode... */
        return_to_super();
    }
    else
    { /* return to user mode */
        return_to_user();
    }
}


static void
patch_handler_stubs ()
{
  extern void _catchException ();
  UInt *exc = (UInt*) &_catchException;
  register unsigned long reg_a5 asm ("%a5");

  MemSemaphoreReserve (1);

  exc[2] = (((ULong)&registers) >> 16) & 0xffff;
  exc[3] = ((ULong)&registers) & 0xffff;

  exc[5] = (reg_a5 >> 16) & 0xffff;
  exc[6] = reg_a5 & 0xffff;

  MemSemaphoreRelease (1);
}

long hex(ch)
char ch;
{
  if ((ch >= 'a') && (ch <= 'f')) return (ch-'a'+10);
  if ((ch >= '0') && (ch <= '9')) return (ch-'0');
  if ((ch >= 'A') && (ch <= 'F')) return (ch-'A'+10);
  return (-1);
}


/* scan for the sequence $<data>#<checksum>     */
void getpacket(buffer)
char * buffer;
{
  unsigned char checksum;
  unsigned char xmitcsum;
  long  i;
  long  count;
  char ch;
  
  do {
    /* wait around for the start character, ignore all other characters */
    while ((ch = (getDebugChar() & 0x7f)) != '$'); 
    checksum = 0;
    xmitcsum = -1;
    
    count = 0;
    
    /* now, read until a # or end of buffer is found */
    while (count < BUFMAX) {
      ch = getDebugChar() & 0x7f;
      if (ch == '#') break;
      checksum = checksum + ch;
      buffer[count] = ch;
      count = count + 1;
      }
    buffer[count] = 0;

    if (ch == '#') {
      xmitcsum = hex(getDebugChar() & 0x7f) << 4;
      xmitcsum += hex(getDebugChar() & 0x7f);
      if ((remote_debug ) && (checksum != xmitcsum)) {
	{
#ifdef DEBUG
	  debug_error ("gdb: bad checksum in serial package", "");
#endif
	}
      }
      
      if (checksum != xmitcsum) putDebugChar('-');  /* failed checksum */ 
      else {
	 putDebugChar('+');  /* successful transfer */
	 /* if a sequence char is present, reply the sequence ID */
	 if (buffer[2] == ':') {
	    putDebugChar( buffer[0] );
	    putDebugChar( buffer[1] );
	    /* remove sequence chars from buffer */
	    count = strlen(buffer);
	    for (i=3; i <= count; i++) buffer[i-3] = buffer[i];
	 } 
      } 
    } 
  } while (checksum != xmitcsum);
  
}

/* send the packet in buffer.  The host get's one chance to read it.  
   This routine does not wait for a positive acknowledge.  */


void putpacket(char *buffer)
{
  unsigned char checksum;
  long  count;
  char ch;
  
  /*  $<packet info>#<checksum>. */
  do {
  putDebugChar('$');
  checksum = 0;
  count    = 0;
  
  while (ch=buffer[count]) {
    checksum += ch;
    count += 1;
  }

  putDebugString (&buffer[0], count);
  
  putDebugChar('#');
  putDebugChar(hexchars[checksum >> 4]);
  putDebugChar(hexchars[checksum % 16]);

  } while (1 == 0);  /* (getDebugChar() != '+'); */
  
}

char  remcomInBuffer[BUFMAX];
char  remcomOutBuffer[BUFMAX];
static short error;

char buffer[512];

static void
debug_error(char *format, char *parm)
{
#ifdef DEBUG
  sprintf (buffer, format, parm);
  ALERT (buffer);
  if (remote_debug) printf (format,parm);
#endif
}

/* convert the memory pointed to by mem into hex, placing result in buf */
/* return a pointer to the last char put in buf (null) */
char* mem2hex(char* mem, char *buf, long count)
{
      long i;
      unsigned char ch;
      for (i=0;i<count;i++) {
          ch = *mem++;
          *buf++ = hexchars[ch >> 4];
          *buf++ = hexchars[ch % 16];
      }
      *buf = 0; 
      return(buf);
}

/* convert the hex array pointed to by buf into binary to be placed in mem */
/* return a pointer to the character AFTER the last byte written */
char* hex2mem(char *buf, char *mem, long count)
{
      long i;
      unsigned char ch;
      for (i=0;i<count;i++) {
          ch = hex(*buf++) << 4;
          ch = ch + hex(*buf++);
          *mem++ = ch;
      }
      return(mem);
}

/* a bus error has occurred, perform a longjmp
   to return execution and allow handling of the error */

static int memerr;

void handle_buserror()
{
  memerr = 2;
  longjmp(remcomEnv,1);
}

void handle_addresserror()
{
  memerr = 3;
  longjmp(remcomEnv,1);
}

/* this function takes the 68000 exception number and attempts to 
   translate this number into a unix compatible signal value */
long computeSignal( long exceptionVector )
{
  long sigval;
  switch (exceptionVector) {
    case 2 : sigval = 10; break; /* bus error           */
    case 3 : sigval = 10; break; /* address error       */
    case 4 : sigval = 4;  break; /* illegal instruction */
    case 5 : sigval = 8;  break; /* zero divide         */
    case 6 : sigval = 8; break; /* chk instruction     */
    case 7 : sigval = 8; break; /* trapv instruction   */
    case 8 : sigval = 11; break; /* privilege violation */
    case 9 : sigval = 5;  break; /* trace trap          */
    case 10: sigval = 4;  break; /* line 1010 emulator  */
    case 11: sigval = 4;  break; /* line 1111 emulator  */

      /* Coprocessor protocol violation.  Using a standard MMU or FPU
	 this cannot be triggered by software.  Call it a SIGBUS.  */
    case 13: sigval = 10;  break;

    case 31: sigval = 2;  break; /* interrupt           */
    case 33: sigval = 5;  break; /* breakpoint          */

      /* This is a trap #8 instruction.  Apparently it is someone's software
	 convention for some sort of SIGFPE condition.  Whose?  How many
	 people are being screwed by having this code the way it is?
	 Is there a clean solution?  */
    case 40: sigval = 8;  break; /* floating point err  */

    case 48: sigval = 8;  break; /* floating point err  */
    case 49: sigval = 8;  break; /* floating point err  */
    case 50: sigval = 8;  break; /* zero divide         */
    case 51: sigval = 8;  break; /* underflow           */
    case 52: sigval = 8;  break; /* operand error       */
    case 53: sigval = 8;  break; /* overflow            */
    case 54: sigval = 8;  break; /* NAN                 */
    default: 
      sigval = 7;         /* "software generated"*/
  }
  return (sigval);
}

/**********************************************/
/* WHILE WE FIND NICE HEX CHARS, BUILD AN INT */
/* RETURN NUMBER OF CHARS PROCESSED           */
/**********************************************/
long hexToInt(char **ptr, long *intValue)
{
    long numChars = 0;
    long hexValue;
    
    *intValue = 0;

    while (**ptr)
    {
        hexValue = hex(**ptr);
        if (hexValue >=0)
        {
            *intValue = (*intValue <<4) | hexValue;
            numChars ++;
        }
        else
            break;
        
        (*ptr)++;
    }

    return (numChars);
}

/*
 * This function does all command procesing for interfacing to gdb.
 */

static int continue_mode = 0;

int first_trap_1 = 0;
void handle_exception(long exceptionVector)
{
  unsigned long    sigval;
  unsigned long    addr, length;
  char * ptr;
  long    newPC;
  Frame  *frame;

  /* if we receive a trace interrupt 
     while we're in Continue mode, we'll silently ignore it.   */
  if ((registers[PS] & 0x8000) == 0x8000
      && continue_mode
      && exceptionVector == 9)
    {
      registers[PS] &= ~0x8000;
      _returnFromException (0);
    }

#ifdef DEBUG
  if (remote_debug) printf("vector=%ld, sr=0x%lx, pc=0x%lx\n", 
			    exceptionVector,
			    registers[ PS ], 
			    registers[ PC ]);
#endif

  /* reply to host that an exception has occurred */
  sigval = computeSignal( exceptionVector );
  remcomOutBuffer[0] = 'S';
  remcomOutBuffer[1] =  hexchars[sigval >> 4];
  remcomOutBuffer[2] =  hexchars[sigval % 16];
  remcomOutBuffer[3] = 0;

  putpacket(remcomOutBuffer); 

  while (1==1) { 
    error = 0;
    remcomOutBuffer[0] = 0;

    EvtResetAutoOffTimer ();

    getpacket(remcomInBuffer);

    switch (remcomInBuffer[0]) {
      case '?' :   remcomOutBuffer[0] = 'S';
                   remcomOutBuffer[1] =  hexchars[sigval >> 4];
                   remcomOutBuffer[2] =  hexchars[sigval % 16];
                   remcomOutBuffer[3] = 0;
                 break; 
     case 'P':
	{
	  ULong reg = 0;
	  ULong value = 0;
	  char *ptr = &remcomInBuffer[1];

	  while (*ptr != '=')
	    {
	      reg *= 16;
	      reg += hex (*ptr++);
	    } 

	  ptr += 1;

	  while (*ptr != 0) {
	    value *= 16;
	    value += hex (*ptr++);
	  } 

	  registers[reg] = value;
	  
	  strcpy (remcomOutBuffer, "OK");
	  break;
	}

      case 'q' : 
	{
	  extern long data_start, bss_start;
	  extern void start ();
	  unsigned long data = (unsigned long)&data_start;
	  unsigned long bss  = (unsigned long)&bss_start;
	  unsigned long text = (unsigned long)&start;

	  asm ("sub.l #start, %0" : "=g" (text) : "0" (text));

	  bcopy ((const char*)"Text=00000000;Data=00000000;Bss=00000000", 
		 (char*) &remcomOutBuffer[0], 
		 (UInt) 40);
	  mem2hex ((char*)&text, &remcomOutBuffer[5], 4);
	  remcomOutBuffer[13] = ';';
	  mem2hex ((char*)&data, &remcomOutBuffer[19], 4);
	  remcomOutBuffer[27] = ';';
	  mem2hex ((char*)&bss,  &remcomOutBuffer[32], 4);

	  break;
	}
	
      case 'd' : remote_debug = !(remote_debug);  /* toggle debug flag */
                 break; 
      case 'g' : /* return the value of the CPU registers */
                mem2hex((char*)&registers[0],
			(char*)&remcomOutBuffer[0],
			(long) NUMREGBYTES);
                break;
      case 'G' : /* set the value of the CPU registers - return OK */
                hex2mem(&remcomInBuffer[1],
			(char*) &registers[0],
			(long) NUMREGBYTES);
                strcpy(remcomOutBuffer,"OK");
                break;
      
      /* mAA..AA,LLLL  Read LLLL bytes at address AA..AA */
      case 'm' : 
	        if (setjmp(remcomEnv) == 0)
                {
                    exceptionHandler(2,handle_buserror); 
                    exceptionHandler(3,handle_addresserror); 

		    /* TRY TO READ %x,%x.  IF SUCCEED, SET PTR = 0 */
                    ptr = &remcomInBuffer[1];
                    if (hexToInt(&ptr,&addr))
                        if (*(ptr++) == ',')
                            if (hexToInt(&ptr,&length)) 
                            {
			      ptr = 0;
			      if (addr > 0x10c80000
				  ||   (addr >= 0x100000 
				      && addr < 0x10000000))
				{
				  strcpy (remcomInBuffer, "E00");
				}
			      else
				{
				  mem2hex((char*) addr,
					  (char*) remcomOutBuffer,
					  (long) length);
				}
                            }

                    if (ptr)
                    {
		      strcpy(remcomOutBuffer,"E01");
#ifdef DEBUG
		      debug_error("malformed read memory command: %s",remcomInBuffer);
#endif
                  }     
                } 
		else
		  {
		    exceptionHandler(2,_catchException);   
		    exceptionHandler(3,_catchException);   
		    strcpy(remcomOutBuffer,"E03");
#ifdef DEBUG
		    debug_error("bus error on read 0x%x", (char*) addr);
#endif
		  }     
                
		/* restore handler for bus error */
		exceptionHandler(2,_catchException);   
		break;
      
      /* MAA..AA,LLLL: Write LLLL bytes at address AA.AA return OK */
      case 'M' : 
	        if (setjmp(remcomEnv) == 0) {
		    exceptionHandler(2,handle_buserror); 
		    exceptionHandler(3,handle_addresserror); 
                    
		    /* TRY TO READ '%x,%x:'.  IF SUCCEED, SET PTR = 0 */
                    ptr = &remcomInBuffer[1];
                    if (hexToInt(&ptr,&addr))
                        if (*(ptr++) == ',')
                            if (hexToInt(&ptr,&length))
                                if (*(ptr++) == ':')
                                {
				  if (addr > 0x10c00000)
				    {
				      strcpy (remcomInBuffer, "E01");
#ifdef DEBUG
				      debug_error ("gdb: trying to "
						   "write in ROM area", 0);
#endif
				      ptr = 0;
				    }
				  else
				    {
				      MemSemaphoreReserve (1);
				      hex2mem(ptr, (char*) addr, length);
				      MemSemaphoreRelease (1);
				      ptr = 0;
				      strcpy(remcomOutBuffer,"OK");
				    }
                                }
                    if (ptr)
                    {
		      strcpy(remcomOutBuffer,"E02");
#ifdef DEBUG
		      debug_error("malformed write memory command: %s",remcomInBuffer);
#endif
		      }     
                } 
		else {
		  exceptionHandler(2,_catchException);   
		  exceptionHandler(3,_catchException);   
		  MemSemaphoreRelease (1);
		  strcpy(remcomOutBuffer,"E03");
#ifdef DEBUG
		  debug_error("bus error on write 0x%x", (char*) addr);
#endif
		}     

                /* restore handler for bus error */
                exceptionHandler(2,_catchException);   
                break;
     
     /* cAA..AA    Continue at address AA..AA(optional) */
     /* sAA..AA   Step one instruction from AA..AA(optional) */
     case 'c' : 
     case 's' : 
          /* try to read optional parameter, pc unchanged if no parm */
         ptr = &remcomInBuffer[1];
         if (hexToInt(&ptr,&addr))
             registers[ PC ] = addr;
             
          newPC = registers[ PC];
          
          /* clear the trace bit */
          registers[ PS ] &= 0x7fff;
	  continue_mode = 1;
          
          /* set the trace bit if we're stepping */
          if (remcomInBuffer[0] == 's') 
	    {
	      registers[ PS ] |= 0x8000;
	      continue_mode = 0;
	    }
          
          /*
           * look for newPC in the linked list of exception frames.
           * if it is found, use the old frame it.  otherwise,
           * fake up a dummy frame in returnFromException().
           */
#ifdef DEBUG
          if (remote_debug) printf("new pc = 0x%lx\n",newPC);
#endif
          frame = lastFrame;
          while (frame)
          {
#ifdef DEBUG
              if (remote_debug)
                  printf("frame at 0x%lx has pc=0x%lx, except#=%ld\n",
                         (ULong)frame, frame->exceptionPC,
                         frame->exceptionVector);
#endif
              if (frame->exceptionPC == newPC) break;  /* bingo! a match */
              /*
               * for a breakpoint instruction, the saved pc may
               * be off by two due to re-executing the instruction
               * replaced by the trap instruction.  Check for this.
               */
              if ((frame->exceptionVector == 33) &&
                  (frame->exceptionPC == (newPC+2))) break;
              if (frame == frame->previous)
	      {
	          frame = 0; /* no match found */ 
	          break; 
	      }
	      frame = frame->previous;
          }
          
          /*
           * If we found a match for the PC AND we are not returning
           * as a result of a breakpoint (33),
           * trace exception (9), nmi (31), jmp to
           * the old exception handler as if this code never ran.
           */
          if (frame) 
          {
              if ((frame->exceptionVector != 9)  && 
                  (frame->exceptionVector != 31) && 
                  (frame->exceptionVector != 33))
              { 
                  /*
                   * invoke the previous handler.
                   */
                  if (oldExceptionHook)
                      (*oldExceptionHook) (frame->exceptionVector);
                  newPC = registers[ PC ];    /* pc may have changed  */
                  if (newPC != frame->exceptionPC)
                  {
#ifdef DEBUG
                      if (remote_debug)
                          printf("frame at 0x%lx has pc=0x%lx, except#=%ld\n",
                                 (ULong)frame, frame->exceptionPC,
                                 frame->exceptionVector);
#endif
                      /* re-use the last frame, we're skipping it (longjump?)*/
		      frame = (Frame *) 0;
	              _returnFromException( frame );  /* this is a jump */
                  }
              }
          }         

    	  /* if we couldn't find a frame, create one */
          if (frame == 0)
	  {
    	      frame = lastFrame -1 ;
	      
	      /* by using a bunch of print commands with breakpoints,
    	         it's possible for the frame stack to creep down.  If it creeps
		 too far, give up and reset it to the top.  Normal use should
    	         not see this happen.
    	      */
	      if ((unsigned long) (frame-2) < (unsigned long) &gdbFrameStack[0])
    	      {
    	         initializeRemcomErrorFrame();
    	         frame = lastFrame; 
	      }
    	      frame->previous = lastFrame;
              lastFrame = frame;
              frame = 0;  /* null so _return... will properly initialize it */ 
	  }    
	  
	  _returnFromException( frame ); /* this is a jump */

          break;
          
      /* kill the program */
      case 'k' :  /* do nothing */
	
	putpacket ("OK");
	SysReset ();
	break;
      } /* switch */ 
    
    /* reply to the request */
    putpacket(remcomOutBuffer); 
    }
}


static void
initializeRemcomErrorFrame()
{
    lastFrame = ((Frame *) &gdbFrameStack[FRAMESIZE-1]) - 1;
    lastFrame->previous = lastFrame;
}

/* this function is used to set up exception handlers for tracing and 
   breakpoints */
void set_debug_traps()
{
  extern void remcomHandler();
  long exception;

  patch_handler_stubs ();

  initializeRemcomErrorFrame();
  remcomStack  = MemPtrNew (STACKSIZE);
  stackPtr = &remcomStack[STACKSIZE/sizeof(long) - 1];
  ErrFatalDisplayIf (remcomStack == 0,
		     "gdb: Can't allocate stack");

  for (exception = 2; exception <= 23; exception++)
      exceptionHandler(exception,_catchException);   

  /* level 7 interrupt              */
  exceptionHandler(31,_debug_level7);    
  
  /* breakpoint exception (trap #1) */
  exceptionHandler(33,_catchException);
  
  /* This is a trap #8 instruction.  Apparently it is someone's software
     convention for some sort of SIGFPE condition.  Whose?  How many
     people are being screwed by having this code the way it is?
     Is there a clean solution?  */
  exceptionHandler(40,_catchException);
  
  /* 48 to 54 are floating point coprocessor errors */
  for (exception = 48; exception <= 54; exception++)
      exceptionHandler(exception,_catchException);   

  if (oldExceptionHook != remcomHandler)
  {
      oldExceptionHook = exceptionHook;
      exceptionHook    = remcomHandler;
  }

  initPilotSerial ();
  
  initialized = 1;

}

static UInt serLib = 0;

static void initPilotSerial()
{
  Err status;

  status = SysLibFind ("Serial Library", &serLib);
  if (status != 0)
    ErrDisplay ("gdb: SysLibFind reports error");

  status = SerOpen (serLib, 0, REMOTE_BAUD);
  if (status != 0)
    {
      if (status != serErrAlreadyOpen)
	ErrDisplay ("gdb: SerOpen unexpected error");

      SerClearErr (serLib);
    }
}

static int
putDebugString (char *data, int length)
{
  Err status;
  status = SerSend (serLib, data, length);
  
  if (status != 0)
    {
      if (status != serErrTimeOut)
	ErrDisplay ("gdb: SerOpen unexpected error");

      SerClearErr (serLib);
    }

  return 1;
}

static int
putDebugChar (int byte)
{
  UChar value = (UChar) byte;
  return putDebugString (&value, 1);
}


int getDebugChar ()
{
  Err status;
  char value;

  status = SerReceive (serLib, &value, 1, -1);
  
  if (status != 0)
    {
      if (status != serErrTimeOut && status != serErrLineErr)
	ErrDisplay ("gdb: SerOpen unexpected error");

      SerClearErr (serLib);
    }

  return value;
}

/* This function will generate a breakpoint exception.  It is used at the
   beginning of a program to sync up with a debugger and can be used
   otherwise as a quick means to stop program execution and "break" into
   the debugger. */
   
void breakpoint()
{
  if (initialized) BREAKPOINT();
}

static UInt *_exceptionTable = 0;

static void
exceptionHandler(int n, Function a)
{
  if (_exceptionTable == 0)
    {
      _exceptionTable = (UInt*) MemPtrNew (3*58*sizeof(UInt));
      
      ErrFatalDisplayIf (_exceptionTable == 0, \
			 "gdb: Can't allocate exception table");
    }

  {
    UInt *entry = &_exceptionTable[3*n];
    ULong target = (ULong)a;

    ((volatile unsigned long *)0x0)[n] = (ULong) entry;
    entry[0] = 0x4eb9;
    entry[1] = (target >> 16) & 0xffff;
    entry[2] = target & 0xffff;
  }
}

