/* Copyright (C) 2003 Peter J. Verveer
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met: 
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the following
 *    disclaimer in the documentation and/or other materials provided
 *    with the distribution.
 *
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
 * GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.      
 */

#include "ni_support.h"
#include "ni_morphology.h"
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <limits.h>

#define NI_ERODE_POINT(_pi, _out, _offsets, _filter_size, _type, _mv,	  \
		       _border_value, _bv, _true, _false, _ivo, _changed) \
{									  \
  int _ii, _oo = *_offsets;						  \
  int _in = *(_type*)_pi ? 1 : 0;					  \
  if (_mv) {								  \
    _out = 1;								  \
    for(_ii = 0; _ii < _filter_size; _ii++) {				  \
      _oo = _offsets[_ii];						  \
      if (_oo == _bv) {							  \
	if (!_border_value) {						  \
	  _out = 0;							  \
	  break;							  \
	}								  \
      } else {								  \
	int _nn = *(_type*)(_pi + _oo) ? _true : _false;		  \
	if (!_nn) {							  \
	  _out = 0;							  \
	  break;							  \
	}								  \
      }									  \
    }									  \
    _out = _ivo ? 1 - _out : _out;					  \
    if (_in != _out)							  \
      _changed = 1;							  \
  } else {								  \
    _out = _in;								  \
  }									  \
}

int NI_BinaryErosion(PyArrayObject* input, PyArrayObject* strct, 
		     PyArrayObject* mask, PyArrayObject** output,  
		     PyObject* output_in, int bdr_value, int *shifts, 
		     int invert_input, int invert_output,  int* changed)
{
  int struct_size = 0, *offsets = NULL, size, *oo, jj, border_flag_value;
  int ssize, mtype = tAny, true, false, msk_value, irank, itype;
  int idims[NI_MAXDIM], sdims[NI_MAXDIM];
  NI_Iterator ii, io, mi;
  NI_FilterIterator fi;
  Bool *ps, out = 0;
  char *pi, *po, *pm = NULL;

  /* complex type not supported: */
  itype = NI_GetArrayType(input);
  if (itype == tComplex32 || itype == tComplex64) {
    PyErr_SetString(PyExc_RuntimeError, "complex arrays not supported");
    goto exit;
  }

  irank = NI_GetArrayRank(input);
  /* input and structure must have equal rank: */
  if (irank != NI_GetArrayRank(strct)) {
    PyErr_SetString(PyExc_RuntimeError, 
		    "structure and input arrays must have equal rank");
    goto exit;
  }

  /* structuring element must be of bool type: */
  if (NI_GetArrayType(strct) != tBool) {
    PyErr_SetString(PyExc_RuntimeError, "structure type must be boolean");
    goto exit;
  }

  /* the structure array must be contigous: */
  if (!PyArray_ISCONTIGUOUS(strct)) {
    PyErr_SetString(PyExc_RuntimeError, "structure array must be contiguous");
    goto exit;
  }
  ps = (Bool*)NI_GetArrayData(strct);
  ssize = NI_Elements(strct);
  for(jj = 0; jj < ssize; jj++) 
    if (ps[jj]) ++struct_size;

  /* allocate output */
  NI_GetArrayDimensions(input, idims);
  if (!NI_OutputArray(tBool, irank, idims, output_in, output))
    goto exit;

  /* structure must be not be emtpy: */
  if (struct_size < 1) {
    PyErr_SetString(PyExc_RuntimeError, "structure size must be > 0");
    goto exit;
  }

  if (mask) {
    /* input and mask must have equal size: */
    if (!NI_ShapeEqual(input, mask)) {
      PyErr_SetString(PyExc_RuntimeError, 
		      "input and mask sizes must be equal");
      return 0;
    }
    /* iterator, data pointer and type of mask array: */
    if (!NI_InitPointIterator(mask, &mi))
      return 0;
    pm = NI_GetArrayData(mask);
    mtype = NI_GetArrayType(mask);
  }

  /* calculate the filter offsets: */
  if (!NI_InitFilterOffsetsFromArray(input, strct, shifts, NI_EXTEND_CONSTANT, 
				     &offsets, &border_flag_value))
    goto exit;

  /* initialize input element iterator: */
  if (!NI_InitPointIterator(input, &ii))
    goto exit;

  /* initialize output element iterator: */
  if (!NI_InitPointIterator(*output, &io))
    goto exit;

  /* initialize filter iterator: */
  NI_GetArrayDimensions(strct, sdims);
  if (!NI_InitFilterIterator(irank, sdims, struct_size, idims, shifts, &fi))
    goto exit;
    
  /* get data pointers an size: */
  pi = NI_GetArrayData(input);
  po = NI_GetArrayData(*output);
  size = NI_Elements(input);

  if (invert_input) {
    bdr_value = bdr_value ? 0 : 1;
    true = 0;
    false = 1;
  } else {
    bdr_value = bdr_value ? 1 : 0;
    true = 1;
    false = 0;
  }

  /* iterator over the elements: */
  oo = offsets;
  *changed = 0;
  msk_value = 1;
  for(jj = 0; jj < size; jj++) {
    if (mask) {							
      switch(mtype) {						
      case tBool:			
	msk_value = *(Bool*)pm ? 1 : 0;		
	break;			
      case tUInt8:		
	msk_value = *(UInt8*)pm ? 1 : 0;	
	break;			
      case tUInt16:		
	msk_value = *(UInt16*)pm ? 1 : 0;	
	break;			
      case tUInt32:		
	msk_value = *(UInt32*)pm ? 1 : 0;	
	break;			
#if HAS_UINT64
      case tUInt64:		
	msk_value = *(UInt64*)pm ? 1 : 0;	
	break;			
#endif
      case tInt8:			
	msk_value = *(Int8*)pm ? 1 : 0;		
	break;			
      case tInt16:		
	msk_value = *(Int16*)pm ? 1 : 0;	
	break;			
      case tInt32:		
	msk_value = *(Int32*)pm ? 1 : 0;	
	break;
      case tInt64:		
	msk_value = *(Int64*)pm ? 1 : 0;	
	break;
      case tFloat32:		
	msk_value = *(Float32*)pm ? 1 : 0;
	break;
      case tFloat64:		
	msk_value = *(Float64*)pm ? 1 : 0;	
	break;			
      default:			
	PyErr_SetString(PyExc_RuntimeError, "data type not supported");	
	return 0; 
      }	
    }
    switch (itype) {
    case tBool:
      NI_ERODE_POINT(pi, out, oo, struct_size, Bool, msk_value, bdr_value, 
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tUInt8:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt8, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tUInt16:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt16, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tUInt32:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt32, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
#if HAS_UINT64
    case tUInt64:
      NI_ERODE_POINT(pi, out, oo, struct_size, UInt64, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
#endif
    case tInt8:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int8, msk_value, bdr_value, 
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tInt16:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int16, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tInt32:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int32, msk_value, bdr_value, 
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tInt64:
      NI_ERODE_POINT(pi, out, oo, struct_size, Int64, msk_value, bdr_value, 
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tFloat32:
      NI_ERODE_POINT(pi, out, oo, struct_size, Float32, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
    case tFloat64:
      NI_ERODE_POINT(pi, out, oo, struct_size, Float64, msk_value, bdr_value,
		     border_flag_value, true, false, invert_output, *changed);
      break;
    default:
      PyErr_SetString(PyExc_RuntimeError, "data type not supported");
      goto exit;
    }
    *(Bool*)po = out;
    if (mask) {
      NI_FILTER_NEXT3(fi, ii, io, mi, oo, pi, po, pm);
    } else {
      NI_FILTER_NEXT2(fi, ii, io, oo, pi, po);
    }
  }

 exit:
  if (offsets) 
    free(offsets);
  return PyErr_Occurred() ? 0 : 1;
}

