#include <stdio.h>
#include "arrayobject.h"

/*
PyArrayObject *PyArray_FromDims(int nd, int *dims, int type)

This function will allocate an numarray of the specified dimensions and type.

An array created with PyArray_FromDims can be used as a temporary or returned
using PyArray_Return.
*/
static PyObject *
PyArray_FromDims(int nd, int*dims, int type)
{
	maybelong i, mb_dims[MAXDIM];
	for(i=0; i<MAXDIM; i++)
		mb_dims[i] = dims[i];
	return (PyObject *) NA_vNewArray(NULL, type, nd, mb_dims);
}


/* 
PyObject *PyArray_FromDimsAndData(nd,dims,type,data)

This function will allocate an appropriate numarray and copy contiguous region
'data' into it.  data is not freed.
*/
static PyObject *
PyArray_FromDimsAndData(int nd, int *dims, int type, char *data)
{
	PyArrayObject *num;
	num = (PyArrayObject *) PyArray_FromDims(nd, dims, type);
	if (!num) return NULL;
	memcpy(num->data, data, PyArray_NBYTES(num));
	return (PyObject *) num;
}

/*
PyObject *PyArray_ContiguousFromObject(op, type, min_dim, max_dim)

Returns a contiguous numarray of 'type' created from the sequence object 'op'.
If 'op' is a contiguous, aligned, non-byteswapped numarray of the correct
'type' and dimensions, then the op is INCREF'ed rather than copied.

Otherwise a new numarray will be created from 'op' and returned.

min_dim and max_dim bound the expected rank as in Numeric.

min_dim==max_dim specifies an exact rank.  min_dim==max_dim==0 specifies *any*
rank.
*/
static PyObject *
PyArray_ContiguousFromObject(PyObject  *op, int type, int min_dim, int max_dim)
{
	PyArrayObject *num = NA_IoArray(op, type, C_ARRAY);
	PyObject *num2 = (PyObject *) num;
	if (num && 
	    ((min_dim && (num->nd < min_dim)) || 
	    (max_dim && (num->nd > max_dim)))) {
		Py_DECREF(num);
		num2 = PyErr_Format(_Error,
				    "PyArray_ContiguousFromObject: array rank:%d "
				    "but required rank between %d and %d.",
				    (int) num->nd, min_dim, max_dim);
	}
	return num2;
}

/*
PyObject *PyArray_CopyFromObject(op,type,min_dim,max_dim)

Returns a contiguous array, similar to PyArray_FromContiguousObject, but
always returning a new array representation of the original sequence.

*/
static PyObject *
PyArray_CopyFromObject(PyObject  *op, int type, int min_dim, int max_dim)
{
	PyArrayObject *num = NA_InputArray(op, type, C_ARRAY);
	PyObject *num2 = (PyObject *) num;
	if (num && 
	    ((min_dim && (num->nd < min_dim)) || 
	     (max_dim && (num->nd > max_dim)))) {
		Py_DECREF(num);
		num2 = PyErr_Format(_Error,
				    "PyArray_CopyFromObject: array rank:%d"
				    "but required rank between %d and %d.",
				    (int) num->nd, min_dim, max_dim);
	}
	if (num2) {
		if (op == num2) {
			num2 = PyObject_CallMethod(op, "copy", NULL);
			if (!num2) goto _exit;
			/* Clear extra refcount from NA_InputArray. */
			Py_DECREF(num); 
		} else {  /* well-behaved temp becomes permanent copy */
			/* Detach from original since we want a copy, not a proxy. */
			if (num->_shadows) {
				Py_DECREF(num->_shadows);
				num->_shadows = NULL;
			}
		}
	} 
_exit:
	return num2;
}

/*
PyObject *PyArray_FromObject(op, type, min_dim, max_dim)

returns an array object based on 'op', possibly discontiguous.  The
strides array must be used to access elements of the array.

If 'op' is a byteswapped or misaligned numarray, FromObject creates a
temporary copy and the array object refers to it.

If 'op' is a nonswapped, aligned numarray, the emulation object refers to it.

If 'op' is some other sequence, it is converted to a numarray and the
array object refers to that.
*/
static PyObject *
PyArray_FromObject(PyObject  *op, int type, int min_dim, int max_dim)
{
	PyArrayObject *num = NA_IoArray(op, type, NUM_NOTSWAPPED | NUM_ALIGNED);
	PyObject *num2 = (PyObject *) num;
	if (num && 
	    ((min_dim && (num->nd < min_dim)) || 
	     (max_dim && (num->nd > max_dim)))) {
		goto _exit;
	} else {
		Py_DECREF(num);
		num2 = PyErr_Format(_Error,
				    "PyArray_FromObject: array rank:%d "
				    " but required rank between %d and %d.",
				    (int) num->nd, min_dim, max_dim);
	}
  _exit:
	return num2;
}

/* PyObject *PyArray_Return(apr)

Returns array object 'apr' to python.  In the case of "misbehaved" numarray,
apr must be a "well-behaved" array referring to the misbehaved original
via its _shadows attribute.

An additional check is (or should be) performed to guarantee that rank-0
numarray are converted to appropriate python scalars.

PyArray_Return has no net effect on the reference count of the underlying
numarray.
*/
static PyObject *
PyArray_Return(PyArrayObject *a)
{
	/* XXX Insert rank-0 check here. */
	if (a->_shadows) {
		PyObject *original = a->_shadows;
		/* Py_INCREF(original); */
		Py_DECREF(a);
		return original;
	}
	return (PyObject *) a;
}

/*  The code below is directly from Numeric: As1D, As2D, Free.    
*/
static int 
PyArray_As1D(PyObject **op, char **ptr, int *d1, int typecode) {
    PyArrayObject *ap;
	
    if ((ap = (PyArrayObject *)PyArray_ContiguousFromObject(*op, typecode, 1,
							    1)) == NULL)
	return -1;
	
    *op = (PyObject *)ap;
    *ptr = ap->data;
    *d1 = ap->dimensions[0];
    return 0;
}

static int 
PyArray_As2D(PyObject **op, char ***ptr, int *d1, int *d2, int typecode) {
    PyArrayObject *ap;
    int i, n;
    char **data;
	
    if ((ap = (PyArrayObject *)PyArray_ContiguousFromObject(*op, typecode, 2, 2)) == NULL)
	return -1;
	
    n = ap->dimensions[0];
    data = (char **)malloc(n*sizeof(char *));
    for(i=0; i<n; i++) {
	data[i] = ap->data + i*ap->strides[0];
    }
    *op = (PyObject *)ap;
    *ptr = data;
    *d1 = ap->dimensions[0];
    *d2 = ap->dimensions[1];
    return 0;
}


static int 
PyArray_Free(PyObject *op, char *ptr) {
    PyArrayObject *ap = (PyArrayObject *)op;
    int rval = 0;
    if (ap->nd > 2) { rval = -1; goto _exit ; }
    if (ap->nd == 2) {
	free(ptr);
    }
    Py_DECREF(ap);
  _exit:
    return rval;
}

static int 
PyArray_Check(PyObject *op) {
	/* See newarray.ch for pNumArrayClass, which is just 
	   PyObject * -> NumArray */
	int rval;
	if (deferred_libnumarray_init() < 0) { rval = -1; goto _exit; }
	rval = PyObject_IsInstance(op, pNumArrayClass);
  _exit:
	return rval;
}

static int 
PyArray_Size(PyObject *op) {
	PyArrayObject *a = (PyArrayObject *) op;
	int i, size;
	if (PyArray_Check(op)) {
		for(i=0, size=1; i<a->nd; i++)
			size *= a->dimensions[i];
		return size;
	} else {
		return 0;
	}
}

static int 
PyArray_CopyArray(PyArrayObject *a, PyArrayObject *b)
{
	PyObject *r;
	int rval;
	r = PyObject_CallMethod((PyObject *) a, "_copyFrom", "O", b);
	if (r) {
		Py_DECREF(r);
		rval = 0;		
	} else {
		rval = -1;
	}
	return rval;
}

static PyObject *
PyArray_Copy(PyArrayObject *a) {
	int i, dims[MAXDIM];
	PyObject *ret;
	for(i=0; i<MAXDIM; i++) {
		dims[i] = (int) a->dimensions[i];
	}
	ret = PyArray_FromDims(a->nd, dims, a->descr->type_num);
	if (ret && PyArray_CopyArray((PyArrayObject *) ret, a) == -1) {
		Py_XDECREF( ret );
		return NULL;
	}
	return ret;
}

static void 
PyArray_INCREF(PyArrayObject *a)
{
	Py_INCREF((PyObject *) a);
}

static void 
PyArray_XDECREF(PyArrayObject *a)
{
	Py_XDECREF((PyObject *) a);
}

/* PyArray_Cast:  see newarray.ch  */

static PyObject *
PyArray_GetItem(PyArrayObject *a, char *where)
{
	long offset = (where - a->data) - a->byteoffset;
	return NA_getPythonScalar( a, offset );
}

static int
PyArray_SetItem(PyArrayObject *a, char *where, PyObject *what)
{
	long offset = (where - a->data) - a->byteoffset;
	return NA_setFromPythonScalar( a, offset, what );
}

static char *
PyArray_Zero(PyArrayObject *a)
{
	static Bool zBool = 0;
	static Int8 zInt8 = 0;
	static UInt8 zUInt8 = 0;
	static Int16 zInt16 = 0;
	static UInt16 zUInt16 = 0;
	static Int32  zInt32 = 0;
	static UInt32 zUInt32 = 0;
	static Int64  zInt64 = 0;
	static UInt64 zUInt64 = 0;
	static Float32 zFloat32 = 0.0;
	static Float64 zFloat64 = 0.0;
	static Complex32 zComplex32 = { 0, 0 };
	static Complex64 zComplex64 = { 0, 0 }; 

	char *rval;

	switch(a->descr->type_num) {
	case tBool:      rval = (char *) &zBool; break;
	case tInt8:      rval = (char *) &zInt8; break;
	case tUInt8:     rval = (char *) &zUInt8; break;
	case tInt16:     rval = (char *) &zInt16; break;
	case tUInt16:    rval = (char *) &zUInt16; break;
	case tInt32:     rval = (char *) &zInt32; break;
	case tUInt32:    rval = (char *) &zUInt32; break;
	case tInt64:     rval = (char *) &zInt64; break;
	case tUInt64:    rval = (char *) &zUInt64; break;
	case tFloat32:   rval = (char *) &zFloat32; break;
	case tFloat64:   rval = (char *) &zFloat64; break;
	case tComplex32: rval = (char *) &zComplex32; break;
	case tComplex64: rval = (char *) &zComplex64; break;
	default:
		PyErr_Format( PyExc_TypeError, 
			      "Unknown type %d in PyArray_Zero", 
			      a->descr->type_num);
		rval =  NULL;
	}
	return rval;
}

static char *
PyArray_One(PyArrayObject *a)
{
	static Bool zBool = 1;
	static Int8 zInt8 = 1;
	static UInt8 zUInt8 = 1;
	static Int16 zInt16 = 1;
	static UInt16 zUInt16 = 1;
	static Int32  zInt32 = 1;
	static UInt32 zUInt32 = 1;
	static Int64  zInt64 = 1;
	static UInt64 zUInt64 = 1;
	static Float32 zFloat32 = 1.0;
	static Float64 zFloat64 = 1.0;
	static Complex32 zComplex32 = { 1, 0 };
	static Complex64 zComplex64 = { 1, 0 }; 

	char *rval;

	switch(a->descr->type_num) {
	case tBool:      rval = (char *) &zBool; break;
	case tInt8:      rval = (char *) &zInt8; break;
	case tUInt8:     rval = (char *) &zUInt8; break;
	case tInt16:     rval = (char *) &zInt16; break;
	case tUInt16:    rval = (char *) &zUInt16; break;
	case tInt32:     rval = (char *) &zInt32; break;
	case tUInt32:    rval = (char *) &zUInt32; break;
	case tInt64:     rval = (char *) &zInt64; break;
	case tUInt64:    rval = (char *) &zUInt64; break;
	case tFloat32:   rval = (char *) &zFloat32; break;
	case tFloat64:   rval = (char *) &zFloat64; break;
	case tComplex32: rval = (char *) &zComplex32; break;
	case tComplex64: rval = (char *) &zComplex64; break;
	default:
		PyErr_Format( PyExc_TypeError, 
			      "Unknown type %d in PyArray_Zero", 
			      a->descr->type_num);
		rval =  NULL;
	}

	return rval;
}

static PyArray_Descr descriptors[ ] = {
	{ tAny,       0,                 '*'}, 
	
	{ tBool,      sizeof(Bool),      '?'},

	{ tInt8,      sizeof(Int8),      '1'},
	{ tUInt8,     sizeof(UInt8),     'b'},

	{ tInt16,     sizeof(Int16),     's'},
	{ tUInt16,    sizeof(UInt16),    't'},

	{ tInt32,     sizeof(Int32),     'i'},
	{ tUInt32,    sizeof(UInt32),    'u'},

	{ tInt64,     sizeof(Int64),     'I'},
	{ tUInt64,    sizeof(UInt64),    'U'},

	{ tFloat32,   sizeof(Float32),   'f'},
	{ tFloat64,   sizeof(Float64),   'd'},

	{ tComplex32, sizeof(Complex32), 'F'},
	{ tComplex64, sizeof(Complex64), 'D'}
};

static PyArray_Descr *
PyArray_DescrFromType(int type)
{
	if ((type >= tAny) && (type <= tComplex64)) {
		return &descriptors[ type ];
	} else
		return NULL;
}

static PyArray_Descr *
PyArray_DescrFromTypeObj(PyObject *obj)
{
	int typeno = NA_typeObjectToTypeNo(obj);
	if (typeno < 0) return NULL;
	return PyArray_DescrFromType(typeno);
}

static int 
PyArray_CanCastSafely(int fromtype, int totype) {
    if (fromtype == totype) return 1;

    switch(fromtype) {
    case PyArray_CHAR:
        return 0;
#if PyArray_CHAR != PyArray_UBYTE
    case PyArray_UBYTE:
#else
    case PyArray_SBYTE:
#endif
    case PyArray_SHORT:
        return (totype >= fromtype);
    case PyArray_INT:
        return (totype >= PyArray_INT) && (totype != PyArray_FLOAT);
#if LP64                 /* Else PyArray_LONG == PyArray_INT */
    case PyArray_LONG:   
	    return 0;    /* 64-bit ints lose precision no matter what */
#endif
    case PyArray_FLOAT:
        return (totype > PyArray_FLOAT);
    case PyArray_DOUBLE:
        return (totype == PyArray_CDOUBLE);
    case PyArray_CFLOAT:
        return (totype == PyArray_CDOUBLE);
    case PyArray_CDOUBLE:
        return 0;
    default:
        return 0;
    }
}

static int 
_PyArray_multiply_list(int *l1, int n) {
    int s=1, i=0;
    while (i < n) s *= l1[i++];
    return s;
}
static int 
_PyArray_compare_lists(int *l1, int *l2, int n) {
    int i;
    for(i=0;i<n;i++) {
	if (l1[i] != l2[i]) return 0;
    } 
    return 1;
}
/*
 * Local Variables:
 * mode: C
 * c-file-style: "python"
 * End:
 */
