// 
//  Copyright (c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu <cengiz@isi.edu>

#include "SetOfSymbol.hh"
#include "debug.hh"


char *SetOfSymbol_format(int i) {
   static char formatbuffer[16]; // This variable is allocated on data segment

   sprintf(formatbuffer, "%d", i);
   return formatbuffer;
}

void SetOfSymbol::do_print (ostream& stream) {
   stream << *this;
}

ostream& operator<<(ostream& stream, SetOfSymbol& set) {
   Pix i, k;
   SymbolConjunctPtr conjunct;

   for (i = set.conjuncts.first(); i;  ) {
      conjunct = set.conjuncts(i);

      stream << set.format_open;
      for (k = conjunct->symbols.first(); k;  ) {
	 stream << (*set.format)(conjunct->symbols(k));
	 conjunct->symbols.next(k);
	 if (k)
	    stream << set.format_delimiter;
      }

      stream << set.format_close;

      set.conjuncts.next(i);
      if (i)
	 stream << " AND ";
   }

   return stream;
}

SetOfSymbol::~SetOfSymbol() {
   clear();
}

void SetOfSymbol::clear() {
   Pix i;

   for (i = conjuncts.first(); i;  conjuncts.next(i))
      delete conjuncts(i);

   conjuncts.clear();
}

int SetOfSymbol::add(int  item) {  // add item; return non-zero if successfull
   SymbolConjunctPtr conjunct;
   Pix i;

   i = conjuncts.first();
   
   if (i)
      conjunct = conjuncts(i);
   else {
      conjunct = new SymbolConjunct;
      if (!conjuncts.add(conjunct)) {
	 delete conjunct;
	 return 0;
      }
   }
   return (int) conjunct->add(item);
}

// assumes *ORDERED* bag implementation
// this may not be reducing to the minimum number of conjuncts
void SetOfSymbol::reduce() {
   Pix i, j;
   _SetOfInt s;
   
   // { s1, s2, s3 } INTERSECTION { s1, s2 } is { s1, s2 }
   for (i = conjuncts.first(); i; conjuncts.next(i))
      for (j = i, conjuncts.next(j); j; conjuncts.next(j))
	 if (conjuncts(i)->symbols <= conjuncts(j)->symbols) {
	    delete conjuncts(j);
	    conjuncts.del_this(j);
	 }

   // for each singleton conjunct { s1 }, insert ~s1 into s
   for (i = conjuncts.first(); 
	i && conjuncts(i)->symbols.length() == 1; 
	conjuncts.next(i))
      s.add((int) (*STable).contents((Pix) conjuncts(i)->symbols(conjuncts(i)->symbols.first())).not);
   
   // if { s1, s2, s3 } SUBSETEQ s, then the result is empty
   // E.g. { s1, s2, s3 } INT { ~s1 } INT { ~s2 } INT { ~s3 } = empty
   // since s = { ~~s1, ~~s2, ~~s3 }
   // E.g. { 690 } INT { ~690 } = empty
   for (i = conjuncts.first(); i; conjuncts.next(i))
      if (conjuncts(i)->symbols <= s) {
	 clear();
	 return;
      }
}

void SetOfSymbol::operator &= (SetOfSymbol& b) { // intersection
   SymbolConjunctPtr conjunct;
   Pix i;

   if (empty() || b._universal)
      return;

   if (b.empty()) { // result is empty
      clear();
      _universal = 0;
      return;
   }

   // append conjuncts of b into conjuncts of *this
   for (i = b.conjuncts.first(); i; b.conjuncts.next(i)) {
      conjunct = new SymbolConjunct(*b.conjuncts(i));
      conjuncts.add(conjunct);
   }

   if (_universal)
      _universal = b._universal;
   else
      reduce();
}

void SetOfSymbol::do_symbolic_or(_SetOfInt& a, _SetOfInt& b) {
   Pix i;

   a |= b;
   // check to see whether we became universal
   // assumes sorted set implementation 
   // and also assumes ~s1 is the next element in the symbol table after s1
   // hence if s1 and ~s1 are in the same set they appear next to each other
   // since the set is sorted and there is no symbol with pix in the middle
   for (i = a.first(); i; a.next(i)) {
      if (a.seek((int) (*STable).contents((Pix) a(i)).not)) { 
	 a.clear();
	 return;
      }
   }
}

void SetOfSymbol::operator |= (SetOfSymbol& b) { // union
   SymbolConjunctPtr conjunct;
   Pix i, j;
   SetOfSymbol tmp(this->STable);

   if (_universal || b.empty())
      return; 

   if (b._universal) { // result is universal
      clear();
      _universal = 1;
      return;
   }

   if (empty()) { // become b
      for (i = b.conjuncts.first(); i; b.conjuncts.next(i)) {
	 conjunct = new SymbolConjunct(*b.conjuncts(i));
	 conjuncts.add(conjunct);
      }
      return;
   }

   // copy conjuncts into conjuncts of tmp
   for (i = conjuncts.first(); i; conjuncts.next(i)) {
      conjunct = new SymbolConjunct(*conjuncts(i));
      tmp.conjuncts.add(conjunct);
   }

   // empty *this
   clear();

   // cartesian product b with tmp result into *this
   for (i = tmp.conjuncts.first(); i; tmp.conjuncts.next(i)) {
      for (j = b.conjuncts.first(); j; b.conjuncts.next(j)) {
	 conjunct = new SymbolConjunct(*tmp.conjuncts(i));
	 do_symbolic_or(conjunct->symbols, b.conjuncts(j)->symbols);
	 if (!conjunct->symbols.empty()) 
            // this actually means universal conjunct
	    conjuncts.add(conjunct);
	 else 
	    delete conjunct;
      }
   }
   
   // check to see whether we became universal
   if (conjuncts.empty())
      _universal = 1;
   else
      reduce();
}

void SetOfSymbol::operator ~  () { // complement
   SymbolConjunctPtr conjunct;
   Pix i, j;
   SetOfSymbol tmp(this->STable), tmp2(this->STable);

   if (_universal) {
      ASSERT(conjuncts.empty());
      _universal = 0;
      return;
   } 
   if (empty()) {
      _universal = 1;
      return;
   }

   // copy conjuncts into conjuncts of tmp
   for (i = conjuncts.first(); i; conjuncts.next(i)) {
      conjunct = new SymbolConjunct(*conjuncts(i));
      tmp.conjuncts.add(conjunct);
   }

   // empty *this
   clear();

   for (i = tmp.conjuncts.first(); i; tmp.conjuncts.next(i)) {
      tmp2.clear();
      for (j = tmp.conjuncts(i)->symbols.first(); 
	   j; 
	   tmp.conjuncts(i)->symbols.next(j)) {
	 conjunct = new SymbolConjunct;
	 conjunct->symbols.add((int) (*STable).contents((Pix) tmp.conjuncts(i)->symbols(j)).not); 
	    // take not here
	 tmp2.conjuncts.add(conjunct);
      }
      *this |= tmp2;
   }

}   

void SetOfSymbol::operator =  (SetOfSymbol& b) { // assignment
   SymbolConjunctPtr conjunct;
   Pix i;

   _universal = b._universal;

   clear();
   // copy conjuncts of b into conjuncts
   for (i = b.conjuncts.first(); i; b.conjuncts.next(i)) {
      conjunct = new SymbolConjunct(*b.conjuncts(i));
      conjuncts.add(conjunct);
   }

}

// Below code assumes *ORDERED* bag implementation
int  SetOfSymbol::operator == (SetOfSymbol& b) { // equivalance
   Pix i, j, lastj;
   int lastlength, found;
   
   if (_universal != b._universal)
      return 0;

   if (_universal == 1)
      return 1;

   if (conjuncts.length() != b.conjuncts.length())
      return 0;

   // for each conjunct find a matching conjunct in b
   lastj = (Pix) 0;
   lastlength = -1;
   for (i = conjuncts.first(); i; conjuncts.next(i)) {
      found = 0;

      if (lastlength != conjuncts(i)->symbols.length()) {
	 lastlength = conjuncts(i)->symbols.length();
	 lastj = b.conjuncts.seek(conjuncts(i), lastj);
      }

      for (j = lastj; 
	   j && lastlength == b.conjuncts(j)->symbols.length(); 
	   b.conjuncts.next(j))
	 if (conjuncts(i)->symbols == b.conjuncts(j)->symbols) {
	    found = 1;
	    break;
	 }

      if (!found)
	 return 0;
   }
   
   return 1;
}
