// 
//  Copyright_SetOfInt(c) 1994 by the University of Southern California
//  and/or the International Business Machines Corporation.
//  All rights reserved.
//
//  Permission to use, copy, modify, and distribute this software and
//  its documentation in source and binary forms for lawful
//  non-commercial purposes and without fee is hereby granted, provided
//  that the above copyright notice appear in all copies and that both
//  the copyright notice and this permission notice appear in supporting
//  documentation, and that any documentation, advertising materials,
//  and other materials related to such distribution and use acknowledge
//  that the software was developed by the University of Southern
//  California, Information Sciences Institute and/or the International
//  Business Machines Corporation.  The name of the USC or IBM may not
//  be used to endorse or promote products derived from this software
//  without specific prior written permission.
//
//  NEITHER THE UNIVERSITY OF SOUTHERN CALIFORNIA NOR INTERNATIONAL
//  BUSINESS MACHINES CORPORATION MAKES ANY REPRESENTATIONS ABOUT
//  THE SUITABILITY OF THIS SOFTWARE FOR ANY PURPOSE.  THIS SOFTWARE IS
//  PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
//  INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
//  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND 
//  NON-INFRINGEMENT.
//
//  IN NO EVENT SHALL USC, IBM, OR ANY OTHER CONTRIBUTOR BE LIABLE FOR ANY
//  SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES, WHETHER IN CONTRACT,
//  TORT, OR OTHER FORM OF ACTION, ARISING OUT OF OR IN CONNECTION WITH,
//  THE USE OR PERFORMANCE OF THIS SOFTWARE.
//
//  Questions concerning this software should be directed to 
//  info-ra@isi.edu.
//
//  Author(s): Cengiz Alaettinoglu (cengiz@isi.edu)

#include <iostream.h>
#include "CommDBSelMap.h"

extern "C" {
//#include <stdlib.h>
//#include <string.h>
//#include <memory.h>
}

Error* CommDBSelMap::get_routes(DBCnxn &dbcnxn, 
                                const char *comm_name, 
                                char cmd, 
                                char *&routes)
{
    char rc,
         *ptr,
         buf[100];
    int  bytes_read,
         bytes_expected,       // Total num of bytes we expect to read.
         num_routes;           // Number of routes read.

	sprintf(buf,"%c%s\n", cmd, comm_name);
	(void) dbcnxn.write_query(buf); // Write query. Throw away return code.

	// Read the first line to determine how much mem to malloc.
	rc = dbcnxn.read_response( bytes_read, buf, sizeof(buf) );

	if ( rc != 'A' ) {                // Verify that we did in fact read the
		routes = NULL;
		return( new Error(rc, buf) );   // line type we expected to read.
	}

	bytes_expected = atoi( buf );

	rc = dbcnxn.readn_response( bytes_read, routes, bytes_expected );

	if ( rc != 'B') {
		cerr << "Error:" << __FILE__ <<"("<<__LINE__<<") Expected to read ";
		cerr << bytes_expected << " but only read "<< bytes_read << " bytes.\n";
		cerr << "Error: Last 1-byte code read was '" << rc << "'.\n'";
		abort();
	}

	// Have read the exact number of bytes that we were told to expect, we
	// now expect to see the End of Data line.  If we don't read an EOD line,
	// we clean up and return an error.
	{
		int tmp_bytes_read;
		if ((rc = dbcnxn.read_response( tmp_bytes_read, buf, sizeof(buf) )) != 'C')
		{
			Error *error = new Error(rc, buf);
			delete[] routes;
			routes = (char *) NULL;
			return( error );
		}
	}
	
	return( (Error *) NULL);
}

Error* CommDBSelMap::get_routes_by_comm(DBCnxn &dbcnxn, 
                                        const char *comm_name, 
                                        char *&routes)
{
	Error *error;

	error = get_routes( dbcnxn, comm_name, 'h', routes );

	return( error );
}

Error* CommDBSelMap::get_routes_by_dbsel( DBCnxn &dbcnxn, 
                                          const char *key, 
                                          char *&routes )
{
    Error *error;
    char code,
         lkey[100];

	if ( strchr( key, (int) '*' ) ) {
		code = 'h';
		strcpy( lkey, "COMM_NSFNET" );
	}

	else {
		code = 'j';
		strcpy( lkey, key );
	}

	error = get_routes( dbcnxn, lkey, code, routes );

	return( error );
}

CommDBSelMap::CommDBSelMap( PrefaskAVLSet *pfsk_tbl, SymTab& dflt ) : STNameSymTabAVLMap( dflt )
{
	prefask_symtbl = pfsk_tbl;
}

// For every Community Name and DB Selector entry in this symbol table,
// try to expand it to a list of prefasks.  For each prefask, insert it into
// the PrefaskST symbol table.  Each time we do this we get a Pix of the
// prefask entry.  Store this Pix in the _SetOfInt associated with the
// SymTab node we are currently operating on.

void CommDBSelMap::expand( DBCnxn *dbptr )
{
	CommDBSelMap &comm_symtbl = *this;  
	SymTab contents;
	PrefaskAVLSet &prfsk_symtbl = *prefask_symtbl;
	Pix p, prefask_pix;
	int i;
	char *key, *array, *prefask;
	Error *error;

	for( p = comm_symtbl.first(); p; comm_symtbl.next(p) ) {

		// Check the evaluated field to see if we've expanded this particular
		// entry before.  If so, don't waste time expanding again.
		// A contents.evaluated value of 0 means that we haven'y yet expanded
		// this particular entry.

		contents = comm_symtbl.contents(p);
		if ( !contents.evaluated ) { 

			key = comm_symtbl.key(p);	
			// If we find a colon or an asterisk, call it a DB Selector string.
			if ( strpbrk( key, "*:" ) ) 
				error = get_routes_by_dbsel( *dbptr, key, array );
	
			else
				error = get_routes_by_comm( *dbptr, key, array );
	
			if ( error ) {
				cerr << "Warning: Unable to expand macro: " << key << endl;
				delete error;
			}
	
			else {
				comm_symtbl[key].evaluated = 1;

				// Add newly discovered routes to the set.
				// Makes entries for newly discovered routes in the Prefask Map.
				prefask = strtok( array, " " );
				while ( prefask ) {
					if ( !( prefask_pix = prfsk_symtbl.seek( prefask ) ) )
						prefask_pix = prfsk_symtbl.add( strdup( prefask ) );
	
						comm_symtbl[key].value.add( (int) prefask_pix );

						prefask = strtok( (char *) NULL, " " );
				}
	
				delete[] array;
			}
		}
	}
}

void CommDBSelMap::dump( void )
{
	Pix prefask_pix, p, q;
	char *key;
	SymTab contents;
	CommDBSelMap &comm_symtbl=*this;
	PrefaskAVLSet &lprefask_symtbl=*prefask_symtbl;

	for( p = comm_symtbl.first(); p; comm_symtbl.next(p) ) {
		key = comm_symtbl.key(p);	
		contents = comm_symtbl.contents(p);

		cout << key << " " << contents.evaluated << endl;

		for( q = contents.value.first(); q; contents.value.next(q) ) {
			prefask_pix = (Pix) contents.value(q);
			cout << "    " << lprefask_symtbl( prefask_pix ) << endl;
		}
	}
}
