/*
 * Copyright (c) 1990, 1992, 1993, 1994, 1995
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
static char copyright[] =
    "Copyright (c) 1990, 1992, 1993, 1994, 1995\n\
The Regents of the University of California.  All rights reserved.\n";
static char rcsid[] =
    "@(#) $Header: arpwatch.c,v 1.39 95/04/08 16:52:22 leres Exp $ (LBL)";
#endif

/*
 * arpwatch - keep track of ethernet/ip address pairings, report changes
 */

#include <sys/param.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/socket.h>

#include <net/if.h>

#include <netinet/in.h>
#include <netinet/if_ether.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include <pcap.h>

#include <lbl/gnuc.h>

#include "arpwatch.h"
#include "db.h"
#include "ec.h"
#include "file.h"
#include "os.h"

#define ARPWATCH_SNAPLEN \
    (sizeof(struct ether_header) + sizeof(struct ether_arp))

/* Forwards */
SIGRET	checkpoint(int);
SIGRET	die(int);
int	dump(void);
void	dumpone(u_long, u_char *, time_t, char *);
int	main(int, char **);
void	process(u_char *, const struct pcap_pkthdr *, const u_char *);
int	readdata(void);
int	sanity(struct ether_header *, struct ether_arp *, int);

static char arpdir[] = ARPDIR;
static char newarpfile[] = NEWARPFILE;
static char arpfile[] = ARPFILE;
static char oldarpfile[] = OLDARPFILE;
static char ethercodes[] = ETHERCODES;

/* Broadcast ethernet addresses */
static u_char zero[6] = { 0, 0, 0, 0, 0, 0 };
static u_char allones[6] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };

char *prog;

int debug = 0;
int can_checkpoint = 0;
int initializing = 1;			/* true if initializing */

static u_long net;
static u_long netmask;
static u_long net252;						/* XXX */
static u_long net32;						/* XXX */

int fddipad;		/* XXX libpcap XXX! */

int
main(argc, argv)
	int argc;
	char **argv;
{
	register char *cp;
	register int op, pid, snaplen, timeout, linktype;
#ifdef TIOCNOTTY
	int fd;
#endif
	register pcap_t *pd;
	register char *interface;
	struct bpf_program code;
	char errbuf[PCAP_ERRBUF_SIZE];

	if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	opterr = 0;
	interface = NULL;
	while ((op = getopt(argc, argv, "di:")) != EOF)
		switch (op) {

		case 'd':
			++debug;
#ifndef DEBUG
			(void)fprintf(stderr,
			    "%s: Warning: Not compiled with -DDEBUG\n", prog);
#endif
			break;

		case 'i':
			interface = optarg;
			break;

		default:
			(void)fprintf(stderr,
			    "usage: %s [-d] [-i interface]\n", prog);
			exit(1);
			/* NOTREACHED */
		}

	/* Determine interface if not specified */
	if (interface == NULL && (interface = pcap_lookupdev(errbuf)) == NULL) {
		(void)fprintf(stderr, "%s: lookup_device: %s\n", prog, errbuf);
		exit(1);
	}

	/* Determine network and netmask */
	if (pcap_lookupnet(interface, &net, &netmask, errbuf) < 0) {
		(void)fprintf(stderr, "%s: bad interface %s: %s\n",
		    prog, interface, errbuf);
		exit(1);
	}

	/* Drop into the background if not debugging */
	if (!debug) {
		pid = fork();
		if (pid < 0) {
			syslog(LOG_ERR, "main fork(): %m");
			exit(1);
		} else if (pid != 0)
			exit(0);
		(void)close(fileno(stdin));
		(void)close(fileno(stdout));
		(void)close(fileno(stderr));
#ifdef TIOCNOTTY
		fd = open("/dev/tty", O_RDWR);
		if (fd >= 0) {
			(void)ioctl(fd, TIOCNOTTY, 0);
			(void)close(fd);
		}
#else
		(void) setsid();
#endif
	}

	openlog(prog, 0, LOG_DAEMON);

	if (chdir(arpdir) < 0) {
		syslog(LOG_ERR, "chdir(%s): %m", arpdir);
		syslog(LOG_ERR, "(using current working directory)");
	}

	snaplen = ARPWATCH_SNAPLEN;
	timeout = 1000;
	pd = pcap_open_live(interface, snaplen, 1, timeout, errbuf);
	if (pd == NULL) {
		syslog(LOG_ERR, "pcap open: %s", errbuf);
		exit(1);
	}

	/* Must be 10 Mbit ethernet */
	linktype = pcap_datalink(pd);
	if (linktype != DLT_EN10MB) {
		syslog(LOG_ERR, "Link layer type %d not 10 Mbit ethernet",
		    linktype);
		exit(1);
	}

	/* Compile and install filter */
	if (pcap_compile(pd, &code, "arp or rarp", 1, netmask) < 0) {
		syslog(LOG_ERR, "pcap_compile: %s", pcap_geterr(pd));
		exit(1);
	}
	if (pcap_setfilter(pd, &code) < 0) {
		syslog(LOG_ERR, "pcap_setfilter: %s", pcap_geterr(pd));
		exit(1);
	}
	syslog(LOG_INFO, "listening on %s", interface);

	/* Read in database */
	initializing = 1;
	if (!readdata())
		exit(1);
	sorteinfo();
#ifdef DEBUG
	if (debug > 2) {
		debugdump();
		exit(0);
	}
#endif
	initializing = 0;

	(void)signal(SIGINT, (SIGRET *)die);
	(void)signal(SIGTERM, (SIGRET *)die);
	(void)signal(SIGHUP, (SIGRET *)die);
	(void)signal(SIGQUIT, (SIGRET *)checkpoint);
	(void)signal(SIGALRM, (SIGRET *)checkpoint);
	(void)alarm(CHECKPOINT);

	net252 = inet_addr("128.3.252.0");			/* XXX */
	net32 = inet_addr("128.3.32.0");			/* XXX */

	pcap_loop(pd, 0, process, NULL);
	syslog(LOG_ERR, "pcap_loop: %s", pcap_geterr(pd));
	exit(1);
}

/* syslog() helper routine */
void
dosyslog(p, s, a, ea, ha)
	int p;
	char *s;
	u_long a;
	u_char *ea, *ha;
{
	char xbuf[64];

	/* No report until we're initialized */
	if (initializing)
		return;

	/* Display both ethernet addresses if they don't match */
	(void)strcpy(xbuf, e2str(ea));
	if (ha != 0 && BCMP(ea, ha, 6) != 0) {
		(void)strcat(xbuf, " (");
		(void)strcat(xbuf, e2str(ha));
		(void)strcat(xbuf, ")");
	}

	syslog(p, "%s %s %s", s, intoa(a), xbuf);
}

/* Process an arp/rarp packet */
void
process(u, h, p)
	u_char *u;
	const struct pcap_pkthdr *h;
	const u_char *p;
{
	register struct ether_header *eh;
	register struct ether_arp *ea;
	register u_char *sea, *sha;
	register time_t t;
	u_long sia;

	eh = (struct ether_header *)p;
	ea = (struct ether_arp *)(eh + 1);

	if (!sanity(eh, ea, h->caplen))
		return;

	/* Source hardware ethernet address */
	sea = (u_char *)&eh->ether_shost;

	/* Source ethernet address */
	sha = (u_char *)&ea->arp_sha;

	/* Source ip address */
	BCOPY(ea->arp_spa, &sia, 4);

	/* Watch for bogons */
	/* XXX make 128.3.32 an alias for 128.3.252 during the transition */
	if (net == net252 && (sia & netmask) == net32) {
		/* OK; we're using 128.3.252 and packet is using 128.3.32 */
	} else if (net == net32 && (sia & netmask) == net252) {
		/* OK; we're using 128.3.32 and packet is using 128.3.252 */
	} else if ((sia & netmask) != net) {
		dosyslog(LOG_INFO, "bogon", sia, sea, sha);
		return;
	}

	/* Watch for ethernet broadcast */
	if (BCMP(sea, zero, 6) == 0 || BCMP(sea, allones, 6) == 0 ||
	    BCMP(sha, zero, 6) == 0 || BCMP(sha, allones, 6) == 0) {
		dosyslog(LOG_INFO, "ethernet broadcast", sia, sea, sha);
		return;
	}

	/* Double check ethernet addresses */
	if (BCMP(sea, sha, 6) != 0) {
		dosyslog(LOG_INFO, "ethernet mismatch", sia, sea, sha);
		return;
	}

	/* Got a live one */
	t = h->ts.tv_sec;
	can_checkpoint = 0;
	if (!ent_add(sia, sea, t, NULL))
		syslog(LOG_ERR, "ent_add(%s, %s, %d) failed",
		    intoa(sia), e2str(sea), t);
	can_checkpoint = 1;
}

/* Perform sanity checks on arp/rarp packet */
int
sanity(eh, ea, len)
	register struct ether_header *eh;
	register struct ether_arp *ea;
	int len;
{
	/* XXX use bsd style ether_header to avoid messy ifdef's */
	struct bsd_ether_header {
		u_char  ether_dhost[6];
		u_char  ether_shost[6];
		u_short ether_type;
	};
	register u_char *shost = ((struct bsd_ether_header *)eh)->ether_shost;

	eh->ether_type = ntohs(eh->ether_type);
	ea->arp_hrd = ntohs(ea->arp_hrd);
	ea->arp_pro = ntohs(ea->arp_pro);
	ea->arp_op = ntohs(ea->arp_op);

	if (len < sizeof(*eh) + sizeof(*ea)) {
		syslog(LOG_ERR, "short (want %d)\n", sizeof(*eh) + sizeof(*ea));
		return(0);
	}

	/* XXX sysv r4 seems to use hardware format 6 */
	if (ea->arp_hrd != ARPHRD_ETHER && ea->arp_hrd != 6) {
		syslog(LOG_ERR, "%s sent bad hardware format 0x%x\n",
		    e2str(shost), ea->arp_hrd);
		return(0);
	}

	/* XXX hds X terminals sometimes send trailer arp replies */
	if (ea->arp_pro != ETHERTYPE_IP && ea->arp_pro != ETHERTYPE_TRAIL) {
		syslog(LOG_ERR, "%s sent packet not ETHERTYPE_IP (0x%x)\n",
		    e2str(shost), ea->arp_pro);
		return(0);
	}

	if (ea->arp_hln != 6 || ea->arp_pln != 4) {
		syslog(LOG_ERR, "%s sent bad addr len (hard %d, prot %d)\n",
		    e2str(shost), ea->arp_hln, ea->arp_pln);
		return(0);
	}

	/*
	 * We're only interested in arp requests, arp replies
	 * and reverse arp replies
	 */
	if (eh->ether_type == ETHERTYPE_ARP) {
		if (ea->arp_op != ARPOP_REQUEST &&
		    ea->arp_op != ARPOP_REPLY) {
			syslog(LOG_ERR, "%s sent wrong arp op %d\n",
			     e2str(shost), ea->arp_op);
			return(0);
		}
	} else if (eh->ether_type == ETHERTYPE_REVARP) {
		if (ea->arp_op == REVARP_REQUEST) {
			/* no useful information here */
			return(0);
		} else if (ea->arp_op != REVARP_REPLY) {
			if (debug)
				syslog(LOG_ERR, "%s sent wrong revarp op %d\n",
				    e2str(shost), ea->arp_op);
			return(0);
		}
	} else {
		syslog(LOG_ERR, "%s sent bad type 0x%x\n",
		    e2str(shost), eh->ether_type);
		return(0);
	}

	return(1);
}

SIGRET
die(signo)
	int signo;
{

	syslog(LOG_DEBUG, "exiting");
	checkpoint(0);
	exit(1);
}

SIGRET
checkpoint(signo)
	int signo;
{

	if (!can_checkpoint)
		(void)alarm(1);
	else {
		(void)alarm(0);
		(void)dump();
		(void)alarm(CHECKPOINT);
	}
	return SIGRETVAL;
}

static FILE *dumpf;

void
dumpone(a, e, t, h)
	u_long a;
	u_char *e;
	time_t t;
	char *h;
{
	(void)fprintf(dumpf, "%s\t%s", e2str(e), intoa(a));
	if (t != 0 || h != NULL)
		(void)fprintf(dumpf, "\t%ld", t);
	if (h != NULL)
		(void)fprintf(dumpf, "\t%s", h);
	(void)putc('\n', dumpf);
}

int
dump()
{
	int fd;

	if ((fd = creat(newarpfile, 0644)) < 0) {
		syslog(LOG_ERR, "creat(%s): %m", newarpfile);
		return(0);
	}
	if ((dumpf = fdopen(fd, "w")) == NULL) {
		syslog(LOG_ERR, "fdopen(%s): %m", newarpfile);
		return(0);
	}

	(void)ent_loop(dumpone);

	(void)fclose(dumpf);
	if (rename(arpfile, oldarpfile) < 0) {
		syslog(LOG_ERR, "rename %s -> %s: %m", arpfile, oldarpfile);
		return(0);
	}
	if (rename(newarpfile, arpfile) < 0) {
		syslog(LOG_ERR, "rename %s -> %s: %m", newarpfile, arpfile);
		return(0);
	}
	return(1);
}

/* Initialize the databases */
int
readdata()
{
	register FILE *f;

	if ((f = fopen(arpfile, "r")) == NULL) {
		syslog(LOG_ERR, "fopen(%s): %m", arpfile);
		return(0);
	}
	if (!file_loop(f, ent_add)) {
		(void)fclose(f);
		return(0);
	}
	(void)fclose(f);

	/* It's not fatal if we can't open the ethercodes file */
	if ((f = fopen(ethercodes, "r")) != NULL) {
		(void)ec_loop(f, ec_add);
		(void)fclose(f);
	}

	return(1);
}
