/*
 * Copyright (c) 1990, 1991, 1992, 1993, 1994, 1995
 *	The Regents of the University of California.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
static char rcsid[] =
    "@(#) $Header: report.c,v 1.31 95/04/08 16:49:25 leres Exp $ (LBL)";
#endif

/*
 * report - arpwatch report generating routines
 */

#include <sys/param.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/wait.h>

#include <net/if.h>

#include <netinet/in.h>

#include <arpa/inet.h>

#include <ctype.h>
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#include <lbl/gnuc.h>

#include "arpwatch.h"
#include "dns.h"
#include "ec.h"
#include "os.h"
#include "report.h"
#include "addresses.h"

#ifndef PATH_SENDMAIL
#define PATH_SENDMAIL "/usr/lib/sendmail"
#endif

#define PLURAL(n) ((n) == 1 || (n) == -1 ? "" : "s")

static int cdepth;	/* number of outstanding children */

static char *fmtdate(time_t);
static char *fmtdelta(time_t);
static char *fmtzone(int, int);
static char *fmtzonename(int, struct tm *);
SIGRET reaper(int);

static char *
fmtdelta(t)
	time_t t;
{
	register char *cp;
	int minus = 0;
	static char buf[132];

	if (t < 0) {
		t = -t;
		++minus;
	}
	if (t < 60) {
		cp = "second";
	} else if (t < 60 * 60) {
		t /= 60;
		cp = "minute";
	} else if (t < 24 * 60 * 60) {
		t /= (60 * 60);
		cp = "hour";
	} else {
		t /= (24 * 60 * 60);
		cp = "day";
	}
	if (minus)
		t = -t;
	(void)sprintf(buf, "%ld %s%s", t, cp, PLURAL(t));
	return(buf);
}

static char *dow[7] = {
	"Sunday",
	"Monday",
	"Tuesday",
	"Wednesday",
	"Thursday",
	"Friday",
	"Saturday"
};

static char *moy[12] = {
	"January",
	"February",
	"March",
	"April",
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December"
};

#define DOW(d) ((d) < 0 || (d) >= 7 ? "?" : dow[d])
#define MOY(m) ((m) < 0 || (m) >= 12 ? "?" : moy[(m)])

static char *
fmtzone(mw, isdst)
	int mw, isdst;
{
	char ch;
	static char buf[32];

	if (isdst)
		mw -= 60;
	if (mw < 0) {
		ch = '+';
		mw = -mw;
	} else {
		ch = '-';
	}
	(void)sprintf(buf, "%c%02d%02d", ch, mw / 60, mw % 60);
	return(buf);
}

static char *
fmtzonename(mw, tm)
	register int mw;
	register struct tm *tm;
{
#ifndef SVR4
	return (timezone(mw, tm->tm_isdst));
#else
	static char buf[32];

	(void)strftime(buf, sizeof(buf), "%Z", tm);
	return (buf);
#endif
}

static char *
fmtdate(t)
	time_t t;
{
	register struct tm *tm;
	static char buf[132];
	static int init = 0;
	static int zone;
#ifndef SVR4
	struct timeval tv;		/* not used but sometimes required */
	struct timezone tz;
#endif

	if (!init) {
		++init;
#ifndef SVR4
		(void)gettimeofday(&tv, &tz);
		zone = tz.tz_minuteswest;
#else
		zone = timezone / 60;
#endif
	}

	if (t == 0)
		return("<no date>");

	tm = localtime(&t);
	(void)sprintf(buf, "%s, %s %d, %d %d:%02d:%02d %s (%s)",
	    DOW(tm->tm_wday),
	    MOY(tm->tm_mon),
	    tm->tm_mday,
	    tm->tm_year + 1900,
	    tm->tm_hour,
	    tm->tm_min,
	    tm->tm_sec,
	    fmtzone(zone, tm->tm_isdst),
	    fmtzonename(zone, tm));
	return(buf);
}

void
reaper(signo)
	int signo;
{
	register pid_t pid;
	WAITSTATUS status;

	for (;;) {
		pid = waitpid((pid_t)0, &status, WNOHANG);
		if ((int)pid < 0) {
			/* ptrace foo */
			if (errno == EINTR)
				continue;
			/* ECHILD means no one left */
			if (errno != ECHILD)
				syslog(LOG_ERR, "reaper: %m");
			break;
		}
		/* Already got everyone who was done */
		if (pid == 0)
			break;
		--cdepth;
		if (debug > 1)
			syslog(LOG_DEBUG, "reaper: child %d", pid);
		if (WEXITSTATUS(status))
			syslog(LOG_DEBUG, "reaper: pid %d, exit status %d",
			    pid, WEXITSTATUS(status));
	}
	return SIGRETVAL;
}

void
report(title, a, e1, e2, t1p, t2p)
	char *title;
	u_long a;
	u_char *e1, *e2;
	time_t *t1p, *t2p;
{
	register char *cp, *hn;
	register int pid;
	register FILE *f;
	char tempfile[64], cpu[64], os[64];
	char *fmt = "%20s: %s\n";
	char *watcher = WATCHER;
	char *watchee = WATCHEE;
	char *sendmail = PATH_SENDMAIL;
	char *unknown = "<unknown>";
	char buf[132];
	static int init = 0;

	/* No report until we're initialized */
	if (initializing)
		return;

	if (debug) {
		f = stdout;
		(void)putc('\n', f);
	} else {
		/* Setup child reaper if we haven't already */
		if (!init) {
			if ((int)signal(SIGCHLD, reaper) < 0)
				syslog(LOG_ERR, "report: signal: %m");
			++init;
		}

		/* Syslog this event too */
		dosyslog(LOG_NOTICE, title, a, e1, e2);
		++cdepth;

		/* Fork off child to send mail */
		pid = fork();
		if (pid) {
			/* Parent */
			if (pid < 0)
				syslog(LOG_ERR, "report fork() 1: %m");
			return;
		}

		/* Child */
		closelog();
		(void)strcpy(tempfile, "/tmp/arpwatch.XXXXXX");
		(void)mktemp(tempfile);
		if ((f = fopen(tempfile, "w+")) == NULL) {
			syslog(LOG_ERR, "child open(%s): %m", tempfile);
			exit(1);
		}
		if (unlink(tempfile) < 0)
			syslog(LOG_ERR, "unlink(%s): %m", tempfile);
	}

	(void)fprintf(f, "From: %s\n", watchee);
	(void)fprintf(f, "To: %s\n", watcher);
	hn = gethname(a);
	if (!isdigit(*hn))
		(void)fprintf(f, "Subject: %s (%s)\n", title, hn);
	else {
		(void)fprintf(f, "Subject: %s\n", title);
		hn = unknown;
	}
	(void)putc('\n', f);
	(void)fprintf(f, fmt, "hostname", hn);
	(void)fprintf(f, fmt, "ip address", intoa(a));
	(void)fprintf(f, fmt, "ethernet address", e2str(e1));
	if ((cp = ec_find(e1)) == NULL)
		cp = unknown;
	(void)fprintf(f, fmt, "ethernet vendor", cp);
	if (hn != unknown && gethinfo(hn, cpu, sizeof(cpu), os, sizeof(os))) {
		(void)sprintf(buf, "%s %s", cpu, os);
		(void)fprintf(f, fmt, "dns cpu & os", buf);
	}
	if (e2) {
		(void)fprintf(f, fmt, "old ethernet address", e2str(e2));
		if ((cp = ec_find(e2)) == NULL)
			cp = unknown;
		(void)fprintf(f, fmt, "old ethernet vendor", cp);
	}
	if (t1p)
		(void)fprintf(f, fmt, "timestamp", fmtdate(*t1p));
	if (t2p)
		(void)fprintf(f, fmt, "previous timestamp", fmtdate(*t2p));
	if (t1p && t2p && *t1p && *t2p)
		(void)fprintf(f, fmt, "delta", fmtdelta(*t1p - *t2p));

	if (debug) {
		fflush(f);
		return;
	}

	(void)rewind(f);
	if (dup2(fileno(f), fileno(stdin)) < 0) {
		syslog(LOG_ERR, "dup2: %m");
		exit(1);
	}
	/* XXX Need to freopen()? */
	/* Deliver interactively or just queue based on number outstanding */
	if (cdepth <= 2)
		cp = "-odi";
	else
		cp = "-odq";
	execl(sendmail, "sendmail", cp, watcher, NULL);
	syslog(LOG_ERR, "execl: %s: %m", sendmail);
	exit(1);
}
