implementation module MipSym;

(******************************************************************************
 * Rudimentary symbol table support for ucode writing
 * $Header: MipSym.mod,v 1.13 90/02/21 20:17:00 lattanzi Locked $
 *****************************************************************************)

from Globals import
    optimFlag, genDebugInfoFlag;

from stsupport import
    indexNil;

from stinfc import
    st_extadd,st_auxbtadd,st_endallfiles,st_dump,
    st_writebinary,st_cuinit,st_filebegin,st_extstradd,st_idn_index_fext,
    st_stradd, st_symadd,st_symadd, st_currentifd, st_setfd,st_writest;

from symconst import
    stProc,stGlobal,scData,btAdr,btUChar,
    btInt,btUInt,btFloat,btDouble,GLEVEL_0,GLEVEL_1,GLEVEL_2,GLEVEL_3,
    langMachine,langPascal,scUndefined,scCommon,scNil,stLocal,stNil,
    btChar;

from Runtime import
    NOPCODEBLOCK;

from hack import 
    FileNumber;

from io import
    WriteF,output, SWriteF;

from strings import
    Assign, Append;

const
     TRACE = False;
     NOVALUE = 0;
     langModula2 = langPascal;
     INITIAL_UCODE_BLOCKS = 5000;
     UNALLOCATED = -1;
     EXTERNAL = 1;
     INTERNAL = 0;
     STDOUT = 0;
     ST_PRINTALL = -1;
     ST_WRITEALL = -1;
     FILEMERGE = 0;			 (* 1 to merge *)

type
    BlockMap = dynarray of integer;	(* Map pcode blocks to ucode *)
    SymbolSection = (Text,Data);
    symclass = (importsym, exportsym, staticsym);
    symclassset = set of symclass;

var
    gleveltab : array [0 .. 3] of integer;
    glevel, uniqueID, junk, stabFileNumber, stabifd : integer;
    textBlockMap, dataBlockMap : BlockMap;
    blockMap : array SymbolSection of BlockMap;
    storageClass : array SymbolSection of integer;
    symbolType : array SymbolSection of integer;
    symInternalIndex : array SymbolSection of integer;

procedure @external st_feinit ();
begin
end st_feinit;

procedure MakeUniqueName(var name : st_string);
begin
     SWriteF(name, ".%d", uniqueID);
     uniqueID := uniqueID + 1;
end MakeUniqueName;

procedure AddSym(const name : st_string;
     	         pcodeBlock : integer;
	  	         sc : SymbolSection;
                 visibility : symclass) : integer;
     var symndx, dn : integer;
         external, newname : boolean;
         Name : array[1..12] of char;
begin
     newname := (name[0] = 0C) and (pcodeBlock = NOPCODEBLOCK);
     if newname then
         MakeUniqueName(Name);
     end;
     dn := UNALLOCATED;
     if pcodeBlock > High(blockMap[sc]) then
     	  WriteF(output, 'pcode block number %d greater than %d\n',pcodeBlock, 
     	       High(blockMap[sc]));
     elsif (pcodeBlock = NOPCODEBLOCK) or
     	   (blockMap[sc]^[pcodeBlock] = UNALLOCATED) then
          case visibility of
          | importsym :
     	      symndx := st_extadd(st_extstradd(name), NOVALUE, symbolType[sc],
	  	   	      	  scUndefined, symInternalIndex[sc]);
          | exportsym :
     	      symndx := st_extadd(st_extstradd(name), NOVALUE, symbolType[sc],
	  	   	      	  storageClass[sc], symInternalIndex[sc]);
          | staticsym :
     	      if newname then
     	          symndx := st_symadd(st_stradd(Name), NOVALUE, symbolType[sc],
	  	       	      	      scData, symInternalIndex[sc]);
              else
     	          symndx := st_symadd(st_stradd(name), NOVALUE, symbolType[sc],
	  	   	      	      scData, symInternalIndex[sc]);
     	      end;
     	  else
              assert(false, 'Unsupported symbol class in AddSym');
          end;
          external := visibility in symclassset {importsym, exportsym};
     	  dn := st_idn_index_fext(symndx, ORD(external));
     	  if pcodeBlock # NOPCODEBLOCK then
		blockMap[sc]^[pcodeBlock] := dn;
     	  end;
	  if TRACE then
     	       WriteF(output, 'Mapping pcode block %d to %d\n',
     	       	    	      	   pcodeBlock,dn);
     	  end;
     else
	  dn := blockMap[sc]^[pcodeBlock];
     end;
     return dn;
end AddSym;

procedure GlobalText(const name : st_string; pcodeBlock : integer) : integer;
begin
     return AddSym(name, pcodeBlock, Text, exportsym);
end GlobalText;

procedure GlobalData(const name : st_string; pcodeBlock : integer) : integer;
begin
     return AddSym(name, pcodeBlock, Data, exportsym);
end GlobalData;

procedure StaticData(const name : st_string; pcodeBlock : integer) : integer;
begin
     return AddSym(name, pcodeBlock, Data, staticsym);
end StaticData;

procedure ExternalData(const name : st_string; pcodeBlock : integer) : integer;
begin
     return AddSym(name, pcodeBlock, Data, importsym);
end ExternalData;

procedure StabData(const str : st_string; const st : integer);
     var curifd, symndx : integer;
begin
     curifd := st_currentifd();
     st_setfd(stabifd);
     symndx := st_symadd(st_stradd(str), ord(st=stGlobal), stNil, scNil, 
     	       	         indexNil);
     st_setfd(curifd);
end StabData;

procedure StabExportData(const str : st_string);
begin
     StabData(str,stGlobal);
end StabExportData;

procedure StabLocalData(const str : st_string);
begin
     StabData(str,stLocal);
end StabLocalData;

procedure StartSyms(const filename : array of char);
     var stabFileName : array[1..100] of char;
begin
     if genDebugInfoFlag then
         if optimFlag then
             glevel := 3;
         else
             glevel := 2;
         end;
     end;
     Assign(stabFileName,'/stab/');
     Append(stabFileName,filename);
     stabFileNumber := st_filebegin(stabFileName,langMachine,
     	       	                    1,	(* doesn't matter *)
     	       	    	            gleveltab[glevel]);
     stabifd := st_currentifd();
     symFileNumber := StartFile(filename);
end StartSyms;

procedure StartFile(const filename : array of char) : integer;
var idn : integer;
begin
    idn := st_filebegin(filename, langModula2, FILEMERGE, gleveltab[glevel]);
    (* Have everyone define these *)
    auxnums[btAdr]    := st_auxbtadd(btAdr);
    auxnums[btChar]   := st_auxbtadd(btChar);
    auxnums[btUChar]  := st_auxbtadd(btUChar); (* M2UM-81 *)
    auxnums[btInt]    := st_auxbtadd(btInt);
    auxnums[btUInt]   := st_auxbtadd(btUInt);
    auxnums[btFloat]  := st_auxbtadd(btFloat); 
    auxnums[btDouble] := st_auxbtadd(btDouble);
    return idn;
end StartFile;

procedure WriteSyms(const symFile : File);
begin
    st_endallfiles;
    st_writest(FileNumber(symFile),ST_WRITEALL);
end WriteSyms;

procedure EndSyms(const symtabFileName : FileName);
begin
    st_endallfiles;
    if TRACE then
        st_dump(STDOUT,ST_PRINTALL);
    end;
    st_writebinary(symtabFileName,ST_WRITEALL);
end EndSyms;

begin
    assert(true,'@(#)$Header: MipSym.mod,v 1.13 90/02/21 20:17:00 lattanzi Locked $');
    new(blockMap[Text],INITIAL_UCODE_BLOCKS);
    new(blockMap[Data],INITIAL_UCODE_BLOCKS);
    for junk := Low(blockMap[Text]^) to High(blockMap[Text]^) do 
        blockMap[Text]^[junk] := UNALLOCATED;
     	blockMap[Data]^[junk] := UNALLOCATED;
    end;
    st_cuinit;
    symbolType[Text] := stProc;
    symbolType[Data] := stGlobal;
    storageClass[Text] := scUndefined (*scText*);
    storageClass[Data] := scCommon;
    symInternalIndex[Text] := indexNil;
    symInternalIndex[Data] := indexNil;
    gleveltab[0] := GLEVEL_0;
    gleveltab[1] := GLEVEL_1;
    gleveltab[2] := GLEVEL_2;
    gleveltab[3] := GLEVEL_3;
    glevel := 0;
    uniqueID := 1;
    for junk := Low(auxnums) to High(auxnums) do
        auxnums[junk] := indexNil;
    end;
end MipSym.
