#include "types.h"
#include "ether.h"
#include "arpcache.h"
#include <stdlib.h>
#include <string.h>
#include <time.h> 

#ifdef TESTBED
#include <stdio.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>
#include <arpa/inet.h>
#endif

static arp_cache_entry *arp_cache[256] = {
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL};

int arp_cache_hash_ip (u32 ip_addr)
{
  return ((ip_addr & 0x000000ff) ^ ((ip_addr >> 8) & 0x000000ff)
	  ^ ((ip_addr >> 8) & 0x000000ff) ^ ((ip_addr >> 8) & 0x000000ff));
}

void arp_cache_add_entry(u32 ip_addr, ether_addr hw_addr)
{
  arp_cache_entry *current = NULL;
  int hash_code = arp_cache_hash_ip (ip_addr);

#ifdef TESTBED
  struct in_addr temp;
  temp.s_addr = ip_addr;
  printf("arpcache: adding %s -> %s mapping\n",
	 inet_ntoa(temp), 
	 ether_ntoa(hw_addr));
#endif
  
  current=arp_cache[hash_code];

  while (current!=NULL && current->ip_addr!=ip_addr) {
    current=current->next;
  }

  if (current!=NULL) {
    memcpy(current->hw_addr,hw_addr,6);
    time(&current->timestamp);
  } else {
    arp_cache_entry *new_entry;

    new_entry=(arp_cache_entry *) malloc(sizeof(arp_cache_entry));
    
    new_entry->ip_addr=ip_addr;
    memcpy(new_entry->hw_addr,hw_addr,6);
    time(&new_entry->timestamp);

    new_entry->next=arp_cache[hash_code];
    arp_cache[hash_code]=new_entry;
  }
  
}

void arp_cache_clear_entry(u32 ip_addr)
{
  arp_cache_entry *current, *last = NULL;
  int hash_code = arp_cache_hash_ip (ip_addr);
  current=arp_cache[hash_code];

  while (current!=NULL && current->ip_addr!=ip_addr) {
    last=current;
    current=current->next;
  }

  if (current!=NULL) {
    if (last==NULL) {
      arp_cache[hash_code]=NULL;
    } else {
      last->next=current->next;
    }
    free (current);
  }
}

void arp_cache_check_timeouts()
{
  int i;
  arp_cache_entry *current, *last, *tmp;
  time_t cutoff;

  /* 10 minutes ago is the cutoff. */
  cutoff=time(NULL)-600;

  for (i=0; i<256; i++) {
    current=arp_cache[i];
    last=NULL;
    while (current!=NULL) {
      if (current->timestamp < cutoff) {
	if (last==NULL) {
	  arp_cache[i]=current->next;
	} else {
	  last->next=current->next;
	}
	tmp=current;
	current=current->next;
	free(current);
      } else {
	last=current;
	current=current->next;
      }
    }
  }

}

arp_cache_entry *arp_cache_lookup_ip(u32 ip_addr)
{
  arp_cache_entry *current;
  int hash_code = arp_cache_hash_ip (ip_addr);
  current=arp_cache[hash_code];
  while (current!=NULL && current->ip_addr!=ip_addr) {
    current=current->next;
  }
  return current;
}
