/* 
 * Touchscreen device driver for SAMCOP/HAMCOP
 *
 * Copyright (c) 2007  Anton Vorontsov <cbou@mail.ru>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/touchscreen-adc.h>
#include <linux/soc/samcop_adc.h>

static int match_pdev_by_name(struct device *dev, void *arg)
{
	struct platform_device *pdev = to_platform_device(dev);
	return strcmp(pdev->name, arg) == 0;
}

static struct platform_device *tsadc_pdev;
static struct tsadc_platform_data tsadc_pdata = {
	.x_pin = "samcop adc:x",
	.y_pin = "samcop adc:y",
	.z1_pin = "samcop adc:z1",
	.z2_pin = "samcop adc:z2",
	.max_jitter = 10,
};

static int __init samcop_ts_init(void)
{
	int ret = 0;
	struct device *adc_dev;

	adc_dev = bus_find_device(&platform_bus_type, NULL, "samcop adc",
	                          match_pdev_by_name);
	if (!adc_dev) {
		printk("samcop_ts: samcop adc device not found\n");
		ret = -ENODEV;
		goto no_samcop_adc;
	}

	tsadc_pdev = platform_device_alloc("ts-adc-debounce", -1);
	if (!tsadc_pdev) {
		printk("samcop_ts: failed to allocate platform device\n");
		ret = -ENOMEM;
		goto pdev_alloc_failed;
	}

	tsadc_pdata.pen_irq = platform_get_irq(to_platform_device(adc_dev), 0);

	ret = platform_device_add_data(tsadc_pdev, &tsadc_pdata,
	                               sizeof(tsadc_pdata));
	if (ret) {
		printk("samcop_ts: failed to add data to platform device\n");
		goto add_data_failed;
	}

	ret = platform_device_add(tsadc_pdev);
	if (ret) {
		printk("samcop_ts: failed to add platform device\n");
		goto pdev_add_failed;
	}

	goto success;

pdev_add_failed:
add_data_failed:
	platform_device_put(tsadc_pdev);
pdev_alloc_failed:
no_samcop_adc:
success:
	return ret;
}

static void __exit samcop_ts_exit(void)
{
	platform_device_unregister(tsadc_pdev);
	return;
}

module_init(samcop_ts_init);
module_exit(samcop_ts_exit);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Anton Vorontsov <cbou@mail.ru>");
