/*
 * Author: Mark Sirota, MIT Project Athena (sirota@athena.mit.edu)
 * August 1987
 */

/*
 * Note the fantastic kludge here: we put a 1x1 0-border window over the
 * normal video window so that we will know when the video window has been
 * moved.  The little window is a sibling of the video window; a child of the
 * RootWindow.  When the video window is moved, an ExposeRegion event is
 * generated on it, and we then move the little window back on top of the
 * video window.  The kludge window lives in the lower left-hand corner of
 * the video window, because it's least visible down there.  The window is
 * called "kludge".
 *
 * This is all because, under X10, there is no way to know when a window is
 * moved, and video windows screw up when they are moved.  In addition, it's
 * nice to align them on 16-pixel boundaries.
 */

/*
 * Another kludge: we paste all portions into two windows: the actual cvideo
 * window and another video window in the upper left corner of the screen.
 * The digitized image isn't updated until either the window is unmapped and
 * the window ID is changed, or something like that.  Doesn't make sense to
 * me, but it works.  This one's called "hack".
 */

/*
 * There are assumptions made in this code which may be non-portable.
 * In particular: (I * F) where I is an int and F is a float.  This can be
 * found in many forms, but that's the root of it.  It seems to work by
 * promoting the smaller operand of a binary operator to the same type as
 * the larger.  It currently works on a Vax, but not on a Sun.  I believe
 * it's included in the proposed ANSI C standard, so that's why I'm willing
 * to use it.
 * 
 * Note that rounding is not done for simplicity of code, and we assume that
 * the rounding errors due to truncation will be minimal (a potentially fatal
 * assumption... :-)
 */

#include <stdio.h>
#include <X/Xlib.h>
#include <X/Xkeyboard.h>

#define SCOPE	extern
#include "XVideo.h"

#include "cvideo.h"

#define PAUSE	100			/* 1/10 second pause for each tile */

#define ZOOM	LeftButton		/* Also SWITCH in PAN mode */
#define PAN	MiddleButton		/* Also SWITCH in ZOOM mode */
#define ZOOMOUT	RightButton		/* Also CANCEL in ZOOM or PAN mode */
#define NONE	-1

typedef union _XeventU {
	XEvent		rep;
	XExposeEvent	expose;
	XButtonEvent	button;
	XMouseEvent	mouse;
} XEventU;

void	XMoveMouse();
void	DrawLine();
void	DrawBox();

CVideo *
InitCVideo(window, width, height, disc, startframe, rows, columns, offset, xoverlap, yoverlap, orientation)
Window		window;
short		width, height;
int		disc;
unsigned short	startframe;
short		rows, columns;
short		offset;
float		xoverlap, yoverlap;
short		orientation;
{
	char		*malloc();
	CVideo		*cvideo = (CVideo *) malloc(sizeof (CVideo));

	cvideo->window = window;

	cvideo->width = width;
	cvideo->height = height;

	cvideo->shown.x = 0;
	cvideo->shown.y = 0;
	cvideo->shown.w = width;
	cvideo->shown.h = height;

	cvideo->disc = disc;
	cvideo->startframe = startframe;
	cvideo->rows = rows;
	cvideo->columns = columns;
	cvideo->offset = offset;
	cvideo->xoverlap = xoverlap;
	cvideo->yoverlap = yoverlap;

	cvideo->function = NONE;

	cvideo->kludge = XCreateWindow(RootWindow, 0, 0, 1, 1, 0, BlackPixmap, BlackPixmap);
	cvideo->hack = XCreateVideoWindow(RootWindow, 0, 0, 16, 12, 0, BlackPixmap, BlackPixmap);

	XSelectInput(cvideo->window, ExposeWindow | KeyPressed | KeyReleased | ButtonPressed | MouseMoved);
	XMapWindow(cvideo->window);
	XMapWindow(cvideo->kludge);

	return cvideo;
}	

/*
 * Process an event.  Handle zooming and panning.
 */
void
CVideoProcess(cvideo, event)
CVideo	*cvideo;
XEventU	event;
{
	WindowInfo	i;
	int		x, y;
	Window		subw;
	void		CorrectSize();
	void		NewShown();
	void		DrawCVideo();
	static int	shift = FALSE;

	switch (event.rep.type) {
	case ExposeWindow:
		XQueryWindow(cvideo->window, &i);

		/*
		 * Insure that each subwindow is a multiple of 16 pixels wide
		 * and that the number of subwindows divides evenly into the
		 * width and height of the big window, and that the
		 * dimensions of each subwindow are 4x3.
		 */
		x = cvideo->width;
		y = cvideo->height;
		CorrectSize(cvideo, i.width, &cvideo->width, &cvideo->height, 16 * cvideo->columns);
		if (cvideo->width != i.width || cvideo->height != i.height) {
			XChangeWindow(cvideo->window, cvideo->width, cvideo->height);
			cvideo->shown.x *= (float) cvideo->width / (float) x;
			cvideo->shown.w *= (float) cvideo->width / (float) x;
			cvideo->shown.y *= (float) cvideo->height / (float) y;
			cvideo->shown.h *= (float) cvideo->height / (float) y;
			return;
			/*
			 * We just generated another ExposeWindow event, so
			 * just skip this one.  We'll get called again for
			 * the next one.
			 */
		}

		/*
		 * Insure that the video begins on a 16-pixel boundary,
		 * taking into account the borderwidth.
		 */
		if ((i.x + i.bdrwidth) % 16) {
			x = ROUND_16(i.x + i.bdrwidth) - i.bdrwidth;
			XMoveWindow(cvideo->window, x, i.y);
		} else
			x = i.x;

		/*
		 * Move the kludge window to the lower left of the video
		 * window.
		 */
		XMoveWindow(cvideo->kludge, x + i.bdrwidth, i.y + i.height + i.bdrwidth - 1);

		cvideo->width = i.width;
		cvideo->height = i.height;

		DrawCVideo(cvideo);
		switch (cvideo->function) {
		case ZOOM:
			DrawBox(cvideo->window, cvideo->rband);
			break;
		case PAN:
			DrawLine(cvideo->window, cvideo->rband);
			break;
		}
		break;
	case KeyPressed:
		if (IsShiftKey(event.button.detail & ValueMask))
			shift = TRUE;
		break;
	case KeyReleased:
		if (IsShiftKey(event.button.detail & ValueMask))
			shift = FALSE;
		break;
	case ButtonPressed:
		if (!shift)
			break;
		switch (event.button.detail & ValueMask) {
		case ZOOM:
			switch (cvideo->function) {
			case NONE:
				cvideo->rband.x = event.button.x;
				cvideo->rband.y = event.button.y;
				cvideo->rband.w = 0;
				cvideo->rband.h = 0;
				DrawBox(cvideo->window, cvideo->rband);
				cvideo->function = ZOOM;
				break;
			case ZOOM:
				DrawBox(cvideo->window, cvideo->rband);
				NewShown(cvideo);
				DrawCVideo(cvideo);
				cvideo->function = NONE;
				break;
			case PAN:
				cvideo->rband.x += cvideo->rband.w;
				cvideo->rband.y += cvideo->rband.h;
				cvideo->rband.w = -cvideo->rband.w;
				cvideo->rband.h = -cvideo->rband.h;
				XMoveMouse(cvideo->window, cvideo->rband.x + cvideo->rband.w, cvideo->rband.y + cvideo->rband.h);
				break;
			}
			break;
		case PAN:
			switch (cvideo->function) {
			case NONE:
				cvideo->rband.x = event.button.x;
				cvideo->rband.y = event.button.y;
				cvideo->rband.w = 0;
				cvideo->rband.h = 0;
				DrawLine(cvideo->window, cvideo->rband);
				cvideo->function = PAN;
				break;
			case PAN:
				DrawLine(cvideo->window, cvideo->rband);
				NewShown(cvideo);
				DrawCVideo(cvideo);
				cvideo->function = NONE;
				break;
			case ZOOM:
				cvideo->rband.x += cvideo->rband.w;
				cvideo->rband.y += cvideo->rband.h;
				cvideo->rband.w = -cvideo->rband.w;
				cvideo->rband.h = -cvideo->rband.h;
				XMoveMouse(cvideo->window, cvideo->rband.x + cvideo->rband.w, cvideo->rband.y + cvideo->rband.h);
				break;
			}
			break;
		case ZOOMOUT:
			switch (cvideo->function) {
			case NONE:
				cvideo->shown.x = 0;
				cvideo->shown.y = 0;
				cvideo->shown.w = cvideo->width;
				cvideo->shown.h = cvideo->height;
				DrawCVideo(cvideo);
				break;
			case ZOOM:
				DrawBox(cvideo->window, cvideo->rband);
				cvideo->function = NONE;
				break;
			case PAN:
				DrawLine(cvideo->window, cvideo->rband);
				cvideo->function = NONE;
				break;
			}
			break;
		}
		break;
	case MouseMoved:
		if (!shift)
			break;
		XUpdateMouse(cvideo->window, &x, &y, &subw);
		switch (cvideo->function) {
		case ZOOM:
			CorrectSize(cvideo, x - cvideo->rband.x, &x, &y, 1);
			if (x != cvideo->rband.w || y != cvideo->rband.h) {
				DrawBox(cvideo->window, cvideo->rband);
				cvideo->rband.w = x;
				cvideo->rband.h = y;
				DrawBox(cvideo->window, cvideo->rband);
			}
			break;
		case PAN:
			if (x - cvideo->rband.w != cvideo->rband.x || y - cvideo->rband.h != cvideo->rband.y) {
				DrawLine(cvideo->window, cvideo->rband);
				cvideo->rband.w = x - cvideo->rband.x;
				cvideo->rband.h = y - cvideo->rband.y;
				DrawLine(cvideo->window, cvideo->rband);
			}
			break;
		}
		break;
	}
}

/*
 * Given a Box and a box on top of it, change the Box to reflect the one
 * determined by that rectangle.
 */
static void
NewShown(cvideo)
CVideo	*cvideo;
{
	void	CorrectSize();

	switch (cvideo->function) {
	case ZOOM:
		/*
		 * Make sure (x,y) is upper left of rband
		 */
		if (cvideo->rband.w < 0) {
			cvideo->rband.w = -cvideo->rband.w;
			cvideo->rband.x -= cvideo->rband.w;
		}
		if (cvideo->rband.h < 0) {
			cvideo->rband.h = -cvideo->rband.h;
			cvideo->rband.y -= cvideo->rband.h;
		}

		cvideo->shown.x += ((float) cvideo->rband.x / (float) cvideo->width) * cvideo->shown.w;
		cvideo->shown.y += ((float) cvideo->rband.y / (float) cvideo->height) * cvideo->shown.h;
		CorrectSize(cvideo, (int) (cvideo->shown.w * ((float) cvideo->rband.w / (float) cvideo->width)), &cvideo->shown.w, &cvideo->shown.h, 1);
		break;
	case PAN:
		cvideo->shown.x -= ((float) cvideo->rband.w / (float) cvideo->width) * cvideo->shown.w;
		cvideo->shown.y -= ((float) cvideo->rband.h / (float) cvideo->height) * cvideo->shown.h;
		break;
	}

	/*
	 * Make sure shown isn't off the map
	 */
	if (cvideo->shown.x < 0)
		cvideo->shown.x = 0;
	else if (cvideo->shown.x + cvideo->shown.w > cvideo->width)
		cvideo->shown.x = cvideo->width - cvideo->shown.w;
	if (cvideo->shown.y < 0)
		cvideo->shown.y = 0;
	else if (cvideo->shown.y + cvideo->shown.h > cvideo->height)
		cvideo->shown.y = cvideo->height - cvideo->shown.h;
}

/*
 * Disclaimer: Images would undoubtably look considerably better if there
 * weren't 16-pixel boundaries to worry about.
 */
void
DrawCVideo(cvideo)
CVideo	*cvideo;
{
	Box	matrix;
	float	fw, fh, lw, lh;
	int	col, row;
	char	command[13];
	int	sx, sy, sw, sh;
	int	dx, dy, dw, dh;

	matrix.x = cvideo->shown.x / (cvideo->width / cvideo->columns);
	matrix.y = cvideo->shown.y / (cvideo->height / cvideo->rows);
	matrix.w = ((cvideo->shown.w + (cvideo->width / cvideo->columns) - 1) / (cvideo->width / cvideo->columns)) + (cvideo->shown.w % (cvideo->width / cvideo->columns) > ((matrix.x + 1) * (cvideo->width / cvideo->columns)) - cvideo->shown.x);
	matrix.h = ((cvideo->shown.h + (cvideo->height / cvideo->rows) - 1) / (cvideo->height / cvideo->rows)) + (cvideo->shown.h % (cvideo->height / cvideo->rows) > ((matrix.y + 1) * (cvideo->height / cvideo->rows)) - cvideo->shown.y);

	fw = (float) (((matrix.x + 1) * (cvideo->width / cvideo->columns)) - cvideo->shown.x) / (float) (cvideo->width / cvideo->columns);
	fh = (float) (((matrix.y + 1) * (cvideo->height / cvideo->rows)) - cvideo->shown.y) / (float) (cvideo->height / cvideo->rows);
	lw = (float) ((cvideo->shown.x + cvideo->shown.w) - ((matrix.x + matrix.w - 1) * (cvideo->width / cvideo->columns))) / (float) (cvideo->width / cvideo->columns);
	lh = (float) ((cvideo->shown.y + cvideo->shown.h) - ((matrix.y + matrix.h - 1) * (cvideo->height / cvideo->rows))) / (float) (cvideo->height / cvideo->rows);

	XClear(cvideo->window);

/**/
#if defined(ORIENTATION) && (ORIENTATION == VERTICAL)
	for (col = matrix.x; col < matrix.x + matrix.w; col++) {
		for (row = matrix.y; row < matrix.y + matrix.h; row++) {
			sprintf(command, "search %u", cvideo->startframe + (((col * cvideo->rows) + row) * cvideo->offset));
#else
	for (row = matrix.y; row < matrix.y + matrix.h; row++) {
		for (col = matrix.x; col < matrix.x + matrix.w; col++) {
			sprintf(command, "search %u", cvideo->startframe + (((row * cvideo->columns) + col) * cvideo->offset));
#endif
			sony_cmd(cvideo->disc, command);

			XMapWindow(cvideo->hack);
			XScaleVideo(cvideo->hack, 0, 0, VIDEO_WIDTH, VIDEO_HEIGHT, 0, 0, 16, 12);
			XUnmapWindow(cvideo->hack);

			sx = VIDEO_WIDTH * cvideo->xoverlap;
			sy = VIDEO_HEIGHT * cvideo->yoverlap;
			sw = VIDEO_WIDTH - (2 * sx);
			sh = VIDEO_HEIGHT - (2 * sy);

			dw = DOWN_TO_16((int) (cvideo->width / ((float) (matrix.w - 2) + fw + lw)));
			dh = DOWN_TO_16((int) (cvideo->height / ((float) (matrix.h - 2) + fh + lh)));
			dx = DOWN_TO_16((int) (dw * fw)) + ((col - matrix.x - 1) * dw);
			dy = DOWN_TO_16((int) (dh * fh)) + ((row - matrix.y - 1) * dh);

			if (col == matrix.x) {
				sx += sw * (1.0 - fw);
				sw *= fw;
				dx = 0;
				dw = DOWN_TO_16((int) (dw * fw));
			} else if (col == matrix.x + matrix.w - 1) {
				sw *= lw;
				dw = DOWN_TO_16((int) (dw * lw));
			}
			if (row == matrix.y) {
				sy += sh * (1.0 - fh);
				sh *= fh;
				dy = 0;
				dh = DOWN_TO_16((int) (dh * fh));
			} else if (row == matrix.y + matrix.h - 1) {
				sh *= lh;
				dh = DOWN_TO_16((int) (dh * lh));
			}

			XScaleVideo(cvideo->window, sx, sy, sw, sh, dx, dy, dw, dh);

#ifdef PAUSE
			mpause(PAUSE);
#endif
		}
	}
}

/*
 * Insure that W and H are in proportion, giving a determining width and an
 * interval which W must be a multiple of.
 */
static void
CorrectSize(cvideo, determiner, w, h, interval)
CVideo	*cvideo;
int	determiner;
int	*w, *h;
int	interval;
{
	static int	xproportion = 0, yproportion = 0;

	if (xproportion <= 0 || yproportion <= 0) {
		int	a, b;
		int	r;

		xproportion = cvideo->columns * 4;
		yproportion = cvideo->rows * 3;

		/*
		 * Reduce the fraction XPROPORTION/YPROPORTION by dividing
		 * each by their GCD, found via Euclid's Algorithm.
		 */
		a = xproportion;
		b = yproportion;
		while ((r = (a % b)) != 0) {
	                a = b;
	                b = r;
	        }

		xproportion /= b;
		yproportion /= b;
	}

	if (interval < xproportion)
		interval = xproportion;

	*w = ((determiner + (interval - 1)) / interval) * interval;
	*h = (*w / xproportion) * yproportion;
}

/*
 * Move the mouse to position (X,Y) in window W safely (removing subsequent
 * MouseMoved events from the queue).
 */
void
XMoveMouse(window, x, y)
Window	window;
int	x, y;
{
	XEvent	event;

	XWarpMouse(window, x, y);
	XSync(FALSE);
	while (XCheckWindowEvent(window, MouseMoved, &event))
		;
}

void
DrawLine(window, box)
Window	window;
Box	box;
{
	XLine(window, box.x, box.y, box.x + box.w, box.y + box.h, 1, 1, BlackPixel, GXxor, AllPlanes);
}

void
DrawBox(window, box)
Window	window;
Box	box;
{
	Vertex	vertices[5];
	void	SetVertex();

	SetVertex(&vertices[0], box.x, box.y, VertexStartClosed);
	SetVertex(&vertices[1], box.w, 0, VertexRelative);
	SetVertex(&vertices[2], 0, box.h, VertexRelative);
	SetVertex(&vertices[3], -box.w, 0, VertexRelative);
	SetVertex(&vertices[4], 0, -box.h, VertexRelative | VertexEndClosed);

	XDraw(window, vertices, 5, 1, 1, BlackPixel, GXxor, AllPlanes);
}	

void
SetVertex(vertex, x, y, flags)
Vertex		*vertex;
short		x, y;
unsigned short	flags;
{
	vertex->x = x;
	vertex->y = y;
	vertex->flags = flags;
}
