/* $Id: oppack.c,v 1.5 1996/04/11 03:22:57 bert Exp $
 * generation of packets and operators
 */

#include "complex.h"
#include "oppack.h"

/** I love spaghetti. **/

#define square(a)   ((a)*(a))

/** global variables **/

real hbar = .05l;
real mass = 1.l;

/** code **/

/*
 * generate matrix of coefficients for potential part of propagator [harmonic]
 *
 * arguments:
 *   V --     array for return matrix.
 *              (must be already allocated and be at least n*n big.)
 *   n --     dimension of matrix.  (must be a power of 2 because of FFT.)
 *   dt --    discrete time step.
 *   delta -- discrete space step.
 *   vcoef -- constant multiplier for the potential.
 * Last three aren't long double because of compiler bugs.
 */
void generate_Vprop_harmonic (complex *V, long n,
			      double dt, double delta, double vcoef, double mu)
{
  long i,ii,ni,k,nk,i2;

  complex arg;
  complex mult = (real)delta*delta*vcoef*dt/hbar*-1i;

  /* it's easiest to do positives and negatives together. */
  /* (we don't realy care about +/- nhalf overlap.) */
  for (i=0,ii=0,ni=0; ii<=ni; i++,ii+=n,ni=(n*n)-ii) {
    i2 =i*i;       /* precalculate */
    for (k=0,nk=0; k<=nk; k++,nk=n-k) {
      /* r^2 = (i^2+k^2)*delta^2 */
      /* (exp(-i*V*dt/hbar))ik = exp(-i*(vcoef*r^2)*dt/hbar) */
      arg = (i2+k*k)*mult;
      V[ii+k] = V[ii+nk] = V[ni+k] = V[ni+nk] = cexp(arg);
    }
  }
}

/*
 * generate matrix of coefficients for potential part of propagator [Nelson]
 *
 * arguments:
 *   V --     array for return matrix.
 *              (must be already allocated and be at least n*n big.)
 *   n --     dimension of matrix.  (must be a power of 2 because of FFT.)
 *   dt --    discrete time step.
 *   delta -- discrete space step.
 *   vcoef -- constant multiplier for the potential.
 *   mu --    coefficient of the x^2 term
 * Last four aren't long double because of compiler bugs.
 */
void generate_Vprop_nelson (complex *V, long n,
			      double dt, double delta, double vcoef, double mu)
{
  long i,ii,ni,k,nk,i2;

  complex arg;
  complex mult = (real)delta*delta*vcoef*dt/hbar*-1i;

  /* it's easiest to do positives and negatives together. */
         /* (only for x, in this case...) */
  /* (we don't realy care about +/- nhalf overlap.) */
  for (i=0,ii=0,ni=0; ii<=ni; i++,ii+=n,ni=(n*n)-ii) {
    i2 =i*i;       /* precalculate */
    for (k=0,nk=0; k<=nk; k++,nk=n-k) {
      /* V = mu*x^2/2 + (y - x^2/2)^2 = */
      /*   = delta^2 [mu*i^2/2 + (k - delta*i^2/2)^2] */
      /* Vprop[i,k] = (exp(-i*V[i,k]*dt/hbar)) */
      arg = (k - delta*i2/2.);
      arg = (mu*i2/2.+arg*arg)*mult;
      V[ii+k] = V[ni+k] = cexp(arg);
      arg = (-k - delta*i2/2.);
      arg = (mu*i2/2.+arg*arg)*mult;
      V[ii+nk] = V[ni+nk] = cexp(arg);
    }
  }
}

/*
 * generate matrix of coefficients for potential part of propagator [constant]
 *
 * arguments:
 *   V --     array for return matrix.
 *              (must be already allocated and be at least n*n big.)
 *   n --     dimension of matrix.  (must be a power of 2 because of FFT.)
 *   dt --    discrete time step.
 *   delta -- discrete space step.
 *   vcoef -- constant multiplier for the potential.
 * Last three aren't long double because of compiler bugs.
 */
void generate_Vprop_constant (complex *V, long n,
			      double dt, double delta, double vcoef, double mu)
{
  long i,k;

  /* (exp(-i*V*dt/hbar)) */
  complex arg = (real)vcoef*dt/hbar*-1i;
  complex val = cexp(arg);

  for (i=0; i<(n*n); i+=n) {
    for (k=0; k<n; k++) {
      V[i+k] = val;
    }
  }
}

/*
 * generate matrix of coefficients for kinetic part of propagator
 *
 * arguments:
 *   K --       array for return matrix.
 *                (must be already allocated and be at least n*n big.)
 *   n --       dimension of matrix.  (must be a power of 2 because of FFT.)
 *   dt --      discrete time step.
 *   delta --   discrete space step.
 * Last two aren't long double because of compiler bugs.
 */
void generate_Kprop (complex *K, long n, double dt, double delta)
{
  long i,ii,ni,k,nk,i2;

  complex arg;
  complex mult = (complex)M_PI/(n*delta);
  mult = dt*2*hbar*mult*mult/mass*-1i;

  /* it's easiest to do positives and negatives together. */
  /* (we don't realy care about +/- nhalf overlap.) */
  for (i=0,ii=0,ni=0; ii<=ni; i++,ii+=n,ni=(n*n)-ii) {
    i2 =i*i;       /* precalculate */
    for (k=0,nk=0; k<=nk; k++,nk=n-k) {
      /* p^2 = (i^2+k^2)*(2*pi*hbar/N/delta)^2 */
      /* (exp(-i*K*dt/hbar))ik = exp(-i*(p^2/2/m)*dt/hbar) */
      arg = (i2+k*k)*mult;
      K[ii+k] = K[ii+nk] = K[ni+k] = K[ni+nk] = cexp(arg);
    }
  }
}

/*
 * generate a Gaussian wave packet in coordinate form
 *
 * arguments:
 *   pack --    array for return
 *                (must be already allocated and be at least n*n big.)
 *   n --       dimension of matrix.  (must be a power of 2 because of FFT.)
 *   rx,ry --   components of initial avg position.
 *   kx,ky --   components of initial avg wave vector.
 *   delta --   discrete space step.
 *   eps --     half-width of Gaussian (in coordinate form).
 *   ncoef --   normalization constant.
 * Last six aren't long double because of compiler bugs.
 */
void generate_packet (complex *pack, long n, double rx, double ry,
	          double kx, double ky, double delta, double eps, double ncoef)
{
  long i,j,ii;
  complex ikr, r2;

  long nhalf = n/2;
 
  /* This one is not symmetric... */
  for (i=-nhalf; i<nhalf; i++) {
    ii = n * ((i+n) % n);
    for (j=-nhalf; j<nhalf; j++) {
      /* \psi = ncoef * exp(i*k0 \cdot r) * exp(-1/4*(r-r0)^2/\epsilon^2) */
      ikr = delta*(i*kx + j*ky)*1i;
      r2 = -.25*(square(delta*i-rx) + square(delta*j-ry))/square(eps);
      pack[ii+((j+n) % n)] = cexp(ikr + r2) * ncoef;
    }
  }

/*
  for (i=0; i<n; i++) {
    pack[(nhalf*n)+i] = 0;
    pack[(i*n)+nhalf] = 0;
  }
*/
}

/*
 * copy an NxN array of complex numbers.
 *   A --    source.  (n x n array)
 *   B --    destination.  (n x n array)
 *   n --    array size.
 */
void copy_data (complex *A, complex *B, long n)
{
  memcpy(B, A, n*n*sizeof(complex));
}
