/*
 * Fast Fourier Transform.
 * Adapted from file "four1.c" from Numerical Recipes.
 */

#include <stdio.h>
#include "complex.h"

unsigned char rfft_debugme = 0;

#ifndef HELL_FREEZES_OVER
#define DEBUGME(fd, foo...) if (rfft_debugme) fprintf(fd, ## foo);
#else
#define DEBUGME(fd, foo...)
#endif

/*
 * calculate a one-dimensional Fast Fourier Transform "in place".
 *    data[] -- array used for both input and output
 *    nn --     array size
 *  if isign==1, calculate the FFT
 *  if isign==-1, calculate the inverse FFT multiplied by nn
 */
void fswap(complex data[], unsigned long nn)
{
  unsigned long m,j,i;
  complex temp;

  j=0;
  for (i=0; i<nn; i++) {
    if (j > i) {
#define SWAP(a,b) temp=(a);(a)=(b);(b)=temp
      SWAP(data[j],data[i]);
#undef SWAP
      DEBUGME(stderr, "%lu: swapped with %lu\n", i, j);
      DEBUGME(stderr, "%lu: swapped by %lu\n",   j, i);
    }
    m=nn >> 1;
    while (m >= 1 && j >= m) {
      j -= m;
      m >>= 1;
    }
    j += m;
  }
}

void fcomb2(complex data[], unsigned long nn, int isign)
{
  unsigned long mmax,m,j,istep,i;
  complex theta;

  complex w,wp,temp;

  mmax=1;
  while (mmax < nn) {
    istep=mmax << 1;
    DEBUGME(stderr, "cycle with mmax=%lu istep=%lu\n", mmax, istep);
    theta=isign*(M_PI/mmax)*1i;
    wp=cexp(theta);	/* function args must be variables */
    			/* because of a bug in GCC (sigh) */
    DEBUGME(stderr, "computed e^(%si*pi/%lu)\n", ((isign>0)?"":"-"), mmax );
    w = 1.0;
    for (m=0; m<mmax; m++) {
      for (i=m; i<nn; i+=istep) {
	j=i+mmax;
	DEBUGME(stderr, "(%2lu,%2lu) <- [%2lu] +- W^%-2lu * [%2lu]\n",
		            i,   j,        i,          m,      j);
	temp=w * data[j];
	data[j]=data[i] - temp;
	data[i] += temp;
      }
      w *= wp;
    }
    mmax=istep;
  }
}

void fcomb1(complex data[], unsigned long nn, int isign)
{
  unsigned long mmax,m,j,istep,i,spread;
  complex theta;

  complex w,wp,temp;

  spread = nn >> 1;
  mmax=1;
  while (mmax < nn) {
    DEBUGME(stderr, "cycle with mmax=%lu spread=%lu\n", mmax, spread);
    theta=isign*(M_PI/mmax)*1i;
    wp=cexp(theta);	/* function args must be variables */
    			/* because of a bug in GCC (sigh) */
    DEBUGME(stderr, "computed e^(%si*pi/%lu)\n", ((isign>0)?"":"-"), mmax );
    w = 1.0;
    for (m=0; m<mmax; m++) {
      for (i=m; i<nn; i+=mmax) {
	j=i+spread;
	DEBUGME(stderr, "(%2lu,%2lu) <- [%2lu] +- W^%-2lu * [%2lu]\n",
		            i,   j,        i,          m,      j);
	temp=w * data[j];
	data[j]=data[i] - temp;
	data[i] += temp;
      }
      w *= wp;
    }
    mmax <<= 1;
    spread >>= 1;
  }
}

void fft(complex data[], unsigned long nn, int isign)
{
  fcomb1(data, nn, isign);
  fswap(data, nn);
}
